/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (Secure Access Module) internal implementation via I2C interface for Reader Library
 * $Author: NXP $
 * $Revision: $
 * $Date: $
 */

#include <ph_Status.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHBAL_REG_SAM

#include "phbalReg_Sam_I2C_T1.h"
#include "phbalReg_Sam_I2C.h"
#include "../phbalReg_Sam_Int.h"
#include <phTools.h>

phStatus_t phbalReg_Sam_I2C_T1_BuildIBlock(phbalReg_Sam_DataParams_t * pDataParams, uint8_t * pTxBuffer,
    uint16_t * pTxLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    uint16_t    PH_MEMLOC_REM wLRC_Pos = 0;
    uint16_t    PH_MEMLOC_REM wIndex = 0;

    uint8_t     PH_MEMLOC_REM bPcb_byte = 0;
    uint8_t     PH_MEMLOC_REM bLRC = 0;

    wLRC_Pos = (uint16_t) (PHBAL_SAM_I2C_T1_START_POS + PHBAL_SAM_I2C_HEADER_LEN + *pTxLen +
        PHBAL_SAM_I2C_LRC_LEN);

    /* Clear Transmit and Receive buffer */
    memset(pDataParams->pTxBuffer, 0x00U, pDataParams->wTxBufSize);
    memset(pDataParams->pRxBuffer, 0x00U, pDataParams->wRxBufSize);

    /* Add Node Address Byte (NAD) to Transmit buffer  */
    pDataParams->pTxBuffer[PHBAL_SAM_I2C_NAD_OFFSET] = 0x00U;

    /* Update the send PCB block number */
    bPcb_byte |= pDataParams->bI2C_PcbBlockNum;

    /* Add Protocol Control Byte (PCB) to Transmit buffer */
    pDataParams->pTxBuffer[PHBAL_SAM_I2C_PCB_OFFSET] = bPcb_byte;

    /* Add Length (LEN) to Transmit buffer */
    pDataParams->pTxBuffer[PHBAL_SAM_I2C_LEN_OFFSET] = (uint8_t) (*pTxLen & 0x00FFU);

    /* Add Information (INF) to Transmit buffer */
    (void) memcpy(&pDataParams->pTxBuffer[PHBAL_SAM_I2C_INF_BYTE_OFFSET], pTxBuffer, *pTxLen);

    /* Calculate LRC */
    pDataParams->pTxBuffer[wLRC_Pos - 1U] = 0x00;
    for(wIndex = PHBAL_SAM_I2C_NAD_OFFSET; wIndex <= wLRC_Pos; wIndex++)
    {
        bLRC = bLRC ^ pDataParams->pTxBuffer[wIndex];
    }

    /* Add Epilogue Field Format to Transmit buffer */
    pDataParams->pTxBuffer[wLRC_Pos - 1U] = bLRC;

    /* Exchange I2C Framed Information to SAM */
    PH_CHECK_SUCCESS_FCT(wStatus, phbalReg_Sam_Int_Exchange_I2C(
        pDataParams,
        PHBAL_SAM_CMD_CLA_COMMUNICATION_I2C,
        PHBAL_SAM_CMD_INS_I2C_TRANSMIT_DATA,
        &pDataParams->pTxBuffer[PHBAL_SAM_I2C_T1_START_POS],
        (uint16_t)(wLRC_Pos - PHBAL_SAM_I2C_T1_START_POS),
        PHBAL_SAM_I2C_READ_UNKNOWN,
        NULL,
        NULL));

    /* Toggle Block number for next I-Block */
    pDataParams->bI2C_PcbBlockNum ^= 1 << 6;

    return wStatus;
}

phStatus_t phbalReg_Sam_I2C_T1_BuildSBlock(phbalReg_Sam_DataParams_t * pDataParams, uint8_t bParam)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    uint16_t    PH_MEMLOC_REM wFrameLen = 0;
    uint8_t     PH_MEMLOC_REM bPcb_byte = 0;
    uint8_t     PH_MEMLOC_REM bLRC = 0;

    if(pDataParams->bI2C_NextFrame == PHBAL_SAM_I2C_SEND_SFRAME_WTX_RESP)
    {
        wFrameLen = (uint16_t) (PHBAL_SAM_I2C_T1_START_POS + PHBAL_SAM_I2C_HEADER_LEN + 1U /* WTX byte */ +
            PHBAL_SAM_I2C_LRC_LEN);

        bPcb_byte = (uint8_t) (PHBAL_SAM_I2C_WTX_REQ_PCB ^ (1U << 5U));

    }
    else /* IFS */
    {
        wFrameLen = (uint16_t) (PHBAL_SAM_I2C_T1_START_POS + PHBAL_SAM_I2C_HEADER_LEN + 1U /* IFS byte */ +
            PHBAL_SAM_I2C_LRC_LEN);

        bPcb_byte = 0xC1U;
    }

    /* Clear Transmit and Receive buffer */
    memset(pDataParams->pTxBuffer, 0x00U, pDataParams->wTxBufSize);
    memset(pDataParams->pRxBuffer, 0x00U, pDataParams->wRxBufSize);

    /* Add Node Address Byte (NAD) to Transmit buffer  */
    pDataParams->pTxBuffer[PHBAL_SAM_I2C_NAD_OFFSET] = 0x00U;

    /* Add Protocol Control Byte (PCB) to Transmit buffer */
    pDataParams->pTxBuffer[PHBAL_SAM_I2C_PCB_OFFSET] = bPcb_byte;

    /* Add Length (LEN) to Transmit buffer */
    pDataParams->pTxBuffer[PHBAL_SAM_I2C_LEN_OFFSET] = 0x01U;

    /* Add Information (INF) to Transmit buffer */
    pDataParams->pTxBuffer[PHBAL_SAM_I2C_INF_BYTE_OFFSET] = bParam;

    /* Calculate LRC */
    bLRC = bLRC ^ bParam;

    /* Add Epilogue Field Format to Transmit buffer */
    pDataParams->pTxBuffer[wFrameLen - 1] = bLRC;

    /* Exchange I2C Framed Information to SAM */
    PH_CHECK_SUCCESS_FCT(wStatus, phbalReg_Sam_Int_Exchange_I2C(
        pDataParams,
        PHBAL_SAM_CMD_CLA_COMMUNICATION_I2C,
        PHBAL_SAM_CMD_INS_I2C_TRANSMIT_DATA,
        NULL,
        0,
        PHBAL_SAM_I2C_READ_UNKNOWN,
        NULL,
        NULL));

    return wStatus;
}

phStatus_t phbalReg_Sam_I2C_T1_ProcessResponse(phbalReg_Sam_DataParams_t * pDataParams, uint16_t wRxBufSize,
    uint8_t * pRxBuffer, uint16_t * pRxBufLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    uint8_t     PH_MEMLOC_REM bLRC = 0x00;
    uint8_t     PH_MEMLOC_REM bIndex = 0x00;

    uint8_t     PH_MEMLOC_REM bWtxResp = 0;

    /* Exchange I2C Framed Information to SAM */
    PH_CHECK_SUCCESS_FCT(wStatus, phbalReg_Sam_Int_Exchange_I2C(
        pDataParams,
        PHBAL_SAM_CMD_CLA_COMMUNICATION_I2C,
        PHBAL_SAM_CMD_INS_I2C_RECEIVE_DATA,
        NULL,
        0,
        wRxBufSize,
        &pRxBuffer,
        pRxBufLen));

    if(wStatus == PH_ERR_SUCCESS)
    {
        if(pRxBuffer[PHBAL_SAM_PCB_INDEX] == PHBAL_SAM_I2C_WTX_REQ_PCB)
        {    /* WTX handling */
            pDataParams->bI2C_NextFrame = PHBAL_SAM_I2C_SEND_SFRAME_WTX_RESP;
            bWtxResp = pRxBuffer[3];

            /* Exchange I2C Framed Information to SAM */
            PH_CHECK_SUCCESS_FCT(wStatus, phbalReg_Sam_Int_Exchange_I2C(
                pDataParams,
                PHBAL_SAM_CMD_CLA_COMMUNICATION_I2C,
                PHBAL_SAM_CMD_INS_I2C_EXCHANGE_DATA,
                &bWtxResp,
                1U,
                PHBAL_SAM_I2C_READ_UNKNOWN,
                NULL,
                NULL));

            return wStatus;
        }

        /* LRCCheck of response */
        bLRC = bLRC ^ pRxBuffer[0];
        bLRC = bLRC ^ pRxBuffer[1];
        bLRC = bLRC ^ pRxBuffer[2];
        *pRxBufLen = pRxBuffer[2];

        for(bIndex = 0; bIndex < *pRxBufLen; bIndex++)
        {
            bLRC = bLRC ^ pRxBuffer[bIndex + 3];
            pRxBuffer[bIndex] = pRxBuffer[bIndex + 3];
        }

        if(bLRC == pRxBuffer[*pRxBufLen + 3])
        {
            wStatus = PH_ERR_SUCCESS;
        }
        else
        {
            wStatus = PH_ERR_INTEGRITY_ERROR;
        }
    }

    return wStatus;
}

#endif /* NXPBUILD__PHBAL_REG_SAM */
