/*
 * Copyright 2018, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#include <ph_Status.h>
#include <ph_RefDefs.h>
#include <ph_TypeDefs.h>
#include <string.h>
#include <phTools.h>

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_X

#include "../phalMfPrimeNtag_Int.h"
#include "phalMfPrimeNtag_Sam_X.h"
#include "phalMfPrimeNtag_Sam_X_Int.h"

phStatus_t phalMfPrimeNtag_Sam_X_Int_ValidateResponse(void * pDataParams, uint16_t wStatus, uint8_t * pPiccRetCode)
{
	phStatus_t	PH_MEMLOC_REM wStatus1 = 0;
	uint16_t	PH_MEMLOC_REM wPiccStat = 0;
	uint16_t	PH_MEMLOC_REM wPiccRetCode = 0;

	/* Evaluate the response. */
	if ((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV3_ERR_DESFIRE_GEN)
	{
		/* Evaluate the PICC Status. */
		wPiccRetCode = (uint16_t) ((pPiccRetCode[0] << 8) | pPiccRetCode[1]);
		wPiccStat = (uint16_t) (((wPiccRetCode & 0xFF00) == 0x9100) ? (wPiccRetCode & 0x00FF) : wPiccRetCode);

		/* Validate the PICC Status. */
		wStatus1 = phalMfprimeNtag_Int_ComputeErrorResponse(pDataParams, wPiccStat);
	}
	else
	{
		if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING)
		{
			wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_AL_MFPRIMENTAG);
		}

		PH_CHECK_SUCCESS(wStatus);
	}

	return wStatus1;
}

phStatus_t phalMfPrimeNtag_Sam_X_Int_Iso7816Wrap(uint8_t bLengthPresent, uint8_t bLengthLen, uint8_t bLePresent, uint32_t dwTotDataLen,
	uint8_t * pPlainData, uint16_t * pDataLen)
{
	uint8_t		PH_MEMLOC_REM bCmdCode = 0;
	uint8_t		PH_MEMLOC_REM bLC = 0;
	uint8_t		PH_MEMLOC_REM bOffset = 0;
	uint8_t		PH_MEMLOC_REM bOffset1 = 0;
	uint8_t		PH_MEMLOC_REM bIsIsoChainnedCmd = PH_OFF;
	uint8_t		PH_MEMLOC_REM bIsIsoChainnedCmd_Read = PH_OFF;
	uint8_t		PH_MEMLOC_REM bIsIsoChainnedCmd_Write = PH_OFF;

	/* Extract the command code. */
	bCmdCode = pPlainData[bLengthPresent ? bLengthLen :0];

	/* Set the flag for data operation commands. */
	bIsIsoChainnedCmd_Read = (uint8_t) ((bCmdCode == PHAL_MFPRIMENTAG_CMD_READ_DATA_ISO) ? PH_ON : PH_OFF);
	bIsIsoChainnedCmd_Write = (uint8_t) ((bCmdCode == PHAL_MFPRIMENTAG_CMD_WRITE_DATA_ISO) ? PH_ON : PH_OFF);
	bIsIsoChainnedCmd = (uint8_t) (bIsIsoChainnedCmd_Read | bIsIsoChainnedCmd_Write);

	/* Set the LC value */
	bLC = (uint8_t) (*pDataLen - 1 /* Removing the command code. */);

	/* Subtract LC by 3 if Length is present. */
	bLC = (uint8_t) (bLengthPresent ? ( bLC - bLengthLen)  : bLC);

	/* Compute the offset. */
	bOffset = (uint8_t) (bLengthPresent ? (bLengthLen + 1) : 1);

	/* Compute the offset to be used for moving the current data. */
	bOffset1 = (uint8_t) (bLengthPresent ? (bLengthLen + 5) : 5);

	/* Copy the data to the Pointer. */
	memmove(&pPlainData[bOffset1], &pPlainData[bOffset], bLC);	/* PRQA S 3200 */

	/* Reset the length buffer. */
	*pDataLen = bLengthPresent ? bLengthLen : 0;

	/* Frame the initial ISO7816 header. */
	pPlainData[(*pDataLen)++] = PHAL_MFPRIMENTAG_WRAPPEDAPDU_CLA;
	pPlainData[(*pDataLen)++] = bCmdCode;
	pPlainData[(*pDataLen)++] = PHAL_MFPRIMENTAG_WRAPPEDAPDU_P1;
	pPlainData[(*pDataLen)++] = PHAL_MFPRIMENTAG_WRAPPEDAPDU_P2;

	/* Add LC if there is data. */
	if(bLC)
	{
		pPlainData[(*pDataLen)++] = (uint8_t) (bIsIsoChainnedCmd_Write ? dwTotDataLen : bLC);

		/* Update Frame length based on LC. */
		*pDataLen = *pDataLen + bLC;
	}

	if(bLePresent)
	{
		pPlainData[(*pDataLen)++] = PHAL_MFPRIMENTAG_WRAPPEDAPDU_LE;
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

phStatus_t phalMfPrimeNtag_Sam_X_Int_ResetAuthentication(void * pDataParams)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;

	/* Reset the Authmode and Key number */
	PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode	= 0xFF;
    PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bKeyNo	= 0xFF;

	/* Kill the PICC Authentication. */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_KillAuthentication(
		PHAL_MFPRIMENTAG_RESOLVE_HAL_DATAPARAMS(pDataParams),
		PHHAL_HW_SAMAV3_CMD_SAM_KILL_AUTHENTICATION_PARTIAL));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

phStatus_t phalMfPrimeNtag_Sam_X_Int_ComputeSDMSessionVectors(void * pDataParams, uint8_t bSdmOption, uint16_t wSrcKeyNo, uint16_t wSrcKeyVer,
	uint16_t wDstKeyNo, uint8_t * pUid, uint8_t bUidLen, uint8_t * pSDMReadCtr)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM aSV[16];
	uint8_t		PH_MEMLOC_REM bSvLen = 0;
	uint32_t	PH_MEMLOC_REM dwSDMReadCtr = 0;

	/* Validate the Counter value. */
	if(pSDMReadCtr != NULL)
	{
		dwSDMReadCtr = (uint32_t) (pSDMReadCtr[0] | (pSDMReadCtr[1] << 8) | (pSDMReadCtr[2] << 16) | (pSDMReadCtr[3] << 24));
		if (dwSDMReadCtr == 0xFFFFFF)
		{
			return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_AL_MFPRIMENTAG);
		}
	}

	/* Clear the session vector SV. */
	memset(aSV, 0, 16);		/* PRQA S 3200 */

	/* Frame the default values in session vector. */
	aSV[bSvLen++] = 0x3C;
	aSV[bSvLen++] = 0xC3;
	aSV[bSvLen++] = 0x00;
	aSV[bSvLen++] = 0x01;
	aSV[bSvLen++] = 0x00;
	aSV[bSvLen++] = 0x80;

	/* Append the UID */
	if(bSdmOption & PHAL_MFPRIMENTAG_VCUID_PRESENT)
	{
		memcpy(&aSV[bSvLen], pUid, bUidLen);	/* PRQA S 3200 */
		bSvLen += bUidLen;
	}

	/* Append the SDM ReadCtr information. */
	if(bSdmOption & PHAL_MFPRIMENTAG_RDCTR_PRESENT)
	{
		aSV[bSvLen++] = (uint8_t) (dwSDMReadCtr & 0xFF);
		aSV[bSvLen++] = (uint8_t) ((dwSDMReadCtr & 0xFF00) >> 8);
		aSV[bSvLen++] = (uint8_t) ((dwSDMReadCtr & 0xFF0000) >> 16);
	}

	/* Update the SV length to 16. */
	bSvLen = 16;

	/* Exchange the session vector information to SAM. */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_DeriveKey(
		PHAL_MFPRIMENTAG_RESOLVE_HAL_DATAPARAMS(pDataParams),
		(uint8_t) wSrcKeyNo,
		(uint8_t) wSrcKeyVer,
		(uint8_t) wDstKeyNo,
		aSV,
		bSvLen));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_X */
