/*
 * Copyright 2016 - 2018, 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Software MIFARE Plus(R) Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phhalHw.h>
#include <phalMfp.h>
#include <phpalMifare.h>
#include <ph_RefDefs.h>
#include <phTools.h>

#ifdef NXPBUILD__PHAL_MFP_SAM_X
#include <phhalHw_SamAV2_Cmd.h>

#ifdef NXPBUILD__PHAL_MFP_SAMAV3_X
#include <phhalHw_SamAV3_Cmd.h>
#endif /* NXPBUILD__PHAL_MFP_SAMAV3_X */

#include "../phalMfp_Int.h"
#include "phalMfp_Sam_X.h"
#include "phalMfp_Sam_X_Int.h"

phStatus_t phalMfp_SamAV2_X_Init(phalMfp_SamAV2_X_DataParams_t * pDataParams, uint16_t wSizeOfDataParams, phhalHw_SamAV2_DataParams_t * pHalDataParams,
	void * pPalMifareDataParams)
{
	/* Check Dataparams size. */
	if (sizeof(phalMfp_SamAV2_X_DataParams_t) != wSizeOfDataParams)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFP);
	}

	/* Verify the parameters. */
	PH_ASSERT_NULL_DATA_PARAM (pDataParams, PH_COMP_AL_MFP);
	PH_ASSERT_NULL_PARAM (pHalDataParams, PH_COMP_AL_MFP);
	PH_ASSERT_NULL_PARAM (pPalMifareDataParams, PH_COMP_AL_MFP);

	/* Store the parameter info to internal structure members. */
	pDataParams->wId                        = PH_COMP_AL_MFP | PHAL_MFP_SAMAV2_X_ID;
	pDataParams->pHalDataParams             = pHalDataParams;
	pDataParams->pPalMifareDataParams       = pPalMifareDataParams;

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}

#ifdef NXPBUILD__PHAL_MFP_SAMAV3_X
phStatus_t phalMfp_SamAV3_X_Init(phalMfp_SamAV3_X_DataParams_t * pDataParams, uint16_t wSizeOfDataParams, phhalHw_SamAV3_DataParams_t * pHalDataParams,
	void * pPalMifareDataParams)
{
	/* Check Dataparams size. */
	if (sizeof(phalMfp_SamAV3_X_DataParams_t) != wSizeOfDataParams)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFP);
	}

	/* Verify the parameters. */
	PH_ASSERT_NULL_DATA_PARAM (pDataParams, PH_COMP_AL_MFP);
	PH_ASSERT_NULL_PARAM (pHalDataParams, PH_COMP_AL_MFP);
	PH_ASSERT_NULL_PARAM (pPalMifareDataParams, PH_COMP_AL_MFP);

	/* Store the parameter info to internal structure members. */
	pDataParams->wId                        = PH_COMP_AL_MFP | PHAL_MFP_SAMAV3_X_ID;
	pDataParams->pHalDataParams             = pHalDataParams;
	pDataParams->pPalMifareDataParams       = pPalMifareDataParams;

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}
#endif /* NXPBUILD__PHAL_MFP_SAMAV3_X */

/****************************************************************************************************************************************/
/* Mifare Plus Generic command for personalization.																						*/
/****************************************************************************************************************************************/
phStatus_t phalMfp_Sam_X_WritePerso(void * pDataParams, uint8_t bLayer4Comm, uint16_t wBlockNr, uint8_t * pValue)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM aBlocks[18];
	uint8_t		PH_MEMLOC_REM bBlockLen = 0;
	uint8_t		PH_MEMLOC_REM bRespCode = 0;

	/* Frame the blocks information. */
	aBlocks[bBlockLen++] = (uint8_t) (wBlockNr & 0x00FF);
	aBlocks[bBlockLen++] = (uint8_t) ((wBlockNr & 0xFF00) >> 8);

	memcpy(&aBlocks[bBlockLen], pValue, 16);	/* PRQA S 3200 */
	bBlockLen += 16;

	/* Exchange the information to SAM Hardware. */
	wStatus = PHAL_MFP_WRITE_PERSO(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		aBlocks,
		bBlockLen,
		&bRespCode);


	/* Return the error code. */
	if (((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV2_ERR_MIFARE_PLUS_GEN) ||
		((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS))
	{
		/* Compute the response code. */
		PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Int_ComputeErrorResponse(1, bRespCode, bLayer4Comm));
	}
	else
	{
		return wStatus;
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}

phStatus_t phalMfp_Sam_X_CommitPerso(void * pDataParams, uint8_t bLayer4Comm)
{
	return phalMfp_Int_CommitPerso(PHAL_MFP_RESOLVE_PAL_MIFARE_DATAPARAMS(pDataParams), bLayer4Comm);
}


/****************************************************************************************************************************************/
/* Mifare Plus Generic command for authentication.																						*/
/****************************************************************************************************************************************/

phStatus_t phalMfp_Sam_X_AuthenticateClassicSL2(void * pDataParams, uint8_t bBlockNo, uint8_t bKeyType, uint16_t wKeyNo, uint16_t wKeyVer,
	uint8_t * pUid, uint8_t bUidLen)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;

	/* Exchange the information to respective SamX PAL layer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_MfcAuthenticateKeyNo(
		PHAL_MFP_RESOLVE_PAL_MIFARE_DATAPARAMS(pDataParams),
		bBlockNo,
		bKeyType,
		(uint8_t) wKeyNo,
        (uint8_t) wKeyVer,
		&pUid[bUidLen - 4]));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}

phStatus_t phalMfp_Sam_X_AuthenticateSL0(void * pDataParams, uint8_t bLayer4Comm, uint8_t bFirstAuth, uint16_t wBlockNr,
	uint16_t wKeyNo, uint16_t wKeyVer, uint8_t bDivInputLen, uint8_t * pDivInput, uint8_t bPcdCap2InLen, uint8_t * pPcdCap2In,
	uint8_t * pPcdCap2Out, uint8_t * pPdCap2)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;

	PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Sam_X_Int_MFP_Authenticate(
		pDataParams,
		bLayer4Comm,
		bFirstAuth,
		PHAL_MFP_AUTHENTICATE_SL1_NO_KDF,
		wBlockNr,
		wKeyNo,
		wKeyVer,
		pDivInput,
		bDivInputLen,
		pPcdCap2In,
		bPcdCap2InLen,
		pPcdCap2Out,
		pPdCap2));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}

phStatus_t phalMfp_Sam_X_AuthenticateSL1(void * pDataParams, uint8_t bLayer4Comm, uint8_t bFirstAuth, uint16_t wBlockNr,
	uint16_t wKeyNo, uint16_t wKeyVer, uint8_t bDivInputLen, uint8_t * pDivInput, uint8_t bPcdCap2InLen, uint8_t * pPcdCap2In,
	uint8_t * pPcdCap2Out, uint8_t * pPdCap2)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;

	PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Sam_X_Int_MFP_Authenticate(
		pDataParams,
		bLayer4Comm,
		bFirstAuth,
		PHAL_MFP_AUTHENTICATE_SL1_NO_KDF,
		wBlockNr,
		wKeyNo,
		wKeyVer,
		pDivInput,
		bDivInputLen,
		pPcdCap2In,
		bPcdCap2InLen,
		pPcdCap2Out,
		pPdCap2));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}

phStatus_t phalMfp_Sam_X_AuthenticateSL2(void * pDataParams, uint8_t bLayer4Comm, uint8_t bFirstAuth, uint16_t wBlockNr,
	uint16_t wKeyNo, uint16_t wKeyVer, uint8_t bDivInputLen, uint8_t * pDivInput, uint8_t bPcdCap2InLen, uint8_t * pPcdCap2In,
	uint8_t * pPcdCap2Out, uint8_t * pPdCap2)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;

	PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Sam_X_Int_MFP_Authenticate(
		pDataParams,
		bLayer4Comm,
		bFirstAuth,
		PHAL_MFP_AUTHENTICATE_SL2_KDF,
		wBlockNr,
		wKeyNo,
		wKeyVer,
		pDivInput,
		bDivInputLen,
		pPcdCap2In,
		bPcdCap2InLen,
		pPcdCap2Out,
		pPdCap2));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}

phStatus_t phalMfp_Sam_X_AuthenticateSL3(void * pDataParams, uint8_t bFirstAuth, uint16_t wBlockNr, uint16_t wKeyNo,
	uint16_t wKeyVer, uint8_t bDivInputLen, uint8_t * pDivInput, uint8_t bPcdCap2InLen, uint8_t * pPcdCap2In,
	uint8_t * pPcdCap2Out, uint8_t * pPdCap2)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM bKdf = PHAL_MFP_AUTHENTICATE_SL3_KDF;

	/* Update the KDF. */
	if((wBlockNr >= PHAL_MFP_SAM_X_ORIGINALITY_KEY_FIRST) && (wBlockNr <= PHAL_MFP_SAM_X_ORIGINALITY_KEY_LAST))
		bKdf = PHAL_MFP_AUTHENTICATE_SL1_NO_KDF;

	PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Sam_X_Int_MFP_Authenticate(
		pDataParams,
		PH_ON,
		bFirstAuth,
		bKdf,
		wBlockNr,
		wKeyNo,
		wKeyVer,
		pDivInput,
		bDivInputLen,
		pPcdCap2In,
		bPcdCap2InLen,
		pPcdCap2Out,
		pPdCap2));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}



/****************************************************************************************************************************************/
/* Mifare Plus Generic command for data operations.																						*/
/****************************************************************************************************************************************/
phStatus_t phalMfp_Sam_X_MultiBlockWrite(void * pDataParams, uint8_t bBlockNr, uint8_t bNumBlocks, uint8_t * pBlocks)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM aData[2];
	uint8_t		PH_MEMLOC_REM bDataLen = 0;

	/* Frame the payload information. */
	aData[bDataLen++] = bBlockNr;
	aData[bDataLen++] = bNumBlocks;

	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV2_Cmd_MfWrite(
		((phalMfp_SamAV2_X_DataParams_t *)pDataParams)->pHalDataParams,
		PH_EXCHANGE_BUFFER_FIRST | 0x02 /* P1 bit set. */,
		aData,
		bDataLen));

	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV2_Cmd_MfWrite(
		((phalMfp_SamAV2_X_DataParams_t *)pDataParams)->pHalDataParams,
		PH_EXCHANGE_BUFFER_LAST,
		pBlocks,
		(uint8_t) (16 * bNumBlocks)));

	return wStatus;
}

phStatus_t phalMfp_Sam_X_MultiBlockRead(void * pDataParams, uint8_t bBlockNr, uint8_t bNumBlocks, uint8_t * pBlocks)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM aData[2];
	uint8_t		PH_MEMLOC_REM bDataLen = 0;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;

	/* Frame the payload information. */
	aData[bDataLen++] = bBlockNr;
	aData[bDataLen++] = bNumBlocks;

	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV2_Cmd_MfRead(
		((phalMfp_SamAV2_X_DataParams_t *)pDataParams)->pHalDataParams,
		PH_EXCHANGE_DEFAULT | 0x01 /* P0 bit set. */,
		aData,
		bDataLen,
		&pBlocks,
		&wRespLen));

	return wStatus;
}

phStatus_t phalMfp_Sam_X_Write(void * pDataParams, uint8_t bEncrypted, uint8_t bWriteMaced, uint16_t wBlockNr, uint8_t bNumBlocks,
	uint8_t * pBlocks)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM bCmdCode = 0;

	/* Evaluate the command code. */
	bCmdCode = (uint8_t) (PHAL_MFP_CMD_WRITE_EN | ((!bEncrypted & 0x01) << 1) | bWriteMaced);

	/* Perform the Write command */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Sam_X_Int_MFP_CombinedWrite(
		pDataParams,
		bCmdCode,
		wBlockNr,
		0x00,
		pBlocks,
		(bNumBlocks * 16)));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}

phStatus_t phalMfp_Sam_X_Read(void * pDataParams, uint8_t bEncrypted, uint8_t bReadMaced, uint8_t bMacOnCmd, uint16_t wBlockNr,
	uint8_t bNumBlocks, uint8_t * pBlocks)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM bCmdCode = 0;

	/* Evaluate the command code. */
	bCmdCode = (uint8_t) (PHAL_MFP_CMD_READ_ENM | (((!bMacOnCmd & 0x01) << 2) | ((!bEncrypted & 0x01) << 1) | bReadMaced));

	/* Perform Read command. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Sam_X_Int_CombinedReadMFP(
		pDataParams,
		bCmdCode,
		wBlockNr,
		bNumBlocks,
		pBlocks));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}



/****************************************************************************************************************************************/
/* Mifare Plus Generic command for value operations.																					*/
/****************************************************************************************************************************************/
phStatus_t phalMfp_Sam_X_WriteValue(void * pDataParams, uint8_t bEncrypted, uint8_t bWriteMaced, uint16_t wBlockNr, uint8_t * pValue,
	uint8_t bAddrData)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM bCmdCode = 0;
    uint8_t		PH_MEMLOC_REM pBlock[16];

	/* Form the value to be written in block format.
	*				 | 00 01 02 03 | 04 05 06 07 | 08 09 0A 0B |  0C  |    0D  |  0E  |   0F   |
	* Value Block = |    Value    |    ~Value   |    Value    | Addr | ~ Addr | Addr | ~ Addr |
	*/
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Int_CreateValueBlock(pValue, bAddrData, pBlock));

	/* Evaluate the command code. */
	bCmdCode = (uint8_t) (PHAL_MFP_CMD_WRITE_EN | ((!bEncrypted & 0x01) << 1) | bWriteMaced);

	PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Sam_X_Int_MFP_CombinedWrite(
		pDataParams,
		bCmdCode,
		wBlockNr,
		0x00,
		pBlock,
		16));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}

phStatus_t phalMfp_Sam_X_ReadValue(void * pDataParams, uint8_t bEncrypted, uint8_t bReadMaced, uint8_t bMacOnCmd, uint16_t wBlockNr,
	uint8_t * pValue, uint8_t * pAddrData)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM bCmdCode = 0;
	uint8_t		PH_MEMLOC_REM aData[16];

	/* Evaluate the command code. */
	bCmdCode = (uint8_t) (PHAL_MFP_CMD_READ_ENM | (((!bMacOnCmd & 0x01) << 2) | ((!bEncrypted & 0x01) << 1) | bReadMaced));

	/* Perform Read command. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Sam_X_Int_CombinedReadMFP(
		pDataParams,
		bCmdCode,
		wBlockNr,
		0x01,
		aData));

	PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Int_CheckValueBlockFormat(aData));

	*pAddrData = aData[12];
	memcpy(pValue, aData, 4); /* PRQA S 3200 */

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}

phStatus_t phalMfp_Sam_X_Increment(void * pDataParams, uint8_t bIncrementMaced, uint16_t wBlockNr, uint8_t * pValue)
{
	phStatus_t PH_MEMLOC_REM wStatus = 0;

	PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Sam_X_Int_MFP_CombinedWrite(
		pDataParams,
		(uint8_t) (PHAL_MFP_CMD_INCR | bIncrementMaced),
		wBlockNr,
		0x00,
		pValue,
		4));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}

phStatus_t phalMfp_Sam_X_Decrement(void * pDataParams, uint8_t bDecrementMaced, uint16_t wBlockNr, uint8_t * pValue)
{
	phStatus_t PH_MEMLOC_REM wStatus = 0;

	PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Sam_X_Int_MFP_CombinedWrite(
		pDataParams,
		(uint8_t) (PHAL_MFP_CMD_DECR | bDecrementMaced),
		wBlockNr,
		0x00,
		pValue,
		4));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}

phStatus_t phalMfp_Sam_X_IncrementTransfer(void * pDataParams, uint8_t bIncrementTransferMaced, uint16_t wSrcBlockNr,
	uint16_t wDstBlockNr, uint8_t * pValue)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;

	PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Sam_X_Int_MFP_CombinedWrite(
		pDataParams,
		(uint8_t) (PHAL_MFP_CMD_INCRTR | bIncrementTransferMaced),
		wSrcBlockNr,
		wDstBlockNr,
		pValue,
		4));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}

phStatus_t phalMfp_Sam_X_DecrementTransfer(void * pDataParams, uint8_t bDecrementTransferMaced, uint16_t wSrcBlockNr,
	uint16_t wDstBlockNr, uint8_t * pValue)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;

	PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Sam_X_Int_MFP_CombinedWrite(
		pDataParams,
		(uint8_t) (PHAL_MFP_CMD_DECRTR | bDecrementTransferMaced),
		wSrcBlockNr,
		wDstBlockNr,
		pValue,
		4));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}

phStatus_t phalMfp_Sam_X_Transfer(void * pDataParams, uint8_t bTransferMaced, uint16_t wBlockNr)
{
	phStatus_t PH_MEMLOC_REM wStatus = 0;

	PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Sam_X_Int_MFP_CombinedWrite(
		pDataParams,
		(uint8_t) (PHAL_MFP_CMD_TRANS | bTransferMaced),
		wBlockNr,
		0x00,
		NULL,
		0));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}

phStatus_t phalMfp_Sam_X_Restore(void * pDataParams, uint8_t bRestoreMaced, uint16_t wBlockNr)
{
	phStatus_t PH_MEMLOC_REM wStatus = 0;

	PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Sam_X_Int_MFP_CombinedWrite(
		pDataParams,
		(uint8_t) (PHAL_MFP_CMD_REST | bRestoreMaced),
		wBlockNr,
		0x00,
		NULL,
		0));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}



/****************************************************************************************************************************************/
/* Mifare Plus Generic command for special commands.																					*/
/****************************************************************************************************************************************/
phStatus_t phalMfp_Sam_X_ResetAuth(void * pDataParams)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;

	/* Perform ResetAuth. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Int_ResetAuth(PHAL_MFP_RESOLVE_PAL_MIFARE_DATAPARAMS(pDataParams)));

	/* Reset the crypto layer */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Sam_X_ResetSecMsgState(pDataParams));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}

phStatus_t phalMfp_Sam_X_ResetSecMsgState(void * pDataParams)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;

    /* Kill only card Auth */
	PH_CHECK_SUCCESS_FCT(wStatus, PHHAL_HW_SAM_KILL_AUTHENTICATION(pDataParams, 0x01));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}

phStatus_t phalMfp_Sam_X_ChangeKey(void * pDataParams, uint8_t bChangeKeyMaced, uint16_t wBlockNr, uint16_t wKeyNo, uint16_t wKeyVer,
	uint8_t bDivInputLen, uint8_t * pDivInput)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM bPiccErrCode = 0;

	/* Exchange the informtion to Sam hardware. */
	wStatus = PHAL_MFP_CHANGE_KEY(
		pDataParams,
		(uint8_t) (bDivInputLen ? 0x02 : 0x00),
		(uint8_t) (bChangeKeyMaced ? PHAL_MFP_CMD_WRITE_EM : PHAL_MFP_CMD_WRITE_EN),
		wBlockNr,
		(uint8_t) wKeyNo,
		(uint8_t) wKeyVer,
		pDivInput,
		bDivInputLen,
		&bPiccErrCode);

	#ifdef NXPBUILD__PHAL_MFP_SAMAV3_X
	/* Updating to 0x90 because SAM AV3 returns 0x00 in case of Success. */
	if((wStatus == PH_ERR_SUCCESS) && (bPiccErrCode == 0x00))
	{
		bPiccErrCode = 0x90;
	}
	#endif /* NXPBUILD__PHAL_MFP_SAMAV3_X */

	/* Return the error code. */
	if ((wStatus == PH_ERR_SUCCESS)
		#ifdef NXPBUILD__PHAL_MFP_SAMAV3_X
		|| ((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV3_ERR_MIFARE_PLUS_GEN)
		#endif /* NXPBUILD__PHAL_MFP_SAMAV3_X */
	   )
	{
		/* Compute the response code. */
		PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Int_ComputeErrorResponse(1, bPiccErrCode, PH_ON));
	}
	else
	{
		return wStatus;
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}

phStatus_t phalMfp_Sam_X_ProximityCheck(void * pDataParams, uint8_t bNumSteps)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM bPiccErrCode = 0;

	/* Exchange the informtion to Sam hardware. */
	wStatus = phhalHw_SamAV2_Cmd_MfpProximityCheck(
		((phalMfp_SamAV2_X_DataParams_t *) pDataParams)->pHalDataParams,
		PH_OFF,
		0xFF,
		0xFF,
		bNumSteps,
		NULL,
		0,
		&bPiccErrCode);

	/* Return the error code. */
	if (((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV2_ERR_MIFARE_PLUS_GEN) ||
		((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV2_ERR_MIFARE_GEN))
	{
		/* Compute the response code. */
		PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Int_ComputeErrorResponse(1, bPiccErrCode, PH_ON));
	}
	else
	{
		return wStatus;
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);

}

#endif /* NXPBUILD__PHAL_MFP_SAM_X */
