/*
 * Copyright 2016 - 2017, 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
* LRP reference implementation
* Author: marcel.medwed@nxp.com
*/

#ifndef __LRP_H__
#define __LRP_H__

#ifndef _WIN32
#define _WIN32
#endif
/* #include<stdint.h> */
#include <phCryptoSym.h>
#define phRetCode_t uint16_t
#define EXPORT
#include <ph_TypeDefs.h>

#define LRP_DIM 4						/* fixed by the definition of LRP (i.e. m=4) */
#define LRP_BLOCK_SIZE 16				/* fixed by the use of AES */
#define LRP_CTR_SIZE	4				/* Current Encryption COunter size is 4 bytes */

#define LRP_GENERATEPLAINTEXTS_OK 1
#define LRP_GENERATEPLAINTEXTS_FAIL -1

#define LRP_COPY_OK 1
#define LRP_COPY_FAIL -1



#define LRP_CTR_ZERO_LENGTH_INPUT 2
#define LRP_CTR_OVERFLOW 3

#define LRP_ATTACK 4

#define LRP_MDC2_INCOMPLETE_BLOCK 5
#define LRP_MDC2_OK 6

#define LRP_MAC_OK 7

#define LRP_BLOCKCMPZERO_OK 0x55
#define LRP_BLOCKCMPZERO_NOK 0xAA

#define LRP_EVAL_DONT_FINALIZE 0x0
#define LRP_EVAL_FINALIZE 0x1


#define LRP_GENERATE_UPDATED_KEY_FIRST_DERIVATION 0x1
#define LRP_GENERATE_UPDATED_KEY_CONSECUTIVE_DERIVATION 0x0

#define LRP_CMAC_DONT_FINALIZE 0x0
#define LRP_CMAC_FINALIZE 0x1

#define LRP_ENCRYPT_CTR_SIZE            4

#define LRP_CEILDIV(value, precision) ((value + precision - 1) / precision)

#ifdef _WIN32

#else
#define pbDataCopro ((uint8_t*) SCOTTY_DATA0_ADDR)
#define pbKeyCopro ((uint8_t*) SCOTTY_KEY0_ADDR)
#define CRYPTO_START_ENCRYPTION        0x01
#define CRYPTO_START_DECRYPTION        0x02
#define CRYPTO_OPERATION_FINISHED      0x01
#endif

#ifdef _WIN32
	#if LRP_LEGACY_IMPLEMENTATION
	#define LRP_copro_MVKY_MACRO        LRP_copro_MVKY();
	#define LRP_copro_MVWK_MACRO        LRP_copro_MVWK();
	#define LRP_copro_RDWK_MACRO        LRP_copro_RDWK();
	#define LRP_copro_EXD_MACRO         LRP_copro_EXD();
	#define LRP_copro_EXK_MACRO         LRP_copro_EXK();
	#define LRP_Encrypt_MACRO            LRP_Encrypt();
	#define LRP_Decrypt_MACRO            LRP_Decrypt();
	#define LRP_WaitForCopro_MACRO      LRP_WaitForCopro();
	#define LRP_copro_EN_WUP_MACRO      LRP_copro_enable = 1;
	#define LRP_copro_DI_WUP_MACRO      LRP_copro_enable = 0;
	#define PHHAL_MEM_RET_COPY_OK 1
	#endif
#else
#define LRP_copro_MVKY_MACRO        CRYP_CONTROL.val = AESLC_CTRL_MV_KY; nop();nop();nop();
#define LRP_copro_MVWK_MACRO        CRYP_CONTROL.val = AESLC_CTRL_MV_WK; nop();nop();nop();
#define LRP_copro_RDWK_MACRO        CRYP_CONTROL.val = AESLC_CTRL_RD_WK; nop();nop();nop();
#define LRP_copro_EXD_MACRO         CRYP_CONTROL.val = AESLC_CTRL_EXC_WK; nop();nop();nop();
#define LRP_copro_EXK_MACRO         CRYP_CONTROL.val = AESLC_CTRL_EXC_KY; nop();nop();nop();
#define LRP_Encrypt_MACRO            CRYP_CONTROL.val = AESLC_CTRL_START_ENC;
#define LRP_Decrypt_MACRO            CRYP_CONTROL.val = AESLC_CTRL_START_DEC;
#define LRP_WaitForCopro_MACRO      nop(); go_idle();

#define LRP_copro_EN_WUP_MACRO        phhalSystem_SetInterruptLevel(15); DIG_CTRL.val |= (uint8_t)DIG_CTRL_CLK_CRYPTO_EN_MASK; nop(); CRYP_WAKEUP.val = AESLC_WUP_ENABLE;
#define LRP_copro_DI_WUP_MACRO        CRYP_WAKEUP.val = AESLC_WUP_DISABLE; DIG_CTRL.val &= (uint8_t)~DIG_CTRL_CLK_CRYPTO_EN_MASK; phhalSystem_SetInterruptLevel(0);
#endif

#if LRP_LEGACY_IMPLEMENTATION
	#define LRP_OK 0
	#define LRP_ERROR 1

	typedef struct SecretPlaintexts
	{
		uint8_t* pbSecretPlaintexts;	/* A pointer to the array where the secret plaintexts are stored. The storage needed is
										   2^LRP_DIM * LRP_BLOCK_SIZE bytes */
		uint8_t* pbUpdatedKey;			/* A pointer to the updated key, the storage needed is LRP_BLOCK_SIZE */
	} SecretPlaintexts_t;

	typedef struct InputVector
	{
		uint8_t bNumNibblesInput;		/* The number of input nibbles to be processed */
		uint8_t bNumNibblesOffset;		/* The nibble from where to start */
		uint8_t* pbInput;				/* A pointer to the input vector to be processed */
	} InputVector_t;

	typedef struct PlaintextVector
	{
		uint16_t wNumBytesInput;		/* The number of input bytes to be processed */
		uint8_t* pbInput;				/* A pointer to the input vector to be processed */
	} PlaintextVector_t;
#endif

phStatus_t phCryptoSym_Lrp_Encrypt(
                                  phCryptoSym_Sw_DataParams_t * pDataParams,
                                  uint16_t wOption,
                                  const uint8_t * pPlainBuffer,
                                  uint16_t  wBufferLength,
                                  uint8_t * pEncryptedBuffer
                                  );

phStatus_t phCryptoSym_Lrp_Decrypt(
                                  phCryptoSym_Sw_DataParams_t * pDataParams,
                                  uint16_t wOption,
                                  const uint8_t * pEncryptedBuffer,
                                  uint16_t  wBufferLength,
                                  uint8_t * pPlainBuffer
                                  );

phStatus_t phCryptoSym_Lrp_Decrypt_VariableIV(
                                  phCryptoSym_Sw_DataParams_t * pDataParams,
                                  uint16_t wOption,
								  uint8_t *pIv,
								  uint8_t wIvSize,
                                  const uint8_t * pEncryptedBuffer,
                                  uint16_t  wBufferLength,
                                  uint8_t * pPlainBuffer
                                  );
phStatus_t phCryptoSym_Lrp_LoadKeyDirect(phCryptoSym_Sw_DataParams_t *pDataParams,
								const uint8_t *pKey,
								uint16_t wKeyType
								);

phStatus_t phCryptoSym_Lrp_CalculateMac(
                                       phCryptoSym_Sw_DataParams_t * pDataParams,
                                       uint16_t wOption,
                                       const uint8_t * pData,
                                       uint16_t  wDataLength,
                                       uint8_t * pMac,
                                       uint8_t * pMacLength
                                       );

phStatus_t phCryptoSym_Lrp_Int_GenerateSessionAuthMasterKey(
                                       phCryptoSym_Sw_DataParams_t * pDataParams,
                                       const uint8_t * pData,
                                       uint16_t  wDataLength,
                                       uint8_t * pMac,
                                       uint8_t * pMacLength
                                       );

phStatus_t phCryptoSym_Lrp_Int_GenerateMac(
                                       phCryptoSym_Sw_DataParams_t * pDataParams,
									   uint16_t wOption,
                                       const uint8_t * pData,
                                       uint16_t  wDataLength,
                                       uint8_t * pMac,
                                       uint8_t * pMacLength
                                       );

phStatus_t phCryptoSym_Lrp_Int_GeneratePcdResponse(
                                       phCryptoSym_Sw_DataParams_t * pDataParams,
									   uint16_t wOption,
                                       const uint8_t * pData,
                                       uint16_t  wDataLength,
                                       uint8_t * pMac,
                                       uint8_t * pMacLength
                                       );


#if LRP_LEGACY_IMPLEMENTATION
/* ----------------------------------------------------------------------------
 * LRP abstraction layer
 * ---------------------------------------------------------------------------- */

/* LRP_GeneratePlaintexts uses pstSecretPlaintexts->pbUpdatedKey as input and generates the secret plaintexts. The plaintexts
                          are stored one after the other in pstSecretPlaintexts->pbSecretPlaintexts. Afterwards it
                          updates the key pstSecretPlaintexts->pbUpdatedKey
*/
EXPORT phRetCode_t LRP_GeneratePlaintexts(SecretPlaintexts_t* pstSecretPlaintexts);
EXPORT phRetCode_t LRP_GenerateUpdatedKey(uint8_t* pbUpdatedKey, uint8_t bOptions);

/* LRP_Eval evaluates the LRP on the input given by pstInput using the secret plaintexts and returns the result in pbOutput.
            pstInput is processed at nibble granularity from nibble pstInput->bNumNibblesOffset to
			pstInput->bNumNibblesOffset + pstInput->bNumNibblesInput - 1.
			A fault protected final transformation is applied if LRP_EVAL_FINALIZE is set. Otherwise LRP_EVAL_DONT_FINALIZE
			is required as a valid option.
*/
EXPORT phRetCode_t LRP_Eval(SecretPlaintexts_t* pstSecretPlaintexts, InputVector_t* pstInput, uint8_t* pbOutput, uint8_t finalize);

/* LRP_CTR uses LRP_Eval to encrypt several blocks in counter mode. For the first evaluation
           min(pstInput->bNumBytesInput, LRP_BLOCK_SIZE-pstInput->bInputBlockOffset) bytes of pbIV xor pstInput->pbInput are evaluated.
		   If a block boundary is hit, a final transformation is performed. For further blocks the output of the last block is xored with the input
		   TODO: currently the checkpoint mechanism is not efficient for longer messages, furthermore, the updatedKey is overwritten inside
*/
EXPORT phRetCode_t LRP_CTR(SecretPlaintexts_t* pstSecretPlaintexts, PlaintextVector_t* pstInput,  InputVector_t* pstIV, uint8_t* workSpace, uint8_t* pbOutput);

/* LRP_MAC implements a MAC following the hash-then-MAC principle. It differentiates between three cases.
		a.) 1 block message	+ finalization ....... a CMAC is directly calculated on the message block, this takes 34 AES encryptions
		b.) 1-n blocks (update) - finalization ... MDC-2 is used to process the message, this takes 2*n AES encryptions
		c:) 1-n blocks (update) + finalization ... MDC-2 is used to process the message including an MD strengthening, afterwards
		                                           the resulting hash digest is processed with CMAC, this takes 2*(n+1+34) AES encryptions.

		SecretPlaintexts_t* pstSecretPlaintexts ... read-only in this case
		PlaintextVector_t* pstInput ............... read-only in this case
		InputVector_t* pstCV ...................... the MDC-2 chaining value, pstCV->pbInput must point to 32 byte buffer, all other
		                                            fields will be set internally
		uint8_t* pbK1 ............................. needs to point to the CMAC key K1, although this could simply become a key ID (TODO)
		uint8_t* pbWorkspace ...................... 16 byte buffer used as workspace internally
		uint16_t* wBytesAlreadyProcessed .......... needs to be set to zero by the user before the first block is processed
		uint8_t finalize .......................... if set to LRP_MAC_FINALIZE a MAC is calculated otherwise chaining is performed

		wBytesAlreadyProcessed == 0 && finalize == LRP_MAC_FINALIZE ... case a.)
																		in this case, the buffer pstCV->pbInput is never used for MDC-2, thus
																		it can be reduced to 16 byte.
																		needed RAM: sizeof(SecretPlaintexts_t) + 16 + 16 bytes
																		(pstCV->pbInput + pbWorkspace)
		finalize == LRP_MAC_DONT_FINALIZE ............................. case b.)
																	    in this case, pstSecretPlaintexts can be NULL
																		needed RAM: 32 + 16 bytes
																		(pstCV->pbInput + pbWorkspace)
		wBytesAlreadyProcessed > 0 && finalize == LRP_MAC_FINALIZE .... case c.)
																		in this case, pbWorkspace can be NULL
																		needed RAM: sizeof(SecretPlaintexts_t) + 32 bytes
																		(pstCV->pbInput)
*/
EXPORT phRetCode_t LRP_MAC(SecretPlaintexts_t* pstSecretPlaintexts, PlaintextVector_t* pstInput, InputVector_t* pstCV, uint8_t* pbK1, uint8_t* pbOutput, uint16_t* wBytesAlreadyProcessed, uint8_t finalize);


/* LRP_CMAC implements a CMAC construction for which the block cipher has been exchanged by LRP_Eval

		SecretPlaintexts_t* pstSecretPlaintexts ... read-only in this case
		PlaintextVector_t* pstInput ............... read-only in this case. However, if it is write-able it can be shared with pstIV->pbInput.
		                                            That is, if pstIV->pbInput == pstInput->pbInput, then the first block of pstInput->pbInput
													will be overwritten, but no extra buffer for pstIV->pbInput is needed.
													Furthermore, if it is write-able, e.g. the second block of pstInput->pbInput can be used as
													pbWorkspace.
		InputVector_t* pstIV ...................... pstIV->pbInput needs to point to a 16 byte buffer. The initialization of the buffer
		                                            and the fields is done internally.
		uint8_t* pbK1 ............................. needs to point to the CMAC key K1, although this could simply become a key ID (TODO)
		uint8_t* pbWorkspace ...................... 16 byte buffer used as a workspace internally to prepare the key Kx
*/
EXPORT phRetCode_t LRP_CMAC(SecretPlaintexts_t* pstSecretPlaintexts, PlaintextVector_t* pstInput, InputVector_t* pstIV, uint8_t* pbK1, uint8_t* pbWorkspace, uint8_t finalize);

/* LRP_MDC2 implement an AES-128 based MDC-2 construction, g resets the leftmost bit of the left half, \tilde{g} set the leftmost bit of
		    the right half. In addition, a Merkle-Damgard strengthening is applied. That is, a 0x80 byte is appended and afterwards a 15 byte
			representation of the byte length in big-endian is appended such that its last byte is aligned with the last byte of the block.
			If the the message length is a multiple of the block length, the 0x80 padding and the length go into the same block, otherwise
			the length is written to an extra block.

		uint8_t* pbIV ...................... a 32 byte buffer where the chaining value is stored.
		uint16_t* wBytesAlreadyProcessed ... if this value is 0, the chaining value bytes are set to 0x52 and 0x25 respectively. Otherwise
											 this value is increased according to the processed bytes and used in the MD strengthening
		PlaintextVector_t* pstInput ........ input to be hashed
		uint8_t* pbWorkspace ............... a 16 byte buffer which is used for the MD strengthening
		uint8_t finalize ................... indicates whether the finalization shall be performed.
*/
EXPORT phRetCode_t LRP_MDC2(uint8_t* pbIV, uint16_t* wBytesAlreadyProcessed, PlaintextVector_t* pstInput, uint8_t* pbWorkspace, uint8_t finalize);


/* ----------------------------------------------------------------------------
 * LRP helper layer
 * ---------------------------------------------------------------------------- */
EXPORT void LRP_Encrypt();
EXPORT void LRP_Decrypt();
void LRP_DoFinalTransformation(uint8_t* pbOutput);
void LRP_DoFinalTransformationCTR(uint8_t* pbPlaintext, uint8_t* pbKey, uint8_t bNumBytesPt);
void LRP_DecryptAndCheckForZero();
EXPORT void LRP_WaitForCopro();
EXPORT phRetCode_t LRP_BlockCopy(uint8_t* pbSrc, uint8_t* pbDest);
EXPORT void LRP_VarCopy(uint8_t* pbSrc, uint8_t* pbDest, uint8_t bNumBytes);
void LRP_BlockXor(uint8_t* pbSrc, uint8_t* pbDest);
void LRP_VarXor(uint8_t* pbSrc, uint8_t* pbDest, uint8_t bNumBytes);
EXPORT void LRP_BlockSet(uint8_t bValue, uint8_t* pbDest);
phRetCode_t LRP_BlockCmpZero(uint8_t* pbArray);
EXPORT phRetCode_t LRP_VarCmp(uint8_t* pbSrc1, uint8_t* pbSrc2, uint8_t bNumBytes);
void LRP_HandleAttack();
void LRP_PrintBlock(char* prefix, uint8_t* block);
void LRP_CTR_Increment_IV(InputVector_t* pstIV);
uint8_t LRP_CheckForOverflowAndGetNeededNibbles(PlaintextVector_t* pstInput, InputVector_t* pstIV);
EXPORT void LRP_CMAC_kTimesX(uint8_t* pbOutput);
void LRP_MDC2_Core(uint8_t* pbIV, uint8_t* pbDataBlock, uint8_t bLength);
EXPORT void LRP_copro_EXK();
EXPORT void LRP_copro_EXD();
EXPORT void LRP_copro_INIT();
EXPORT void LRP_copro_RDWK();
EXPORT void LRP_copro_RDKY();
EXPORT void LRP_copro_MVWK();
EXPORT void LRP_copro_MVKY();
EXPORT void LRP_copro_XRWK();

#else

	typedef enum {
		LRP_OK = 0,		/**< LRP operation result is OK */
		LRP_ERROR = 1	/**< LRP operation result is NOT OK, any output parameter should not be used */
	} LRP_Result_t;

	/**
	* @brief Generates LRP plaintexts and updated keys from the supplied key.
	* According the keyGen algorithm specified in the [LRP whitepaper].
	* @param[out] pt_table		Generated LRP plaintexts; the required storage size is (16 * 16) bytes.
	* @param[out] ukey_table	Generated updated keys; the required storage size is (q_num_ukeys * 16) bytes.
	* @param[in] key			The key from which the plaintexts and updated keys are derived; the required key size is 16 bytes.
	* @param[in] q_num_ukeys	The required number of updated-keys.
	* @return					LRP_OK if the function operated correctly, else LRP_ERROR
	*/
	LRP_Result_t LRP_key_gen(uint8_t * pt_table, uint8_t * ukey_table, const uint8_t * key, uint8_t q_num_ukeys);

	/**
	* @brief Computes the LRP function evaluation from the LRP plaintexts, an updated key and an input vector IV.
	* The function optionally also performs the finalization step.
	* According the evalLRP algorithm specified in the [LRP whitepaper].
	* @param[out] result		Evaluation result; the required storage size is 16 bytes.
	* @param[in] pt_table		LRP plaintexts; the required plaintexts size is (16 * 16) bytes.
	* @param[in] updated_key	The key from which the plaintexts and updated keys are derived; the required key size is 16 bytes.
	* @param[in] input_vector	Input for the evaluation; the required input size is ((lenIV_nibbles + 1) / 2) bytes.
	* @param[in] finalize		Perform the finalization step (value 0x01) or dont (value 0x00).
	* @return					LRP_OK if the function operated correctly, else LRP_ERROR
	*/
	LRP_Result_t LRP_eval(uint8_t * result, const uint8_t * pt_table, const uint8_t * updated_key, const uint8_t * input_vector, uint8_t lenIV_nibbles, uint8_t finalize);

	/**
	* @brief Performs the LRP encryption according option A from the LRP plaintexts, an updated key and an input plaintext.
	* According the CTR_LRP algorithm specified in the [LRP whitepaper].
	* Actually the algorithm internally employs a stream cipher, performing an xor of the input with the cipher stream, resulting in the output.
	* @param[out] ciphertext	Computation result; the required storage size is ptsize_bytes bytes.
	* @param[in] plaintext		Computation input; the required plaintexts size is ptsize_bytes bytes.
	* @param[in, out] counter	The unsigned counter is incremented for every processed 16-byte input block, and once more if the input is larger than a whole number of blocks. The counter wraps around from 0xff..0xff to 0x00..0x0. The required size is countersize_bytes.
	* @param[in] countersize_bytes	The size of the counter in bytes.
	* @param[in] ptsize_bytes	The size of the plaintext in bytes.
	* @param[in] pt_table		LRP plaintexts; the required plaintexts size is (16 * 16) bytes.
	* @param[in] updated_key	The required key size is 16 bytes.
	* @return					LRP_OK if the function operated correctly, else LRP_ERROR
	*/
	LRP_Result_t LRP_CTR(uint8_t * ciphertext, const uint8_t * plaintext, uint8_t * counter, uint8_t countersize_bytes, uint16_t ptsize_bytes, const uint8_t * plaintexts_array, const uint8_t * updated_key);


	/**
	* @brief Performs the LRP encryption according option B from the LRP plaintexts, an updated key and an input plaintext.
	* According the LRICB algorithm specified in the [LRP whitepaper].
	* This algorithm operates as a block cipher. The function optionally performs padding. In case the input is not a multiple of 16 bytes in length, padding is mandatory.
	* @param[out] ciphertext	Encryption result; the required storage size is ptsize_bytes bytes if padding is not performed. If padding is performed, the required storage size = up to 16 bytes larger than ptsize_bytes.
	* @param[in] plaintext		Encryption input; the required size is ptsize_bytes bytes.
	* @param[in, out] counter	The unsigned counter is incremented for every processed 16-byte block, including for the padded block if padding is performed. The counter wraps around from 0xff..0xff to 0x00..0x0. The required size is countersize_bytes.
	* @param[in] countersize_bytes	The size of the counter in bytes.
	* @param[in] ptsize_bytes	The size of the plaintext in bytes.
	* @param[in] pt_table		LRP plaintexts; the required plaintexts size is (16 * 16) bytes.
	* @param[in] updated_key	The required key size is 16 bytes.
	* @param[in] pad			Perform padding (value 0x01) or dont (value 0x00).
	* @return					LRP_OK if the function operated correctly, else LRP_ERROR
	*/
	LRP_Result_t LRP_LRICB_encrypt(uint8_t * ciphertext, const uint8_t * plaintext, uint8_t * counter, uint8_t countersize_bytes, uint16_t ptsize_bytes, const uint8_t * plaintexts_array, const uint8_t * updated_key, uint8_t pad);

	/**
	* @brief Performs the LRP decryption according option B from the LRP plaintexts, an updated key and an input ciphertext.
	* According the LRICB algorithm specified in the [LRP whitepaper].
	* This algorithm operates as a block cipher. The function optionally strips padding. If padding is selected at the decryption, padding must have been applied at the encryption.
	* @param[out] plaintext		Decryption result; the required storage size is ctsize_bytes bytes, or 1 less if padding is performed.
	* @param[in] ciphertext		Decryption input; the required ciphertext size is ctsize_bytes bytes, being a multiple of 16 bytes.
	* @param[in, out] counter	The unsigned counter is incremented for every processed 16-byte block. The counter wraps around from 0xff..0xff to 0x00..0x0. The required size is countersize_bytes.
	* @param[in] countersize_bytes	The size of the counter in bytes.
	* @param[out] ptsize_bytes	The size of the plaintext in bytes.
	* @param[in] ctsize_bytes	The size of the ciphertext in bytes.
	* @param[in] pt_table		LRP plaintexts; the required plaintexts size is (16 * 16) bytes.
	* @param[in] updated_key	The required key size is 16 bytes.
	* @param[in] pad			Strip padding (value 0x01) or dont (value 0x00).
	* @return					LRP_OK if the function operated correctly, else LRP_ERROR
	*/
	LRP_Result_t LRP_LRICB_decrypt(uint8_t * plaintext, const uint8_t * ciphertext, uint8_t * counter, uint8_t countersize_bytes, uint16_t * ptsize_bytes, uint16_t ctsize_bytes, const uint8_t * plaintexts_array, uint8_t * updated_key, uint8_t pad);

	/**
	* @brief Performs the LRP CMAC from the LRP plaintexts, an updated key and an input message.
	* According the CMAC_LRP algorithm specified in the [LRP whitepaper].
	* @param[out] result		Mac'ing result; the required storage size is 16 bytes.
	* @param[in] msg			Input message the required size is msg_len_bytes bytes.
	* @param[in] msg_len_bytes	The size of the input message in bytes.
	* @param[in] plaintexts_array		LRP plaintexts; the required plaintexts size is (16 * 16) bytes.
	* @param[in] updated_key	The required key size is 16 bytes.
	* @return					LRP_OK if the function operated correctly, else LRP_ERROR
	*/

	LRP_Result_t LRP_CMAC(uint8_t * result, const uint8_t * msg, uint16_t msg_len_bytes, const uint8_t * plaintexts_array, const uint8_t * updated_key);


/**
* @brief Performs the LRP CMAC from the LRP plaintexts, an updated key and an input message.
* The function can be used to build the LRP CMAC incrementally by calling the function repeatedly.
* For the first call, the supplied iv must be all zeroes.
* For all calls but the last, the input message size must be a multiple of 16 bytes.
* For all calls but the last, the finalize value must be 0, it must be 1 for the last call.
* For all calls but the last, the Mac'ing result contains the CMAC state, this value must be supplied as iv with the next call.
* For the last call, the Mac'ing result contains the CMAC value for the complete message.
* According the CMAC_LRP algorithm specified in the [LRP whitepaper].
* @param[out] result		Mac'ing result; the required storage size is 16 bytes.
* @param[in] msg			Input message the required size is msg_len_bytes bytes.
* @param[in] msg_len_bytes	The size of the input message in bytes.
* @param[in] iv				State of the CMAC; the required storage size is 16 bytes.
* @param[in] plaintexts_array		LRP plaintexts; the required plaintexts size is (16 * 16) bytes.
* @param[in] updated_key	The required key size is 16 bytes.
* @param[in] finalize		The required key size is 16 bytes.
* @return					LRP_OK if the function operated correctly, else LRP_ERROR
*/
LRP_Result_t LRP_CMAC_incremental(uint8_t * result, const uint8_t * msg, uint16_t msg_len_bytes, const uint8_t * iv, const uint8_t * plaintexts_array, const uint8_t * updated_key, uint8_t finalize);

#endif
#endif
