/** ###################################################################
**     Filename  : globals.h
**     Abstract  :
**         This module contains global types and definitions
**
** ###################################################################*/

#ifndef __GLOBALS_H
#define __GLOBALS_H

/******************************************************************************
* Constants and macros
******************************************************************************/
#define SCALE_VOLTAGE 9.0	/* Volts */
#define SCALE_CURR 1.0 /* Amps */
#define SCALE_MOTOR_RPM 10000.0 /* RPM */
#define MOTOR_RPM_MAX 8600.0 /* RPM */
#define MOTOR_COMMUTATIONS_PER_REV 24 /* Motor commutations per one motor revolution */
#define MOTOR_CURRENT_LIMT 0.5 /* Amps */
#define I_SAMPLE_DELAY 10 /* pwm_clk */

/* Under-voltage fault and over-voltage fault levels */
#define UNDERVOLTAGE	6.0 /* Volts */
#define OVERVOLTAGE 	12.0 /* Volts */

/* BLDC motor speed limits */
#define SPEED_CMD_MIN FRAC16(500.0/SCALE_MOTOR_RPM) /* Min motor speed = 500 RPM */
#define SPEED_CMD_MAX FRAC16(MOTOR_RPM_MAX/SCALE_MOTOR_RPM) /* Max motor speed = 8600 RPM */
#define SPEED_CMD_INC FRAC16((10*SCALE_MOTOR_RPM)/(100*SCALE_MOTOR_RPM)) /* Button inc step = 1/10 */
#define SPEED_CMD_PROCESS_INIT {SPEED_CMD_INC, SPEED_CMD_MIN, SPEED_CMD_MAX, 1}
#define SPEED_RAMP_INC FRAC16(5000.0/(SCALE_MOTOR_RPM*500)) /* Accel rate = 5000 RPM/sec */

 

/* System variables */
#define PWM_MODULO 5000
#define IP_BUS_CLK_MHZ 100.0	/* MHz*/
#define TMRA0_PRSC	128 /* TMR0 Prescaler */
#define TMRA0_CLK ((IP_BUS_CLK_MHZ * 1E6)/TMRA0_PRSC) /* TMRA0 Clock Frequency*/
/* Dividend constant used to evaluate motor revolutions from timer tics
   captured by Hall sensor edges */
#define SPEED_DIV_CONST (Word16)((TMRA0_CLK * 60 * 3)/(2 * MOTOR_RPM_MAX * MOTOR_COMMUTATIONS_PER_REV))
#define LOW_SPEED_THRESHOLD 1400 /* PWM reload cycles - 1400*100us ~ 140ms */

/* Motor Current Limit Constant in Frac16 */
#define MOTOR_CURRENT_LIMT_F16 FRAC16(MOTOR_CURRENT_LIMT/(2*SCALE_CURR))

/* initialize PI speed controller parameters */
#define SPEED_CTRL_PARAMS_INIT \
{\
	16384, /* ProportionalGain */\
	16384, /* IntegralGain */\
	0,  /* IntegralPortionK_1 */\
	0x7FFF, /* PositivePILimit */\
	0x0, /* NegativePILimit */\
	-1, /*  ProportionalGainScale */\
	5 /*  IntegralGainScale */\
}

/* initialize PI current controller parameters */
#define CURRENT_CTRL_PARAMS_INIT\
{\
	16384, /* ProportionalGain */\
	16384, /* IntegralGain */\
	0x7FFFFFFF,  /* IntegralPortionK_1 */\
	0x7FFF, /* PositivePILimit */\
	0x0, /* NegativePILimit */\
	-2, /* ProportionalGainScale */\
	0 /* IntegralGainScale */\
}

/******************************************************************************
* Global Types
******************************************************************************/
/* Definition of Control Bits of Application  Status and Control Word (appControl) */
typedef union
{
    struct
    {
        unsigned int pwm_on					: 1; /**< PWM Output Enable On/Off */
        unsigned int lowspeed				: 1; /**< Low Speed Region */
        unsigned int button_up			: 1; /**< Push button up pressed */
        unsigned int button_down		: 1; /**< Push button down pressed */
        unsigned int Bit4						: 1; /* RESERVED */
        unsigned int Bit5        		: 1; /* RESERVED */
        unsigned int Bit6           : 1; /* RESERVED */
        unsigned int faultclr       : 1; /**< Fault Clear */
        unsigned int faultpending   : 1; /* < Fault Pending */
        unsigned int Bit9           : 1; /* RESERVED */
        unsigned int Bit10		      : 1; /* RESERVED */
        unsigned int Bit11          : 1; /* RESERVED */
        unsigned int lv_fault       : 1; /**< Low-Voltage Fault */
        unsigned int ov_fault       : 1; /**< Over-Voltage Fault */
        unsigned int oi_fault       : 1; /**< Over-Current Fault */
        unsigned int fault          : 1; /**< Application Fault */
    } B;
    uint16_t W16;
} APPCONTROL_T;     /* Main application state machine control word */


/******************************************************************************
* Function headers
******************************************************************************/
void ComutateBLDC(unsigned int direction);

#endif /* __GLOBALS_H */