/******************************************************************************
* 
* Copyright (c) 2012 Freescale Semiconductor;
* All Rights Reserved                       
*
*******************************************************************************
*
* THIS SOFTWARE IS PROVIDED BY FREESCALE "AS IS" AND ANY EXPRESSED OR 
* IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
* OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  
* IN NO EVENT SHALL FREESCALE OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
* INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
* SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
* HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
* STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
* IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF 
* THE POSSIBILITY OF SUCH DAMAGE.
*
/*!
********************************************************************************
*
* FILE NAME: main.c   
*
* TARGET: MC56F8400 device
*
*******************************************************************************
*
* This application demonstrates the PWMs, Timer, ADC, XBAR and PDB on 56F847xx
* 
* 1. Each peripheral is configured in separate .c and .h file
* 
* 2. PWM_A generates 3 pairs complementary PWM signals with frequency 16kHz 
*    dead time 1us. Outputs are available on GPIOE 0-5. Trigger 0 from PWMA_SM0 
*    is enabled. Reload ISR is optional.
*    
* 3. PWM_B generates 3 pairs complementary PWM signals with frequency 10kHz 
*    dead time 1.5us. Outputs are available on GPIOG 0-3 and GPIOE 6&7.
*    Trigger 0 from PWMB_SM0.
*    
* 4. Timers A0 and A1 are configured in cascade mode to generate 1sec pulse.
*    CMP ISR can be enabled.
* 
* 5. ADC cyclic 12bit samples in channels ANA0 (GPIOA0) and ANB0 (GPIOB0) 
*   in simultaneous mode and clock 10MHz. Conversion is triggered using SYNC0
*   pulse via XBARA channel. End of Scan ISR can enabled.
* 
* 6. PDB module is configured to provide synchronization between two peripherals 
*    via XBARA channel. As input is assigned PWMA SM0 trigger 0
*
* 7. Each ISR can be enabled or disabled. Set in dedicated X_test.c files
* 
* 8. Debug pins GPIOC 14&15 are configured as XBAR OUT 4&5. 
*    GPIOF6 is set as output pin to control LED8 on TWR-MC56F8400
* 
* 9. FreeMASTER communicates via USB-TAP
* 
* 10. PWM value register values can be set using bPWM_Update set to 1 
*     from FreeMASTER
***************************************************************************//*!

/* Including needed modules to compile this module/procedure */
#include "Events.h"
#include "Main.h"
#include "Types.h"
#include "derivative.h" /* include peripheral declarations */

#include "freemaster.h"




/********************************* CONSTANTS *************************************/

/********************************* GLOBAL VARIABLES *************************************/
/* PWM module */
static bool   bPWM_Update;

/* variable structure defined in dedicated files */
PWMA_REG_UPDATE  udtPWMAreg;
PWMB_REG_UPDATE  udtPWMBreg;
ADC_RESULT		 udtADCresults;	

/************************************ FUNCTIONS DEFINITIONS *****************************/
void CLKO_enable(void);
void Debug_Pins_Init(void);
extern void PWM_A_Init(PWMA_REG_UPDATE *ptr);
extern void PWM_A_Update(PWMA_REG_UPDATE *ptr);
extern void PWM_B_Init(PWMB_REG_UPDATE *ptr);
extern void PWM_B_Update(PWMB_REG_UPDATE *ptr);
extern void ADC12_Init(void);
extern void PDB_Init(void);



/************************************** FUNCTIONS ***************************************/
void CLKO_enable(void)
{
	SIM_PCE0 |= SIM_PCE0_GPIOF; /* Enable GPIOF Clock - pin 1 */
	SIM_GPSFL = SIM_GPSFL & ~(SIM_GPSFL_F1); 	// GPIOF1 as CLKO_1
	GPIOF_PER |= GPIOF_PER_PE_1; /* PER Enable for GPIOF1 */            
	 /* CLKOUT_ = MASTER Clock */	
	SIM_CLKOUT = SIM_CLKOUT_CLKOSEL1 & (SIM_CLKOUT_CLKOSEL1_0 |
										SIM_CLKOUT_CLKOSEL1_1 |
								      (~SIM_CLKOUT_CLKOSEL1_2));	 
}

void Debug_Pins_Init(void)
{
	SIM_PCE0 |= SIM_PCE0_GPIOC; // Enable GPIOC clock
	GPIOC_DDR |= GPIOC_DDR_DD_14;
	GPIOC_PER |= GPIOC_PER_PE_14;
	GPIOC_DRIVE |= GPIOE_DRIVE_DRIVE_14 ;
	SIM_GPSCH |= SIM_GPSCH_C14 | SIM_GPSCH_C15; 	/* GPIOC14 as XB_OUT4 */
	
	GPIOF_DDR |= GPIOF_DDR_DD_6;    /* GPIOF6 as output */
	
	/* PWMB_SM0 TRIGGER on XOUT4 & XOUT5*/
	XBARA_SEL2	= 0x141CU;
}

/************************************** MAIN ********************************************/
void main()
{
	
	/*Set the interrupt vectors base address*/
	setReg16(INTC_VBA, 0x0000);  
	/*Disable COP running after reset*/
	setReg16(COP_CTRL, 0x00);		
	
	/* Initialize peripherals clock, LED indicator ports and system interrupt*/
	PE_low_level_init();
	OCCS_Init();
	
  	/* Timer initialization function */
  	Timer_Init();
  	
  	/* initialize FreeMASTER */
  	FMSTR_Init();
	
	/* Testing functions */
  	CLKO_enable();
  	
  	/* configure debug pins */
  	Debug_Pins_Init();
  	
  	/* call PWMA initialization function */
  	PWM_A_Init(&udtPWMAreg);
  	
  	/* call PWMB initialization function */
  	PWM_B_Init(&udtPWMBreg);
  	
  	
  	/* call PDB initialization function */
  	PDB_Init();
  	
  	/* call ADC12 initialization function */
  	ADC12_Init();

  	/* enable interrupts with priority (x) */
  	__EI(0);

  	
  	
  	for(;;) 
  	{
  		
  		/* PWM Register Update  */
  		if (bPWM_Update)
  		{
  			/* update PWMA value registers */
  			PWM_A_Update(&udtPWMAreg);
  			
  			/* update PWMA value registers */
  			PWM_B_Update(&udtPWMBreg);
  			
  			/* clear FreeMASTER variable */
  			bPWM_Update = FALSE;
  		}
  		 		
  		/* The FreeMASTER polling call must be called periodically in the main 
       	* application loop to handle the communication interface and protocol. 
       	* Only in LONG_INTR FreeMASTER interrupt mode, all the processing is done 
       	* during the communication interrupt so the FMSTR_Poll() is not needed
       	* (the function is compiled empty in this case).*/
       	FMSTR_Poll(); 
  
  	}
}

/* END Main */
