/*****************************************************************************
* File:    bootloader_lib.h
*
* Purpose: Provide utilities for MC56F80xx serial bootloader application.
* Dept:    Freescale TSPG MCD DSC Team (Tempe, AZ)
* Email:   support@freescale.com
* Date:    2005.08.15
******************************************************************************
* Notes:
*
*****************************************************************************/

#ifndef _BOOTLOADER_LIB_H
#define _BOOTLOADER_LIB_H   1

#include "bootloader_cfg.h"
#include "flash.h"

/********************************** START OF PORT ********************************/
// for Nevis
#define BOOT_START_ADDRESS_PLACE  (PFLASH_END_ADDR-2)	// location that contains application start address
#define BOOT_START_DELAY_PLACE    PFLASH_END_ADDR		// location that contains boot timeout in seconds

// Flash Parameters and Functions
#define FLASHP_PAGE_SIZE    PFLASH_SECTOR_SIZE     // PFlash Page Size (in words)
#define FLASHD_PAGE_SIZE    DFLASH_SECTOR_SIZE     // DFlash Page Size (in words)

#define FLASH_CONFIG_PAGE	0x200	// flash configuration fields address


#define FLASH_SECURITY	    0xBE   	// enable flash security and backdoor key


#define fmerase(x)          hfm_command(FTFL_ERASE_BLOCK, (UInt16*)FLASH_LO, 0, 1);
#define fperase(x)          hfm_command(FTFL_ERASE_SECTOR, (UInt16*)x, 0, 1);
#define fprogram(x,y,z)     hfm_command(FTFL_PROGRAM_LONGWORD, (UInt16*)x, (UInt16*)y, z);


// Peripheral Access


#if	defined(USE_SCI0)
	#define QSCI_RATE       0x0E080  		// SCI0 RATE register address
#elif	defined(USE_SCI1)
	#define QSCI_RATE       0x0E090  		// SCI1 RATE register address
#elif	defined(USE_SCI2)
	#define QSCI_RATE       0x0E0A0  		// SCI2 RATE register address
#endif
#define QSCI_CTRL1      (QSCI_RATE+1)  
#define QSCI_CTRL2      (QSCI_RATE+2)  
#define QSCI_STAT       (QSCI_RATE+3)  
#define QSCI_DATA       (QSCI_RATE+4)  


//
#define Timer_Disable() TMRA_0_CTRL &= ~0x2000
/********************************** END OF PORT ********************************/

// Receiver Buffer Info, Status Flags, and Error Codes
#define ERR_SREC_CKSUM      0x0001
#define ERR_SREC_TYPE       0x0002
#define ERR_HFM_ACCERR      0x0004
#define ERR_HFM_PVIOL       0x0008
#define ERR_HFM_PROG        0x0010
#define ERR_ADDR_START      0x0020
#define ERR_SCI_RX          0x0040
#define ERR_ADDR_RANGE      0x0080		// out of given flash address range
#define TX_XOFF             0x1000
#define RX_HEADER           0x2000
#define RX_END_OF_FILE      0x4000
#define DOWNLOAD_OK         (RX_HEADER|RX_END_OF_FILE)
#define DOWNLOAD_ERROR      0x00FF
#define RX_DATA_SIZE        1024 //550
#define MAX_SREC_WORDS		128		

// ASCII Character Values
#define XON             17
#define XOFF            19
#define NEWLINE         10
#define RETURN          13

// Application State Machine
#define INITIAL_STATE       ((UInt16)0)
#define WAIT_FOR_S          ((UInt16)1)
#define WAIT_FOR_0          ((UInt16)2)
#define WAIT_FOR_EOL        ((UInt16)3)
#define ERROR_HANDLER       ((UInt16)4)
#define APPLICATION_DONE    ((UInt16)5)
#define PROCESS_DATA        ((UInt16)8)

// Miscellaneous
#define bitset(x,y)     (y |= x)
#define bitclr(x,y)     (y &= ~x)


// Global Variables
extern UInt16 bytesum;
extern UInt16 status;
extern char *data_in;
extern char *data_out;
extern char hex_string[5];
extern char rx_data[RX_DATA_SIZE];

// Linker Command File Variables
extern word _vba;


// Function Prototypes
extern char get_char(char **ptr);
extern char int_to_char(UInt16 i);
extern char *int_to_string(UInt16 i);
extern void fisr_sci_rx_full(void);
extern void mem_init(register char *mem, register char init, register UInt16 sz);
extern void sci_tx(char *msg);
extern void sci_tx_char(char msg);
extern void srec_decode(void);
extern UInt16 char_to_int(char c);
extern UInt16 get_byte(char **ptr);
extern UInt16 hfm_command(UInt16 cmd, UInt16 *addr, UInt16 *data, UInt16 cnt);
extern void pmemcpy(register word *pSrc, register word *pDst, register dword sizeWords);
/////////////////////////////////////////////////////////////////////////////
// I N L I N E D   F U N C T I O N S
/////////////////////////////////////////////////////////////////////////////
inline unsigned int pmem_read(register UInt16 *addr)
{
   register unsigned int data;

   asm(move.w  p:(addr)+,data);
   return data;
}

inline void pmem_write(register UInt16* addr, register UInt16 data)
{
   asm(move.w  data,p:(addr)+n);
}


#endif	//BOOTLOADER_LIB_H