/** ########################################################################################################################
**     Filename  : bootloader.C
**     Project   : bootloader
**     Processor : 56F82xx
**     Version   : Driver 01.10
**     Compiler  : Metrowerks DSP C Compiler
**     Date/Time : 4/13/2006, 2:06 PM
**     Abstract  : Static/Dynamic Serial bootloader for 56F82xx. 
**     Author:	   William Jiang
**     Description:
**     The configuration is done via DYNAMIC_BOOT macro: 
**     
**     if it is defined, then the bootloader 
**     will  always stay in the flash 
**     (so-called static bootloader) 
**     and if it is undefined, 
**     then the bootloader can only load application code once 
**     (next time after reset, application code will run).
**     The configuration file is bootloader_cfg.h.
**     The default SCI bit rate is 57600. Y
**     ou may change it to below 115200bps.
**     
**     The  default SCI port is SCI0 
**     and SCI pins used are GPIOC2 and GPIOF8. 
**     You can also change SCI port  
**     by defining one of the following macros:
**     USE_SCI0
**     USE_SCI1
**     USE_SCI2
**     
**     You can also change SCI pins  
**     by defining one of the following macros 
**     according to the given SCI port:
**     SCI0_PIN_GPIOF11_GPIOF15
**     SCI1_PIN_GPIOC11_GPIOC12
**     SCI2_PIN_GPIOF9_GPIOF10
**     SCI2_PIN_GPIOD5_GPIOD6
**     
**     Please make sure that 
**     Terminal setting be 8-N-1 format 
**     (8-bit data, no parity and 1 stop bit) 
**     with flow control of Xon/Xoff.
**     
**     Please also note that if your chip is 
**     untrimmed version with default trim value for internal ROSC 8MHz, 
**     you need specify a trim value which will be written to trim register.
**     
**     To do this, uncomment the definition for CLOCK_UNTRIMMED macro 
**     in bootloader_cfg.h and
**     modify macro value for TRIM_VALUE. 
**     
**     For trimmed silicons, please comment the definition of CLOCK_UNTRIMMED macro.
**     
**     
**     The CPU interrupt must be disabled 
**     shadow registers initialization.
				
 	   		    It also requires modification of user's application code to support static bootloader; 
 				on _EntryPoint function (after vector table) the following sequence must be
 				placed:
				extern unsigned int _vba;
				INTC_VBA = (int)&_vba >>7;  // restore VBA register to point to user defined vector table
 				vba is defined in the linker command file as below:
			        .interrupt_vectors :
			        {
			          # interrupt vectors
			          F_vba = .;
			          * (interrupt_vectors.text)
			        } > .p_Interrupts
			    where interrupt_vectors.text is defined as below
				#pragma define_section interrupt_vectors "interrupt_vectors.text"  RX
				#pragma section interrupt_vectors begin
				static asm void _vect(void) {
				  JMP  0x7C00          # Hardware reset vector must jump to this address for static bootloader         
				  JMP  0x7C00          # COP reset vector must jump to this address           
				  JSR  Cpu_Interrupt                   
				  JSR  Cpu_Interrupt                   
				  JSR  Cpu_Interrupt                  
				  JSR  Cpu_Interrupt  
				  ...
					}
				 #pragma section interrupt_vectors end
				                  
			    NOTE: 
			    The above first two reset vectors must replaces those defined by ProcessorExpert or CodeWarrior.   	
 				Users can change the starting address of the vector table to fit their applications.
  				
 				Because the bootloader reads boot delay value
 				from address 0x7BFF,users must define the boot delay value in 0x7BFF by adding a memory segment .xBootCfg 
 				and a memory section .ApplicationConfiguration  in
 				the linker command file  
 				as below:
     			#Serial bootloader configuration section
    			.xBootCfg (RWX) : ORIGIN = 0x00007BFE, LENGTH = 0x2
 				
 
 
 				#Serial bootloader configuration section
		        .ApplicationConfiguration :
		        {
		        	  # Application Entry Point must be written here
		              WRITEH(F_EntryPoint);
		              
		              # Bootloader start delay in seconds
		              WRITEH(10);      # 10 is the boot delay value in seconds before bootloader
		              				   # starts up the application code
		              
		        } > .xBootCfg
 		
**		Change History:	
**			
**			Aug. 12,2009	combined static/dynamic feature.
**			Aug. 18,2009    increased NO_FLASH_PAGES_ERASE by 1 and bootloader start address to
**							0x7C00. Also changed the application entry point location and
**                          boot delay value location to 0x7BFE and 0x7BFF respectively.
**          Aug. 19,2009    separated the bootloader into two targets: one for 56F824x and the other for
**                          56F825x due to different memory map
**          Aug. 20,2009    added support to program flash configuration fields set up by applications.
**			Sept.10,2009	changed the default SCI pin settings to GPIOC2 (TXD0),GPIOC3(RXD0) for all packages
**
** #########################################################################################################################
*/
//#define DEBUG_FLASH_CODE
/* MODULE bootloader */

/* Including used modules for compiling procedure */
#include "Cpu.h"
#include "Events.h"
/* Including shared modules, which are used for whole project */
#include "Types.h"
//#include "Registers_Nevis.h"
#include "MC56F84789.h"
/* Include shared modules, which are used for whole project */

/* Include custom modules */
#include "bootloader_lib.h"
#include "sci.h"


UWord32  StartAddress;
UWord16  StartDelay;
char address[4];

char StrCopyright[]     = "\r\nMC56F84xx Serial Bootloader v1.0\r\n";
char StrWaitingSrec[]   = "Waiting for application S-Record.\r\n";
char StrLoaded[]        = "Download Complete\r\n";
char StrStarted[]       = "Application started!\r\n\0"; // NOTE: the last ending byte 0 is added 
														// to workaround the compiler issue with
														// even # of chars in the string that 
														// will not be padded with 0 by the compiler
char StrLock[]          = "Lock Flash?";
char StrNewLine[]       = "\r\n\0";
char StrErr[]           = "\r\nError: ";

// externals

// prototypes
void boot_state_machine(void);
void bootTimerCheck(void);
void bootExit(void);
void bootTimerInit(void);



/*****************************************************************************
* void main(void)
*
* Main application.
*
* In:  n/a
*
* Out: n/a
*****************************************************************************/
void main(void)
{
  // NOTE: see LCF for the _vba calculation
  INTC_VBA = ((word)&_vba);			   /* Set Vector Base Address */
  
  /*** Processor Expert internal initialization. DON'T REMOVE THIS CODE!!! ***/
  PE_low_level_init();
  /*** End of Processor Expert internal initialization.                    ***/

  
  /****************************************************************************
  / read start delay if no delay jump to user application
  /****************************************************************************/
    StartDelay = pmem_read((UInt16 *)BOOT_START_DELAY_PLACE) & 0x00FF;

    if(!StartDelay)  bootExit();
    else bootTimerInit(); 
   
  /* Write your code here */

   // Initialize R0,R1,M01, and N for circular buffer (including shadows)
   asm {
        swap shadows                    ;// Switch to shadow registers
        moveu.w #(RX_DATA_SIZE-1),M01   ;// Set buffer size
        moveu.w #rx_data,R0             ;// Set starting address
        moveu.w #QSCI_STAT,R1           ;// Use R1 for SCI status register
        moveu.w #0,N                    ;// N is unused
        swap shadows                    ;// Switch back to normal registers
   }

   // Initialize globals
   mem_init(rx_data, NULL, RX_DATA_SIZE);
   data_out = data_in = rx_data; 
   
   // Disable protection
   // NOTE:
   // If protected, NEED save this flash sector at 0x200 and then erase it,
   // reset the chip finally
   if(FTFL_ProtectionCheck())
   {
	   // Some sectors protected, need un-protect them
   }
      
   // Output banner
   sci_tx(&StrCopyright[0]);
   
   // Now it is safe to enable interrupts
   Cpu_EnableInt();
   
   // Start bootloader statemachine
   boot_state_machine();
}

#pragma optimization_level 4

#ifdef	DYNAMIC_BOOT
#pragma section pram_code begin
#endif
/*****************************************************************************
* void state_machine(void)
*
* Bootloader state machine which reads s-record from SCI, and decodes s-record
* as well as programs user's code/data into flash.
*
* In:  n/a
*
* Out: n/a
*****************************************************************************/

void boot_state_machine(void)
{
  UInt16 state;
  char *temp_ptr;
  char ch;
  Int16 i; 

   state = INITIAL_STATE;
   temp_ptr = rx_data;

   do  {
      // State: Initial State
      if (state == INITIAL_STATE) {
         status = 0;
         sci_tx(&StrWaitingSrec[0]);
         sci_tx_char(XON);
         state = WAIT_FOR_S;
      }

      if (data_in != temp_ptr) {
         Timer_Disable();

         // State: Wait for Header "S"
         if (state == WAIT_FOR_S) {
            temp_ptr = data_out;
            if (get_char(&temp_ptr) == 'S') { state = WAIT_FOR_0; }
            else { get_char(&data_out); }
         }
         // State: Wait for Header "0"
         else if (state == WAIT_FOR_0) {
            if (get_char(&temp_ptr) == '0') { state = WAIT_FOR_EOL; }
            else {
               get_char(&data_out);
               state = WAIT_FOR_S;
            }
         }
         // State: Wait for EOL
         else if (state == WAIT_FOR_EOL) {
            if (get_char(&temp_ptr) == '\r') {
               if (!(status & TX_XOFF)) {
                  sci_tx_char(XOFF);
                  status |= TX_XOFF;
               }
               srec_decode();
               temp_ptr = data_out;
            }
         }
      }
      else {
         if (status & TX_XOFF) {
            sci_tx_char(XON);
            status &= ~TX_XOFF;
         }
      }
      // State: Error
      if (status & DOWNLOAD_ERROR) {
         sci_tx(StrErr);
         sci_tx(int_to_string(status));
         state = INITIAL_STATE;
         sci_tx(StrNewLine);
      }
      
      bootTimerCheck();
      
   } while (status != DOWNLOAD_OK);
   
   sci_tx(StrNewLine); 
   sci_tx(&StrLoaded[0]);
  	
   bootExit();
} 
#ifdef	DYNAMIC_BOOT
#pragma section pram_code end
#endif 
 
 
#ifdef	DYNAMIC_BOOT
#pragma section pram_code begin
#endif
 

/***************************************************************************
* Functions:   bootTimerCheck()
*
* Description: Check TIMER specified in Timer bean
*
* Returns:     1 - timer is reached defined timevalue
*              0 - count in progress
*
* Notes:
*
**************************************************************************/
void bootTimerCheck(void)
{
    if(TMRA_1_SCTRL & TMRA_1_SCTRL_TCF)
    {
        TMRA_1_SCTRL&=~TMRA_1_SCTRL_TCF;
        if (--StartDelay == 0) 
        {
        	Timer_Disable();
            bootExit();	
        }
    }
}

/***************************************************************************
* Functions:   bootExit()
*
* Description: read the new appliaction start address
*              and start user's application

* Returns:     None
*
* Notes:
*
**************************************************************************/
#pragma optimization_level 0
void bootExit(void)
{
    int i=0;
    word securityWord = 0xFFBE;
    
   sci_tx(&StrStarted[0]);

   while(!(*((UInt16*)QSCI_STAT) & QSCI0_STAT_TIDLE)){}; 
  
   // Disable flash cache
   disable_flash_cache();
   
   // now check if it is secured by bootloader
   i = pmem_read(FTFL_Config_Security_Addr);
   if(i == -1)
   {
	   // secured, unsecure it
	   fprogram(FTFL_Config_Security_Addr,&securityWord,1);  
   }
   // re-enable flash cache
   enable_flash_cache();
#ifndef	DYNAMIC_BOOT
	// Disable all interrupts
	Cpu_DisableInt();
    
    // Disable timer
    Timer_Disable();

    // Restore Vector Base Address (VBA)
    INTC_VBA = (VBA_BASE_ADDRESS) >>(21-INTC_VBA_BITS);
    
    
  /****************************************************************************
  / read user start address
  /****************************************************************************/
    securityWord = pmem_read((unsigned int *)BOOT_START_ADDRESS_PLACE);
   i  = pmem_read((unsigned int *)BOOT_START_ADDRESS_PLACE+1);
   StartAddress = (((dword)i) << 16) | securityWord;

    asm(move.l StartAddress, N);
    asm(jmp (N));
#else
    
	// Issue a software reset to restart the chip
    SIM_CTRL =  SIM_CTRL_SWRst;
	while(1)
	{
		asm(nop);
	}
#endif    
}

/***************************************************************************
* Functions:   bootTimerInit()
*
* Description: Initialize timer to generate 1s.
*
* Returns:     None
*
* Notes:
*
**************************************************************************/
void bootTimerInit(void)
{   

	/* Enable timer clock */
	SIM_PCE0 |= SIM_PCE0_TA1 | SIM_PCE0_TA0 ;		
//	SIM_PCE0 |= SIM_PCE0_TA1_MASK | SIM_PCE0_TA0_MASK ;		

	/* Use 32-bit cascaded timer counter */
	
	TMRA_0_LOAD = 0;
	TMRA_0_CNTR = 0;
	TMRA_0_COMP1 = BUS_CLOCK_HZ/64/1000;		// generate 10ms
	                        
	
	TMRA_1_LOAD = 0;   
	TMRA_1_CNTR = 0;                          
	TMRA_1_COMP1 = 99;

	
	/* Set up the cascaded timer 1 & 0 */
		
	TMRA_1_CTRL = 0xE820; 	// enabled  timer 1 as cascaded counter    

	
	TMRA_0_CTRL = 0x3C20;     // enabled  timer 0      
                            // [12:09]  = 1110 Count Source IPbus clock divide by 64    
							// [8:7]    = 00                                             
							// [6]      = 0 ONCE Count repeatedly                       
							// [5]      = 1 LENGTH until compare                            
							// [4]      = 0 DIR Count up                                 
							// [3]      = 0                                              
							// [2:0]    = 000 OFLAG asserted          
								                   
}




#ifdef	DYNAMIC_BOOT
#pragma section pram_code end
#endif


/* END bootloader */
/*
** ###################################################################
**
**     This file was created by UNIS Processor Expert 2.98 [03.78]
**     for the Freescale 56800 series of microcontrollers.
**
** ###################################################################
*/
