#include "board.h"
#include <string.h>
#include "spifilib_api.h"
#include <cr_section_macros.h>
	/* Local memory, 32-bit aligned that will be used for driver context (handle) */
static uint32_t lmem[21];

#define PROGRAM_ERASE  1

#ifndef SPIFLASH_BASE_ADDRESS
#define SPIFLASH_BASE_ADDRESS (0x14000000)
#endif
/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/
#define TEST_BUFFSIZE (4096)
#define TICKRATE_HZ1 (1000)	/* 1000 ticks per second I.e 1 mSec / tick */
static uint32_t ledTickCount = 500;

#if PROGRAM_ERASE
/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/
//SubSector
#define ERASE_SUB_SECT    			0
#define SUB_SECT_OFFSET				0x10000
#define SUBSECTOR_SIZE       		(4*1024)
#define numSubBlocks				2
#define beginSubBlockAdrs				(SPIFLASH_BASE_ADDRESS + SUB_SECT_OFFSET)
#define beginSubBlock               16
//Block
#define SECT_64K_ERASE				1
#define BLOCK_SIZE					(64*1024)   //64KB
#define BLOCK_OFFSET				0x10000
#define NUMBER_OF_BLOCK             1
#define firstAddr					(SPIFLASH_BASE_ADDRESS + BLOCK_OFFSET)
#define lastAddr					(firstAddr+(NUMBER_OF_BLOCK)*(BLOCK_SIZE))
#define PROGRAM_64KB_SET			1
#if PROGRAM_64KB_SET
#define PROGRAM_BLOCK_SIZE       	(64*1024)   //64KB
#else
#define PROGRAM_BLOCK_SIZE			(256)
#endif

// Variables for the SPIFI handling
__BSS(RAM3) uint8_t data_buffer[PROGRAM_BLOCK_SIZE] ; // create a zero-init buffer in RAM2

// ****************
// Verify the spifi data
static int verify_spifi_data(const uint8_t *buff, int size)
{
	int i;
	for (i = 0; i < size; i++) {
		if (*buff++ == (uint8_t) i) break;
	}
	return i == size ? 0 : i;
}

// ****************
// Initializes the buffer from 00 to FF
static void prepare_write_data(uint8_t *buff, int size)
{
	int i;
	for (i = 0; i < size; i ++) {
		*buff++ = (uint8_t) i;
	}
}
#endif
/*****************************************************************************
 * Private functions
 ****************************************************************************/

STATIC const PINMUX_GRP_T spifipinmuxing[] = {
	{0x3, 3,  (SCU_PINIO_FAST | SCU_MODE_FUNC3)},	/* SPIFI CLK */
	{0x3, 4,  (SCU_PINIO_FAST | SCU_MODE_FUNC3)},	/* SPIFI D3 */
	{0x3, 5,  (SCU_PINIO_FAST | SCU_MODE_FUNC3)},	/* SPIFI D2 */
	{0x3, 6,  (SCU_PINIO_FAST | SCU_MODE_FUNC3)},	/* SPIFI D1 */
	{0x3, 7,  (SCU_PINIO_FAST | SCU_MODE_FUNC3)},	/* SPIFI D0 */
	{0x3, 8,  (SCU_PINIO_FAST | SCU_MODE_FUNC3)}	/* SPIFI CS/SSEL */
};

static void setLedBlinkRate(uint32_t newValue) 
{
	ledTickCount = newValue;
}
static uint32_t getLedBlinkRate(void) 
{
	return ledTickCount;
}

/* Displays error message and dead loops */
static void fatalError(SPIFI_ERR_T errNum)
{
		/* Loop forever */
	while (1) {
		__WFI();
	}
}

static SPIFI_HANDLE_T *initializeSpifi(void)
{
	int idx;
	int devIdx;
	uint32_t memSize;
	SPIFI_HANDLE_T *pReturnVal;

	/* Initialize LPCSPIFILIB library, reset the interface */
	spifiInit(LPC_SPIFI_BASE, true);

	/* register support for the family(s) we may want to work with
	     (only 1 is required) */
	spifiRegisterFamily(SPIFI_REG_FAMILY_SpansionS25FLP);
	spifiRegisterFamily(SPIFI_REG_FAMILY_SpansionS25FL1);
	spifiRegisterFamily(SPIFI_REG_FAMILY_MacronixMX25L);
	
	/* Return the number of families that are registered */
	idx = spifiGetSuppFamilyCount();

	/* Get required memory for detected device, this may vary per device family */
	memSize = spifiGetHandleMemSize(LPC_SPIFI_BASE);
	if (memSize == 0) {
		/* No device detected, error */
		fatalError(SPIFI_ERR_GEN);
	}

	/* Initialize and detect a device and get device context */
	/* NOTE: Since we don't have malloc enabled we are just supplying
	     a chunk of memory that we know is large enough. It would be
	     better to use malloc if it is available. */
	pReturnVal = spifiInitDevice(&lmem, sizeof(lmem), LPC_SPIFI_BASE, SPIFLASH_BASE_ADDRESS);
	if (pReturnVal == NULL) {
		fatalError(SPIFI_ERR_GEN);
	}
	return pReturnVal;
}


void spifiSetUp(void)
	{
	uint32_t idx;
	uint32_t spifiBaseClockRate;
	uint32_t maxSpifiClock;
	uint16_t libVersion;
	uint32_t pageAddress;

	uint32_t bytesRemaining;
	uint32_t deviceByteCount;

	SPIFI_HANDLE_T *pSpifi;
	SPIFI_ERR_T errCode;
	static uint32_t buffer[TEST_BUFFSIZE / sizeof(uint32_t)];
		
	/* Report the library version to start with */
	libVersion = spifiGetLibVersion();
	DEBUGOUT("\r\n\r\nSPIFI Lib Version %02d%02d\r\n", ((libVersion >> 8) & 0xff), (libVersion & 0xff));

	/* set the blink rate while testing */
	setLedBlinkRate(500);
	
	/* Setup SPIFI FLASH pin muxing (QUAD) */
	Chip_SCU_SetPinMuxing(spifipinmuxing, sizeof(spifipinmuxing) / sizeof(PINMUX_GRP_T));

	/* SPIFI base clock will be based on the main PLL rate and a divider */
	spifiBaseClockRate = Chip_Clock_GetClockInputHz(CLKIN_MAINPLL);

	/* Setup SPIFI clock to run around 1Mhz. Use divider E for this, as it allows
	   higher divider values up to 256 maximum) */
	Chip_Clock_SetDivider(CLK_IDIV_E, CLKIN_MAINPLL, ((spifiBaseClockRate / 1000000) + 1));
	Chip_Clock_SetBaseClock(CLK_BASE_SPIFI, CLKIN_IDIVE, true, false);
	DEBUGOUT("SPIFI clock rate %d\r\n", Chip_Clock_GetClockInputHz(CLKIN_IDIVE));
	//__disable_irq();
	/* Initialize the spifi library. This registers the device family and detects the part */
	pSpifi = initializeSpifi();

	/* Get some info needed for the application */
	maxSpifiClock = spifiDevGetInfo(pSpifi, SPIFI_INFO_MAXCLOCK);
	pageAddress = SPIFLASH_BASE_ADDRESS;
	deviceByteCount = spifiDevGetInfo(pSpifi, SPIFI_INFO_DEVSIZE);

	/* Enable quad.  If not supported it will be ignored */
	spifiDevSetOpts(pSpifi, SPIFI_OPT_USE_QUAD, true);

	/* Enter memMode */
	spifiDevSetMemMode(pSpifi, true);
	//__enable_irq();
	/* Get info */
	DEBUGOUT("Device family       = %s\r\n", spifiDevGetFamilyName(pSpifi));
	DEBUGOUT("Capabilities        = 0x%x\r\n", spifiDevGetInfo(pSpifi, SPIFI_INFO_CAPS));
	DEBUGOUT("Device size         = %d\r\n", spifiDevGetInfo(pSpifi, SPIFI_INFO_DEVSIZE));
	DEBUGOUT("Max Clock Rate      = %d\r\n", maxSpifiClock);
	DEBUGOUT("Erase blocks        = %d\r\n", spifiDevGetInfo(pSpifi, SPIFI_INFO_ERASE_BLOCKS));
	DEBUGOUT("Erase block size    = %d\r\n", spifiDevGetInfo(pSpifi, SPIFI_INFO_ERASE_BLOCKSIZE));
	DEBUGOUT("Erase sub-blocks    = %d\r\n", spifiDevGetInfo(pSpifi, SPIFI_INFO_ERASE_SUBBLOCKS));
	DEBUGOUT("Erase sub-blocksize = %d\r\n", spifiDevGetInfo(pSpifi, SPIFI_INFO_ERASE_SUBBLOCKSIZE));
	DEBUGOUT("Write page size     = %d\r\n", spifiDevGetInfo(pSpifi, SPIFI_INFO_PAGESIZE));
	DEBUGOUT("Max single readsize = %d\r\n", spifiDevGetInfo(pSpifi, SPIFI_INFO_MAXREADSIZE));
	//DEBUGOUT("Current dev status  = 0x%x\r\n", spifiDevGetInfo(pSpifi, SPIFI_INFO_STATUS));
	DEBUGOUT("Current options     = %d\r\n", spifiDevGetInfo(pSpifi, SPIFI_INFO_OPTIONS));

	DEBUGOUT("SPIFI final Rate    = %d\r\n", Chip_Clock_GetClockInputHz(CLKIN_IDIVE));
	DEBUGOUT("\r\n");

	Chip_Clock_SetDivider(CLK_IDIV_E, CLKIN_MAINPLL, ((spifiBaseClockRate / maxSpifiClock) + 1));
	SysTick_Config(SystemCoreClock / TICKRATE_HZ1);
#if PROGRAM_ERASE
	__disable_irq();
	/* Make sure we are not in mem mode and turn off QUAD for now */
	spifiDevSetOpts(pSpifi, SPIFI_OPT_USE_QUAD, false);
	spifiDevSetMemMode(pSpifi, false);
	/* UnLock device */
	errCode = spifiDevUnlockDevice(pSpifi);
	if (errCode != SPIFI_ERR_NONE) {
		 		while(1);
		 	}
	 prepare_write_data(data_buffer, sizeof(data_buffer));
	 /* UnLock device */
	 	errCode = spifiDevUnlockDevice(pSpifi);
	 	if (errCode != SPIFI_ERR_NONE) {
	 		while(1);
	 	}
#if ERASE_SUB_SECT
	 	//beginSubBlock = spifiGetSubBlockFromAddr(pSpifi,beginSubBlockAdrs);
	/* Erase using sub-blocks */
		for (idx = 0; idx < numSubBlocks; ++idx) {
			errCode = spifiDevEraseSubBlock(pSpifi, (idx + beginSubBlock));
			if (errCode != SPIFI_ERR_NONE) {
				while(1);
			}
		}
#endif
		errCode = spifiEraseByAddr(pSpifi, firstAddr, lastAddr);
			if (errCode != SPIFI_ERR_NONE) {
				while(1);
			}
			errCode=spifiProgram(pSpifi,firstAddr,(char *)data_buffer,PROGRAM_BLOCK_SIZE);
			if (errCode != SPIFI_ERR_NONE) {
							while(1);
						}

  spifiDevSetMemMode(pSpifi, true);
__enable_irq();
#endif
}
	
/*****************************************************************************
 * Public functions
 ****************************************************************************/

uint32_t LEDOn, LEDOff; 

/*-----------------------------------------------------------------------------
  SysTick IRQ Handler
 *----------------------------------------------------------------------------*/
void SysTick_Handler (void) {
  static uint32_t ticks;
  
  switch (ticks++) {
    case  0: LEDOn  = 1; break;
    case  50: LEDOff = 1; break;
    case  99: ticks  = 0; break;
    
    default:
      if (ticks > 100) {
        ticks = 0;
      }
  }
}
