/** @file wlmgr_cli.c
  *
  * @brief This file contains WLAN application specific defines etc.
  *
  * Copyright 2019-2020 NXP
  *
  * NXP CONFIDENTIAL
  * The source code contained or described herein and all documents related to
  * the source code ("Materials") are owned by NXP, its
  * suppliers and/or its licensors. Title to the Materials remains with NXP,
  * its suppliers and/or its licensors. The Materials contain
  * trade secrets and proprietary and confidential information of NXP, its
  * suppliers and/or its licensors. The Materials are protected by worldwide copyright
  * and trade secret laws and treaty provisions. No part of the Materials may be
  * used, copied, reproduced, modified, published, uploaded, posted,
  * transmitted, distributed, or disclosed in any way without NXP's prior
  * express written permission.
  *
  * No license under any patent, copyright, trade secret or other intellectual
  * property right is granted to or conferred upon you by disclosure or delivery
  * of the Materials, either expressly, by implication, inducement, estoppel or
  * otherwise. Any license under such intellectual property rights must be
  * express and approved by NXP in writing.
  *
  */

#include <stdio.h>
#include <ctype.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <malloc.h>
#include <dirent.h>
#include "ctrl_client.h"

#define VERSION_STR    "v1.0"
#define CTRL_FILE_DEF  "wdev0ap0"
#define CTRL_IFACE_DIR "/var/run/wlmgr"

#define CMD_BUF_LEN 4096
static char cmdbuf[CMD_BUF_LEN];
static int cmdbuf_pos = 0;

static const char *wlmgr_cli_version = "wlmgr_cli version: " VERSION_STR "\n";
static char *ctrl_ifname = NULL;
static int ctrl_sock = -1;

static void wlmgr_cli_msg_cb(char *msg, size_t len)
{
    printf("%s\n", msg);
}

static int _wlmgr_cli_send_cmd(char *cmd, int print)
{
    char replyBuf[CMD_BUF_LEN];
    size_t replyLen;
    int ret;

    if (ctrl_sock == -1) {
        printf("Not connected to wlmgr App - command dropped.\n");
        return -1;
    }

    memset(replyBuf, 0, sizeof(replyBuf));
    replyLen = sizeof(replyBuf) - 1;
    ret = ctrl_client_request(ctrl_sock, cmd, strlen(cmd), replyBuf, &replyLen);
    if (ret < 0) {
        printf("'%s' command failed.\n", cmd);
        return -1;
    }

    if (print) {
        replyBuf[replyLen] = '\0';
        printf("%s", replyBuf);
    }

    return 0;
}

static int wlmgr_cli_send_cmd(char *cmd)
{
    return _wlmgr_cli_send_cmd(cmd, 1);
}

static int wlmgr_cli_cmd_get(int argc, char *argv[])
{
    char cmd[CMD_BUF_LEN];
    int ret;

    if (argc != 1) {
        printf("Invalid GET command: needs one argument\n");
	return -1;
    }

    memset(cmd, 0, CMD_BUF_LEN);
    ret = snprintf(cmd, sizeof(cmd), "GET %s", argv[0]);
    if (ret < 0 || (size_t)ret >= sizeof(cmd) - 1) {
        printf("Too long GET command.\n");
        return -1;
    }
    return wlmgr_cli_send_cmd(cmd);
}

static int wlmgr_cli_cmd_set(int argc, char *argv[])
{
    char cmd[CMD_BUF_LEN];
    int ret;

    if (argc != 2) {
        printf("Invalid SET command: needs two arguments (variable "
               "name and value)\n");
        return -1;
    }

    memset(cmd, 0, CMD_BUF_LEN);
    ret = snprintf(cmd, sizeof(cmd), "SET %s %s", argv[0], argv[1]);
    if (ret < 0 || (size_t)ret >= sizeof(cmd) - 1) {
        printf("Too long SET command.\n");
        return -1;
    }
    return wlmgr_cli_send_cmd(cmd);
}

struct wlmgr_cli_cmd {
    const char *cmd;
    int (*handler)(int argc, char *argv[]);
    const char *usage;
};

static struct wlmgr_cli_cmd wlmgr_cli_commands[] = {
    { "get", wlmgr_cli_cmd_get,
      "= Get information for wlmgr Application\n"
      "      version               Show the version of wlmgr and wlmgr_cli.\n"
      "      config                List all configuration used by wlmgr.\n"
      "      debug                 Show the debug level for wlmgr.\n"
      "      mumimo-version        Show the version of Advanced MU-MIMO Grouping Algorithm.\n"
      "      mumimo-config         List configuration used by MU-MIMO Grouping.\n"
      "      mumimo-enable_mode    Show if 11AX or 11AC or both enabled for MU-MIMO Grouping.\n"
      "      mumimo-corrthresh     List the correlation thresholds for MU-MIMO Grouping.\n"
      "      mumimo-tonefactor     Show the tone factor used by MU-MIMO correlation calculation.\n"
      "      mumode-version        Show the version of MU Mode Selection Algorithm.\n"
      "      mumode-config         List configuration used by MU Mode Selection.\n"
      "      mumode-max_nss        Get maximum NSS in one group.\n"
      "      mumode-near_rssi_thresh    Get rssi threshold for near range.\n"
      "      mumode-far_rssi_thresh     Get rssi threshold for far range.\n"
    },
    { "set", wlmgr_cli_cmd_set,
      "= Set configurations for wlmgr Application\n"
      "      debug                 Set the debug level for wlmgr.\n"
      "      mumimo-enable         Set the enable mode for MU-MIMO Grouping.\n"
      "      mumimo-enable_mode    Set the enable mode MU-MIMO 11AX or 11AC or both.\n"
      "                            \"possible value could be <11ax>, <11ac>, <11axac> or <none>\".\n"
      "      mumimo-interval       Set the interval for MU-MIMO Grouping to check SM data.\n"
      "      mumimo-corrthresh     Set the correlation thresholds for MU-MIMO Grouping.\n"
      "                            \"nss=<n> near=<corrthresh> mid=<corrthresh> far=<corrthresh>\".\n"
      "      mumimo-tonefactor     Set the tone factor used by MU-MIMO correlation calculation.\n"
      "      mumode-interval       Set the interval for MU Mode Selection\n"
      "      mumode-dl_thresh      Set Down Link threshold bytes\n"
      "      mumode-ul_thresh      Set Up Link threshold bytes\n"
      "      mumode-sched_mode     Set sched_mode to \"bsrp_on\" or \"bsrp_off\" \n"
      "      mumode-max_nss        Set maximum NSS in one group.\n"
      "      mumode-near_rssi_thresh    Set rssi threshold for near range.\n"
      "      mumode-far_rssi_thresh     Set rssi threshold for far range.\n"
    },
    { NULL, NULL, NULL }
};

static void wlmgr_cli_process_cmd(int argc, char *argv[])
{
    struct wlmgr_cli_cmd *cmd = NULL, *match = NULL;

    cmd = wlmgr_cli_commands;
    while (cmd->cmd) {
        if (strcasecmp(cmd->cmd, argv[0]) == 0) {
            match = cmd;
            break;
        }
	cmd++;
    }

    if (match == NULL) {
        printf("Unknown command '%s'\n", cmd->cmd);
    } else {
        match->handler(argc - 1, &argv[1]);
    }
}

#define max_args 10

static void wlmgr_cli_parse_cmd(char *cmd)
{
    char *pos, *pos2;
    char *argv[max_args];
    int argc = 0;

    pos = cmd;
    for (;;) {
        while (*pos == ' ')
            pos++;
        if (*pos == '\0')
            break;

        if (*pos == '"')
            argv[argc] = pos + 1; //skip '"'
        else
            argv[argc] = pos;
        argc++;
        if (argc == max_args)
            break;

        if (*pos == '"') {
            pos2 = strrchr(pos, '"');
            if (pos2)
                pos = pos2 + 1;
        }

        while (*pos != '\0' && *pos != ' ')
            pos++;

        if ((*pos == ' ') || (*pos == '"')) //skip '"'
            *pos++ = '\0';
    }

    if (argc)
        wlmgr_cli_process_cmd(argc, argv);
}

static void wlmgr_cli_print_cmd_help(const struct wlmgr_cli_cmd *cmd)
{
    size_t n;

    if (cmd->usage == NULL)
        return;

    printf("    %s %s\n", cmd->cmd, cmd->usage);
}

static void wlmgr_cli_print_help(const char *cmd)
{
    int i;

    printf("commands:\n");
    for (i = 0; wlmgr_cli_commands[i].cmd; i++) {
        if (cmd == NULL || strcasecmp(wlmgr_cli_commands[i].cmd, cmd))
            wlmgr_cli_print_cmd_help(&wlmgr_cli_commands[i]);
    }
}

static void wlmgr_cli_usage(void)
{
    printf("%s", wlmgr_cli_version);
    printf("\n"
            "usage: wlmgr_cli [-i <wlanif>][-hv] [commands...]"
            "\n"
            "where:\n"
            "    -i <wlanif> set wireless network interface\n"
            "    -h          help (show this usage text)\n"
            "    -v          shown version information\n"
          );
    wlmgr_cli_print_help(NULL);
}

int main(int argc, char *argv[])
{
    int c;

    for (;;) {
        c = getopt(argc, argv, "i:vh");
        if (c < 0)
            break;
        switch (c) {
        case 'i':
            if (ctrl_ifname != NULL)
                free(ctrl_ifname);
            ctrl_ifname = strdup(optarg);
            break;
        case 'v':
            printf("%s\n", wlmgr_cli_version);
            return 0;
        case 'h':
            wlmgr_cli_usage();
            return 0;
        default:
            wlmgr_cli_usage();
            return 0;
        }
    }

    if (argc == optind) {
        wlmgr_cli_usage();
        return 0;
    }

    for (;;) {
        ctrl_sock = ctrl_client_init(CTRL_IFACE_DIR, ctrl_ifname);
        if (ctrl_sock >= 0) {
            //printf("Connect to wlmgr \"%s/%s\" Successfully!\n",
            //       CTRL_IFACE_DIR, ctrl_ifname);
            break;
        }

        printf("Could not connect to wlmgr Application - re-trying\n");
        usleep(50000);
    }

    wlmgr_cli_process_cmd(argc - optind, &argv[optind]);

    if (ctrl_sock >= 0) {
        ctrl_client_deinit(ctrl_sock);
        ctrl_sock = -1;
    }

    return 0;
}
