/** @file netlink.c
  *
  * @brief This file contains WLAN application specific defines etc.
  *
  * Copyright 2019-2020 NXP
  *
  * NXP CONFIDENTIAL
  * The source code contained or described herein and all documents related to
  * the source code ("Materials") are owned by NXP, its
  * suppliers and/or its licensors. Title to the Materials remains with NXP,
  * its suppliers and/or its licensors. The Materials contain
  * trade secrets and proprietary and confidential information of NXP, its
  * suppliers and/or its licensors. The Materials are protected by worldwide copyright
  * and trade secret laws and treaty provisions. No part of the Materials may be
  * used, copied, reproduced, modified, published, uploaded, posted,
  * transmitted, distributed, or disclosed in any way without NXP's prior
  * express written permission.
  *
  * No license under any patent, copyright, trade secret or other intellectual
  * property right is granted to or conferred upon you by disclosure or delivery
  * of the Materials, either expressly, by implication, inducement, estoppel or
  * otherwise. Any license under such intellectual property rights must be
  * express and approved by NXP in writing.
  *
  */

#include "netlink.h"
#include "pthread.h"
#include "wlmgr_debug.h"

static void netlink_recv_process(struct netlink_data *netlink,
				 void (*cb)(void *ctx, struct ifinfomsg *ifi,
					    u8 *buf, size_t len),
				 struct nlmsghdr *hdr)
{
    if (cb == NULL || NLMSG_PAYLOAD(hdr, 0) < sizeof(struct ifinfomsg))
        return;

    cb(netlink->callback_ctx, NLMSG_DATA(hdr),
       (u8 *) NLMSG_DATA(hdr) + NLMSG_ALIGN(sizeof(struct ifinfomsg)),
       NLMSG_PAYLOAD(hdr, sizeof(struct ifinfomsg)));
}

static void netlink_recv_handler(void *priv)
{
    struct netlink_data *netlink = (struct netlink_data *)priv;
    char recv_buf[8192];
    int recv_len;
    struct sockaddr_nl from;
    socklen_t from_len;
    struct nlmsghdr *hdr;

    while (1) {
        from_len = sizeof(from);
        recv_len = recvfrom(netlink->sock, recv_buf, sizeof(recv_buf), MSG_DONTWAIT,
			(struct sockaddr *) &from, &from_len);
        if (recv_len < 0) {
            if (errno != EINTR && errno != EAGAIN) {
                wlmgr_printf(WLMGR_ERROR, "netlink: recvfrom failed: %s", strerror(errno));
                return;
            } else {
                sleep(1);
                continue;
            }
        }

        hdr = (struct nlmsghdr *)recv_buf;
        while (NLMSG_OK(hdr, recv_len)) {
            switch (hdr->nlmsg_type) {
            case RTM_NEWLINK:
                netlink_recv_process(netlink, netlink->newlink_cb, hdr);
                break;
            case RTM_DELLINK:
                netlink_recv_process(netlink, netlink->dellink_cb, hdr);
                break;
            }

            hdr = NLMSG_NEXT(hdr, recv_len);
        }

        if (recv_len > 0) {
            wlmgr_printf(WLMGR_INFO, "netlink: %d extra bytes in the end of netlink message", recv_len);
        }
    }
}

void netlink_deinit(struct netlink_data *netlink)
{
    if (netlink == NULL)
        return;

    if (netlink->sock >= 0) {
        //eloop_unregister_read_sock(netlink->sock);
        close(netlink->sock);
        netlink->sock = -1;
    }
    wlmgr_free(netlink);
    netlink = NULL;
}

struct netlink_data *netlink_init(
                          void (*newlink_cb)(void *ctx, struct ifinfomsg *ifi,
			                     u8 *buf, size_t len),
			  void (*dellink_cb)(void *ctx, struct ifinfomsg *ifi,
			                     u8 *buf, size_t len),
                          void *callback_ctx)
{
    struct netlink_data *netlink;
    struct sockaddr_nl nl;
    pthread_t t_netlink;

    netlink = wlmgr_malloc(sizeof(struct netlink_data));
    if (netlink == NULL)
        return NULL;

    netlink->newlink_cb = newlink_cb;
    netlink->dellink_cb = dellink_cb;
    netlink->callback_ctx = callback_ctx;

    netlink->sock = socket(PF_NETLINK, SOCK_RAW, NETLINK_ROUTE);
    if (netlink->sock < 0) {
        wlmgr_printf(WLMGR_ERROR, "%s: socket(PF_NETLINK) failed: %s", __func__, strerror(errno));
        netlink_deinit(netlink);
        return NULL;
    }

    memset(&nl, 0, sizeof(struct sockaddr_nl));
    nl.nl_family = PF_NETLINK;
    nl.nl_groups = RTMGRP_LINK;
    if (bind(netlink->sock, (struct sockaddr *)&nl, sizeof(struct sockaddr_nl)) < 0) {
        wlmgr_printf(WLMGR_ERROR, "%s: bind[PF_NETLINK] failed: %s", __func__, strerror(errno));
        netlink_deinit(netlink);
        return NULL;
    }

    if (pthread_create(&t_netlink, NULL, (void *)netlink_recv_handler, (void *)netlink) != 0) {
        wlmgr_printf(WLMGR_ERROR, "%s: pthread_create failed", __func__);
        return NULL;
    }

    return netlink;
}
