/** @file driver.h
  *
  * @brief This file contains WLAN application specific defines etc.
  *
  * Copyright 2019-2020 NXP
  *
  * NXP CONFIDENTIAL
  * The source code contained or described herein and all documents related to
  * the source code ("Materials") are owned by NXP, its
  * suppliers and/or its licensors. Title to the Materials remains with NXP,
  * its suppliers and/or its licensors. The Materials contain
  * trade secrets and proprietary and confidential information of NXP, its
  * suppliers and/or its licensors. The Materials are protected by worldwide copyright
  * and trade secret laws and treaty provisions. No part of the Materials may be
  * used, copied, reproduced, modified, published, uploaded, posted,
  * transmitted, distributed, or disclosed in any way without NXP's prior
  * express written permission.
  *
  * No license under any patent, copyright, trade secret or other intellectual
  * property right is granted to or conferred upon you by disclosure or delivery
  * of the Materials, either expressly, by implication, inducement, estoppel or
  * otherwise. Any license under such intellectual property rights must be
  * express and approved by NXP in writing.
  *
  */


#ifndef DRIVER_H
#define DRIVER_H

#include "wlmgr.h"
#include "utils.h"

/* mu_version >= 1, support data_len param for ul_muset */
#define DRV_MU_VERSION_1        1

/* Reason codes (IEEE 802.11-2007, 7.3.1.7, Table 7-22) */
#define WLAN_REASON_UNSPECIFIED 1
#define WLAN_REASON_PREV_AUTH_NOT_VALID 2
#define WLAN_REASON_DEAUTH_LEAVING 3
#define WLAN_REASON_DISASSOC_DUE_TO_INACTIVITY 4
#define WLAN_REASON_DISASSOC_AP_BUSY 5
#define WLAN_REASON_CLASS2_FRAME_FROM_NONAUTH_STA 6
#define WLAN_REASON_CLASS3_FRAME_FROM_NONASSOC_STA 7
#define WLAN_REASON_DISASSOC_STA_HAS_LEFT 8
#define WLAN_REASON_STA_REQ_ASSOC_WITHOUT_AUTH 9

/**
 * enum DRV_DEV_ID - Device ID
 */
typedef enum
{
    DRV_DEV_ID_SC5,
    DRV_DEV_ID_SCBT,
    DRV_DEV_ID_FAIL = -1
} DRV_DEV_ID;

/**
 * enum DRV_BW_TYPE - BandWidth types
 */
typedef enum
{
    DRV_BW_TYPE_AUTO,
    DRV_BW_TYPE_10MHZ,
    DRV_BW_TYPE_20MHZ,
    DRV_BW_TYPE_40MHZ,
    DRV_BW_TYPE_80MHZ,
    DRV_BW_TYPE_160MHZ,
    DRV_BW_TYPE_FAIL = -1
} DRV_BW_TYPE;

struct driver_version {
    u8 major;
    u8 minor;
    u8 rel;
    u8 patch;
    u8 pfwBuild;
    u8 drvBuild;
    u16 pfw;
    u16 drv;
};

struct driver_data {
    void *data; /* back pointer */
    u8 iface[IFNAMSIZ + 1];
    int ifindex;
    struct netlink_data *netlink;
    int ioctl_sock;
    int	we_version;
    u8 wdev_hwaddr[ETH_ADDR_LEN];
    struct driver_version ver;
};

/**
 * struct driver_ops - Driver interface API definition
 *
 * This structure defines the API that each driver interface needs to implement
 * for wlmgr. All driver specific functionality is captured
 * in this wrapper.
 */
struct driver_ops {
    /** Name of the driver interface */
    const char *name;
    /** One line description of the driver interface */
    const char *desc;
    /** hapd_init - Initialize driver interfac */
    void * (*init)(void *data, const char *iface);
    /**
     * deinit - Deinitialize driver interface
     * @priv: private driver interface data from init()
     */
    void (*deinit)(void *priv);
    /**
     * get_bssid - Get the current BSSID
     * @priv: private driver interface data
     * @bssid: buffer for BSSID (ETH_ALEN = 6 bytes)
     *
     * Returns: Length of the current BSSID on success,
     *           -1 on failure
     */
    int (*get_bssid)(void *priv, u8 *buf);
    /**
     * get_essid - Get the current SSID (AP only)
     * @priv: Private driver interface data
     * @buf: Buffer for returning the SSID
     * @len: Maximum length of the buffer
     * Returns: Length of the SSID on success, -1 on failure
     */
    int (*get_essid)(void *priv, u8 *buf, int len);
    /**
     * get_curchan - Get the current channel
     * @priv: private driver interface data
     * @buf: buffer for WPA/WPA2 mode
     *
     * Returns: 0 on success, -1 on failure
     */
    int (*get_curchan)(void *priv, u8 *buf);
    /**
     * get_chanutil - Get the current channel Utility
     * @priv: private driver interface data
     *
     * Returns: channel Utility on success,
     *           0 on failure
     */
    unsigned int (*get_chanutil)(void *priv);
    /**
     * get_agingtime - Get the current channel Utility
     * @priv: private driver interface data
     *
     * Returns: STA agingtime on success,
     *           0 on failure
     */
    unsigned int (*get_agingtime)(void *priv);
    /**
     * get_deviceid - Get the Device ID
     * @priv: private driver interface data
     *
     * Returns: DRV_DEV_ID
     */
    DRV_DEV_ID (*get_deviceid)(void *priv);
    /**
     * get_bandwidth - Get the BandWidth
     * @priv: private driver interface data
     *
     * Returns: DRV_BW_TYPE
     */
    DRV_BW_TYPE (*get_bandwidth)(void *priv);
    /**
     * get_stalist - Get the STA list
     * @priv: private driver interface data
     * @buf: buffer for STA list
     *
     * Returns: Length of the STA list on success,
     *           -1 on failure
     */
    unsigned int (*get_stalist)(void *priv, u8 *buf);
    /**
     * get_stacnt - Get the count of current stations connected
     * @priv: private driver interface data
     *
     * Returns: count of STAs on success,
     *           0 on failure
     */
    unsigned int (*get_stacnt)(void *priv);
    /**
     * get_maxsta - Get the maximum number of connected STA
     * @priv: private driver interface data
     *
     * Returns: maximum number of STAs on success,
     *           0 on failure
     */
    unsigned short (*get_maxsta)(void *priv);
    /**
     * get_stabtm - Get BTM capability of station
     * @addr: MAC address of the station
     * @priv: private driver interface data
     *
     * Returns: BTM capability of station,
     *           -1 on failure
     */
    unsigned int (*get_stabtm)(void *priv, const u8 *addr);
    /**
     * set_btm - Enable/disable BTM capability
     * @priv: private driver interface data
     * @enable: 1 = BTM enabled, 0 = disabled
     *
     * Returns: 0 on success, -1 on failure
     *
     */
    int (*set_btm)(void *priv, const u8 enable);
    /**
     * send_disassoc - Disassociate a station (AP only)
     * @priv: Private driver interface data
     * @addr: MAC address of the station to disassociate
     * @reason: Reason code for the Disassociation frame
     * Returns: 0 on success, -1 on failure
     *
     * This function requests a specific station to be disassociated and
     * a Disassociation frame to be sent to it.
     */
    int (*send_disassoc)(void *priv, const u8 *addr, int reason);
    /**
     * send_deauth - Deauthenticate a station (AP only)
     * @priv: Private driver interface data
     * @addr: MAC address of the station to deauthenticate
     * @reason: Reason code for the Deauthentiation frame
     * Returns: 0 on success, -1 on failure
     *
     * This function requests a specific station to be deauthenticated and
     * a Deauthentication frame to be sent to it.
     */
    int (*send_deauth)(void *priv, const u8 *addr, int reason);
    /**
     * config_btm_target - config BTM target for steering
     * @priv: Private driver interface data
     * @target: MAC address of target for steering
     * @channel: target chanel
     * Returns: 0 on success, -1 on failure
     *
     * This function requests a specific station to be deauthenticated and
     * a Deauthentication frame to be sent to it.
     */
    int (*config_btm_target)(void *priv, const u8 *target, const u8 channel);
    /**
     * send_wtsbtmreq - send BTM request to guide the STA to 5G/2.4G band
     * @priv: Private driver interface data
     * @addr: MAC address of the station to deauthenticate
     * @deauth_interval: count of beacon interval
     * Returns: 0 on success, -1 on failure
     *
     * This function requests a specific station to be deauthenticated and
     * a Deauthentication frame to be sent to it.
     */
    int (*send_wtsbtmreq)(void *priv, const u8 *addr, const u8 *target);
    /**
     * send_btmreq - send BTM request to guide the STA to a different AP (AP only)
     * @priv: Private driver interface data
     * @addr: MAC address of the station to deauthenticate
     * Returns: 0 on success, -1 on failure
     *
     * This function requests a specific station to be steer to a different AP.
     */
    int (*send_btmreq)(void *priv, const u8 *addr);
    /**
     * set_apsteermode - Enable/disable AP Steering
     * @priv: private driver interface data
     * @enable: 1 = AP Steering enabled, 0 = disabled
     *
     * Returns: 0 on success, -1 on failure
     *
     */
    int (*set_apsteermode)(void *priv, u8 enable);
    /**
     * get_aprssi - Get RSSI of each neighbor AP
     * @priv: private driver interface data
     * @buf: buffer for neighbor AP list
     *
     * Returns: Length of the neighbor AP list on success,
     *           -1 on failure
     */
    int (*get_aprssi)(void *priv, u8 *buf);
    /**
     * set_mumimo - Enable MU MIMO
     * @priv: private driver interface data
     * @enable: 1 = SM data could be obtain
     *
     * Returns: 0 on success, -1 on failure
     *
     */
    int (*set_mumimo)(void *priv, u8 enable);
    /**
     * set_acnt_enable - enable acnt record
     * @priv: private driver interface data
     * @action: HostCmd_ACT_GEN_SET is enable acnt with mu, HostCmd_ACT_GEN_DEL is disable
     *
     * Returns: 0 on success, -1 on failure
     *
     */
    int (*set_acnt_enable)(void *priv, u16 action);
    /**
     * set_mumimo - Set MUSet
     * @priv: private driver interface data
     * @sta_cnt: MUSet sta number
     * @option: 0 is SU mode, 1 is VHT MU group, 2 is HE MU group, 3 is OFDMA
     * @gid: group id
     * @StnId: sta list
     *
     * Returns: 0 on success, -1 on failure
     *
     */
    int (*set_muset)(void *priv, u16 sta_cnt, u8 option, u8 gid, u16 *StnId);

    /**
     * set_ofdma_config - enable ofdma config
     * @priv: private driver interface data
     * @enable: 1 = acnt record enabled, 0 = disabled
     * @sta_count: sta number
     *
     * Returns: 0 on success, -1 on failure
     *
     */
    int (*set_ofdma_config)(void *priv, u8 enable, u8 sta_count);

    /**
     * set_ofdma_config - enable ofdma config
     * @priv: private driver interface data
     * @enable: 1 = set sta to ofdma, 0 = set sta to su
     * @sta_count: sta number
     * @StnId: sta list
     *
     * Returns: 0 on success, -1 on failure
     *
     */
    int (*set_ofdma)(void *priv, u8 enable, u8 sta_cnt, u16 *StnId);

    /**
     * set_ul_muset - Set UL MUSet
     * @priv: private driver interface data
     * @action: HostCmd_ACT_GEN_SET is create group, HostCmd_ACT_GEN_DEL is delete group
     * @rateinfo: rate info
     * @flag: [0]: Reinitialize SU Rate, 1: StaList changed, Oterhs: reserved
     * @gid: group id
     * @mode: TXINFO_TF_TYPE_MUMIMO, TXINFO_TF_TYPE_OFDMA
     * @bw: BW_20MHz = 0 ... BW_160MHz = 3
     * @sta_cnt: MUSet sta number
     * @param_cnt: paramer count
     * @param: parameter
     * @mu_version: Driver mu version
     *
     * Returns: 0 on success, -1 on failure
     *
     */
    int (*set_ul_muset)(void *priv, u8 action, u32 rateinfo, u32 flag, u8 gid, u8 mode, u8 bw, u16 sta_cnt, u32 param_cnt, u32 *param, u32 mu_version);

    /**
     * get_dl_gid - Get Down Link group id from AP.
     * @priv: private driver interface data
     *
     * Returns: gid
     *           -1 on failure
     */
    u8 (*get_dl_gid)(void *priv);

    /**
     * get_ul_gid - Get Up Link group id from AP.
     * @priv: private driver interface data
     *
     * Returns: gid
     *           -1 on failure
     */
    u8 (*get_ul_gid)(void *priv);

    /**
     * set_dl_gid - Delete Down Link group id from AP.
     * @priv: private driver interface data
     *
     * Returns: gid
     *           -1 on failure
     */
    s32 (*set_dl_gid)(void *priv, u8 gid);

    /**
     * set_ul_gid - Delete Up Link group id from AP.
     * @priv: private driver interface data
     *
     * Returns: gid
     *           -1 on failure
     */
    s32 (*set_ul_gid)(void *priv, u8 gid);

    /**
     * get_dl_ofdma - Get Down Link OFDMA enable/disable/force from AP.
     * @priv: private driver interface data
     *
     * Returns: action
     *          -1 on failure
     *          0 is disable
     *          1 is auto mode
     *          2 is force group
     */
    u8 (*get_dl_ofdma)(void *priv);

    /**
     * get_dl_mimo - Get Down Link MIMO enable/disable/force from AP.
     * @priv: private driver interface data
     *
     * Returns: action
     *          -1 on failure
     *          0 is disable
     *          1 is auto mode
     *          2 is force group
     */
    u8 (*get_dl_mimo)(void *priv);

    /**
     * get_ul_ofdma - Get Up Link OFDMA enable/disable/force from AP.
     * @priv: private driver interface data
     *
     * Returns: action
     *          -1 on failure
     *          0 is disable
     *          1 is auto mode
     *          2 is force group
     */
    u8 (*get_ul_ofdma)(void *priv);

    /**
     * driver_get_ul_mimo - Get Up Link MIMO enable/disable/force from AP.
     * @priv: private driver interface data
     *
     * Returns: action
     *          -1 on failure
     *          0 is disable
     *          1 is auto mode
     *          2 is force group
     */
    u8 (*get_ul_mimo)(void *priv);
    /**
     * set_muedca - Enable/disable mu_edca
     * @priv: private driver interface data
     *
     * Returns: gid
     *           -1 on failure
     */
    s32 (*set_muedca)(void *priv, u8 on_off);
    /**
     * set_fw_mib - Set fw_mib
     * @priv: private driver interface data
     *
     * Returns: gid
     *           -1 on failure
     */
    s32 (*set_fw_mib)(void *priv, char *mib_name, u8 valcnt, u32 *val);
    /**
     * compare_drv_ver - compare target version with current driver version
     * @priv: private driver interface data
     * @version: target version, used to compare with current driver version
     *
     * Returns: 1 larger than 0 less than
     *
     */
    u8 (*compare_drv_ver)(void *priv, u16 version);
    /**
     * set_sched_mode - Set UL trigger frame to BSRP or not.
     * @priv: private driver interface data
     * @mode: string of "bsrp"
     * @on_off: string of "on" or "off"
     *
     * Returns: 0 on success, -1 on failure
     *
     */
    s32 (*set_sched_mode)(void *priv, u8 *mode, u8 *on_off);
    /**
     * set_dl_expiretime - Set UL expire time
     * @priv: private driver interface data
     * @expiretime: us
     *
     * Returns: 0 on success, -1 on failure
     *
     */
    s32 (*set_dl_expiretime)(void *priv, u32 expiretime);
    /**
     * set_sta_aggregation - set sta ampdu & amsdu
     * @priv: private driver interface data
     * @enable: 0 is disable, 1 is enable, 2 is restore
     * @StnId: Station ID
     *
     * Returns: 0 on success, -1 on failure
     *
     */
    s32 (*set_sta_aggregation)(void *priv, u8 enable, u16 StnId);
};

#endif /* DRIVER_H */
