/** @file ctrl_client.c
  *
  * @brief This file contains WLAN application specific defines etc.
  *
  * Copyright 2019-2020 NXP
  *
  * NXP CONFIDENTIAL
  * The source code contained or described herein and all documents related to
  * the source code ("Materials") are owned by NXP, its
  * suppliers and/or its licensors. Title to the Materials remains with NXP,
  * its suppliers and/or its licensors. The Materials contain
  * trade secrets and proprietary and confidential information of NXP, its
  * suppliers and/or its licensors. The Materials are protected by worldwide copyright
  * and trade secret laws and treaty provisions. No part of the Materials may be
  * used, copied, reproduced, modified, published, uploaded, posted,
  * transmitted, distributed, or disclosed in any way without NXP's prior
  * express written permission.
  *
  * No license under any patent, copyright, trade secret or other intellectual
  * property right is granted to or conferred upon you by disclosure or delivery
  * of the Materials, either expressly, by implication, inducement, estoppel or
  * otherwise. Any license under such intellectual property rights must be
  * express and approved by NXP in writing.
  *
  */

#include <malloc.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
//#include <pthread.h>
#include "ctrl_client.h"

//#define CTRL_CLIENT_DEBUG 1

int ctrl_client_request(const int sock, const char *cmd, size_t cmd_len,
                        char *reply, size_t *reply_len)
{
    struct timeval timeout;
    fd_set fds;
    int retry = 0;
    int ret = 0;

    if (sock < 0)
        return -1;

    do {
        ret = send(sock, cmd, cmd_len, 0);
        if (ret < 0 ) {
            printf("%s: send sock failed (ret=%d) try to re-send\n", __func__, ret);
            if ((errno == EAGAIN || errno == EBUSY || errno == EWOULDBLOCK)) {
                usleep(50000);
                retry++;
            } else
                return ret;
        }
    } while ((ret < 0) && (retry < 3));

    /** wait for reply, if send sock sucessfully and caller needs the reply */
    if ((ret > 0) && (reply != 0)) {
        for (;;) {
            timeout.tv_sec = 10;
            timeout.tv_usec = 0;
            FD_ZERO(&fds);
            FD_SET(sock, &fds);
            ret = select(sock + 1, &fds, NULL, NULL, &timeout);
            if (ret < 0)
                return ret;
            if (FD_ISSET(sock, &fds)) {
                ret = recv(sock, reply, *reply_len, 0);
                if (ret < 0)
                    return ret;
                *reply_len = ret;
                reply[*reply_len] = '\0';
                break;
            } else
                return -2;
	}
    }

    return 0;
}

int ctrl_client_init(const char *ctrl_interface, const char *ctrl_file)
{
    struct sockaddr_un local;
    struct sockaddr_un dest;
    char *ctrl_path;
    size_t len;
    int retry = 0;
    int ret = 0;
    int flags;
    int sock = -1;

    if ((ctrl_interface == NULL) || (ctrl_file == NULL) ||
        (strlen(ctrl_interface) == 0) || (strlen(ctrl_file) == 0))
        return -1;

    sock = socket(PF_UNIX, SOCK_DGRAM, 0);
    if (sock < 0) {
        printf("%s: socket(PF_UNIX) failed: %s", __func__, strerror(errno));
        goto exit;
    }

    memset(&local, 0, sizeof(local));
    local.sun_family = AF_UNIX;
    strncpy(local.sun_path, "/tmp/ctrl_client", sizeof(local.sun_path));

    do {
        ret = bind(sock, (struct sockaddr *)&local, sizeof(local));
        if (ret < 0 ) {
            if (errno == EADDRINUSE) {
                unlink(local.sun_path);
                retry++;
                usleep(50000);
            } else
                goto exit;
        }
    } while ((ret < 0) && (retry < 3));

    /* sync with hostapd, the format ctrl_path is ctrl_interface/ctrl_file */
    len = strlen(ctrl_interface) + strlen(ctrl_file) + 2;
    ctrl_path = (char *)malloc(len);
    if (ctrl_path == NULL)
        goto exit;

    snprintf(ctrl_path, len, "%s/%s", ctrl_interface, ctrl_file);
    ctrl_path[len - 1] = '\0';

    memset(&dest, 0, sizeof(dest));
    dest.sun_family = AF_UNIX;
    strncpy(dest.sun_path, ctrl_path, sizeof(dest.sun_path));
    if (connect(sock, (struct sockaddr *)&dest, sizeof(dest)) < 0) {
        //printf("%s: could not connect ctrl_iface socket '%s'\n", __func__, dest.sun_path);
        goto exit;
    }

    flags = fcntl(sock, F_GETFL);
    if (flags >= 0) {
        flags |= O_NONBLOCK;
        if (fcntl(sock, F_SETFL, flags) < 0) {
            printf("%s: fcntl(ctrl->s, O_NONBLOCK) failed\n", __func__);
            /* Not fatal, continue on */
        }
    }

    free(ctrl_path);
    return sock;

exit:
    if (sock >= 0)
        close(sock);

    if (ctrl_path != NULL)
        free(ctrl_path);

    return -1;
}

void ctrl_client_deinit(int sock)
{
    if (sock >= 0)
        close(sock);
}
