/*******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2006-2012 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
* (c) Copyright 2015 MicroSys GmbH
*
********************************************************************************
*
* $File Name:       SIUL2.c$
* @file             SIUL2.c
*
* $Date:            Jan-29-2015$
* @date             Jan-29-2015
*
* $Version:         1.0$
* @version          1.0
*
* Description       SIUL2 driver source file
* @brief            SIUL2 driver source file
*
* --------------------------------------------------------------------
* $Name:  $
*******************************************************************************/
/****************************************************************************//*!
*
*  @mainpage SIUL driver for MPC5744
*
*  @section Intro Introduction
*
*	This package contains SIUL driver for MPC5643L allowing to configure and 
*	control general Input/Output ports.

*  The key features of this package are the following:
*  - Configure pads as digital/analog
*  - Configure pads as input/outputs
*  - Set actual state of the output pads
*
*  For more information about the functions and configuration items see these documents: 
*
*******************************************************************************
*
* @attention 
*            
*******************************************************************************/
/*==================================================================================================
*   Project              : SBC5744
*   Platform             : MPC5744
*   Dependencies         : MPC5744 - Basic SW drivers.
*   All Rights Reserved.
==================================================================================================*/

/*==================================================================================================
Revision History:
                             Modification     Function
Author (core ID)              Date D/M/Y       Name		  Description of Changes
B35993		 				  01/03/2012 	   ALL		  Driver created
MicroSys                      29/01/2015       ALL        Modifications for MPC5744
MivroSys                      04/03/2015   ClearExtIntFlag  clear extint flag
---------------------------   ----------    ------------  ------------------------------------------
==================================================================================================*/

#include "hal_def.h"
#include "siul2.h"

/***************************************************************************//*!
*   @brief Configures GPIO pin as a digital input.
*	@par Include 
*					SIUL.h
* 	@par Description 
*					Function configures specified IO pin as a digital input.
* 	@param[in] pad
*					Pad to be configured - use either MSCR pad number or some 
*					predefined name.
*						Predefined values:		
*						- <tt>"SIUL_PA0" </tt><tt>"SIUL_PA1" </tt><tt>"SIUL_PA2" </tt>
*						- <tt>"SIUL_PA3"</tt><tt>"SIUL_PA4"</tt><tt>"SIUL_PA5"</tt>
*						- <tt>"SIUL_PA6"</tt><tt>"SIUL_PA7"</tt><tt>"SIUL_PA8"</tt>
*						- <tt>"SIUL_PA9"</tt><tt>"SIUL_PA10"</tt><tt>"SIUL_PA11"</tt>
*						- <tt>"SIUL_PA12"</tt><tt>"SIUL_PA13"</tt><tt>"SIUL_PA14"</tt>
*						- <tt> "SIUL_PA15" </tt> <tt>"SIUL_PB0"</tt><tt>"SIUL_PB1"</tt>
*						- <tt> "SIUL_PB2" </tt> <tt> "SIUL_PB3" </tt> <tt> "SIUL_PB4"</tt>
*						- <tt> "SIUL_PB5" </tt> <tt> "SIUL_PB6" </tt> <tt> "SIUL_PB7"</tt>
*						- <tt> "SIUL_PB8" </tt> <tt> "SIUL_PB9" </tt> <tt> "SIUL_PB10" </tt>
*						- <tt> "SIUL_PB11" </tt> <tt> "SIUL_PB12" </tt> <tt> "SIUL_PB13" </tt>
*						- <tt> "SIUL_PB14" </tt> <tt> "SIUL_PB15" </tt> <tt> "SIUL_PC0" </tt>
*						- <tt> "SIUL_PC1" </tt> <tt> "SIUL_PC2" </tt> <tt> "SIUL_PC3" </tt>
*						- <tt> "SIUL_PC4" </tt> <tt> "SIUL_PC5" </tt> <tt> "SIUL_PC6" </tt>
*						- <tt> "SIUL_PC7" </tt> <tt> "SIUL_PC8" </tt> <tt> "SIUL_PC9" </tt>
*						- <tt> "SIUL_PC10" </tt> <tt> "SIUL_PC11" </tt> <tt> "SIUL_PC12" </tt>
*						- <tt> "SIUL_PC13" </tt> <tt> "SIUL_PC14" </tt> <tt> "SIUL_PC15" </tt>
*						- <tt> "SIUL_PD0" </tt> <tt> "SIUL_PD1" </tt> <tt> "SIUL_PD2" </tt>
*						- <tt> "SIUL_PD3" </tt> <tt> "SIUL_PD4" </tt> <tt> "SIUL_PD5" </tt>
*						- <tt> "SIUL_PD6" </tt> <tt> "SIUL_PD7" </tt> <tt> "SIUL_PD8" </tt>
*						- <tt> "SIUL_PD9" </tt> <tt> "SIUL_PD10" </tt> <tt> "SIUL_PD11" </tt>
*						- <tt> "SIUL_PD12" </tt> <tt> "SIUL_PD13" </tt> <tt> "SIUL_PD14" </tt>
*						- <tt> "SIUL_PD15" </tt> <tt> "SIUL_PE0" </tt> <tt> "SIUL_PE1" </tt>
*						- <tt> "SIUL_PE2" </tt> <tt> "SIUL_PE3" </tt> <tt> "SIUL_PE4" </tt>
*						- <tt> "SIUL_PE5" </tt> <tt> "SIUL_PE6" </tt> <tt> "SIUL_PE7" </tt>
*						- <tt> "SIUL_PE8" </tt> <tt> "SIUL_PE9" </tt> <tt> "SIUL_PE10" </tt>
*						- <tt> "SIUL_PE11" </tt> <tt> "SIUL_PE12" </tt> <tt> "SIUL_PE13" </tt>
*						- <tt> "SIUL_PE14" </tt> <tt> "SIUL_PE15" </tt> <tt> "SIUL_PF0" </tt>
*						- <tt> "SIUL_PF1" </tt> <tt> "SIUL_PF2" </tt> <tt> "SIUL_PF3" </tt>
*						- <tt> "SIUL_PF4" </tt> <tt> "SIUL_PF5" </tt> <tt> "SIUL_PF6" </tt>
*						- <tt> "SIUL_PF7" </tt> <tt> "SIUL_PF8" </tt> <tt> "SIUL_PF9" </tt>
*						- <tt> "SIUL_PF10" </tt> <tt> "SIUL_PF11" </tt> <tt> "SIUL_PF12" </tt>
*						- <tt> "SIUL_PF13" </tt> <tt> "SIUL_PF14" </tt> <tt> "SIUL_PF15" </tt>
*						- <tt> "SIUL_PG0" </tt> <tt> "SIUL_PG1" </tt> <tt> "SIUL_PG2" </tt>
*						- <tt> "SIUL_PG3" </tt> <tt> "SIUL_PG4" </tt> <tt> "SIUL_PG5" </tt>
*						- <tt> "SIUL_PG6" </tt> <tt> "SIUL_PG7" </tt> <tt> "SIUL_PG8" </tt>
*						- <tt> "SIUL_PG9" </tt> <tt> "SIUL_PG10" </tt> <tt> "SIUL_PG11" </tt>
*						- <tt> "SIUL_PG12" </tt> <tt> "SIUL_PG13" </tt> <tt> "SIUL_PG14" </tt>
*						- <tt> "SIUL_PG15" </tt> <tt> "SIUL_PH0" </tt> <tt> "SIUL_PH1" </tt>
*						- <tt> "SIUL_PH2" </tt> <tt> "SIUL_PH3" </tt> <tt> "SIUL_PH4" </tt>
*						- <tt> "SIUL_PH5" </tt> <tt> "SIUL_PH6" </tt> <tt> "SIUL_PH7" </tt>
*						- <tt> "SIUL_PH8" </tt> <tt> "SIUL_PH9" </tt> <tt> "SIUL_PH10" </tt>
*						- <tt> "SIUL_PH11" </tt> <tt> "SIUL_PH12" </tt> <tt> "SIUL_PH13" </tt>
*						- <tt> "SIUL_PH14" </tt> <tt> "SIUL_PH15" </tt> <tt> "SIUL_PI0" </tt>
*						- <tt> "SIUL_PI1" </tt> <tt> "SIUL_PI2" </tt> <tt> "SIUL_PI3" </tt>
*						- <tt> "SIUL_PI4" </tt> <tt> "SIUL_PI5" </tt> <tt> "SIUL_PI6" </tt>
*						- <tt> "SIUL_PI7" </tt> <tt> "SIUL_PI8" </tt> <tt> "SIUL_PI9" </tt>
*						- <tt> "SIUL_PI10" </tt> <tt> "SIUL_PI11" </tt> <tt> "SIUL_PI12" </tt>
*						- <tt> "SIUL_PI13" </tt> <tt> "SIUL_PI14" </tt> <tt> "SIUL_PI15" </tt>
*						- <tt> "SIUL_PJ0" </tt> <tt> "SIUL_PJ1" </tt> <tt> "SIUL_PJ2" </tt>
*						- <tt> "SIUL_PJ3" </tt> <tt> "SIUL_PJ4" </tt>
*	@param[in] config
*					Additional configuration of the specified pad. Use number 
*						0 if no additional configuration is desired.
*						Predefined values:
*							</param>
*						Pad Pull-Up/Down resistors			</param> 
*						- "SIUL_PULL_UP" - Internal Pull-Up resistor will 
*						be used.	
*						- "SIUL_PULL_DOWN" - Internal Pull-Down resistor 
*						will be used.	
*						- "SIUL_PULL_DIS" - Internal Pull-Up/DOwn resistors 
*						will be disabled.
*						Pad direction						</param>
*						- "SIUL_INPUT" - Input buffer will be activated.
*						- "SIUL_OUTPUT" - Output buffer will be activated.
*						- "SIUL_BIDIR" - Input and output buffers will be 
*						activated (0x0300).
*						Pad function						</param>	 
*						- "SIUL_GPIO" - Pad will be used as an General IO pin.
*						- "SIUL_ALT1" - Alternate function no.1 will be used on 
*						this pin.
*						- "SIUL_ALT2"	- Alternate function no.2 will be 
*						used on	this pin.
*						- "SIUL_ALT3" - Alternate function no.3 will be used 
*						on this pin.
*						Safe mode control					</param>
*						- "SIUL_SMC"	0x4000
*						Pad type	 
*						- "SIUL_DIGITAL" - Pin will be configured as a digital one.
*						- "SIUL_ANALOG" - Pin will be configured as an analog one 
*						(to be used as an input for ADC).
*						Pad slew rate						</param>	 
*						- "SIUL_HALF_S" - half stength with slew rate will be used. 	
*						- "SIUL_FULL_S" - full stength with slew rate will be used.
*						- "SIUL_HALF" - half stength without slew rate will be used. 	
*						- "SIUL_FULL" - full stength without slew rate will be used.
*						input hysteresis					</param>
						- "SIUL_HYS"	- hysteresis enabled
*						Output type							</param> 
*						- "SIUL_NORMAL"	- Output will be in a Normal (Push-Pull) 
*						configuration.
*						- "SIUL_OPEND" - Output will be in an Open-Drain 
*						configuration.
*	@remarks 	Predefined values can be combined using logical | operator.
*	@par Code sample
*			SIUL_DigitalInput(SIUL_PA8, SIUL_PULL_UP | SIUL_FAST);
*			- Command sets PA8 pad as a digital input with a pull-up resitor 
*			and fast slew rate.
********************************************************************************/
void SIUL_DigitalInput(uint32_t pad, uint32_t config){
	SIUL2.MSCR[pad].R = SIUL_INPUT | config;
}

/***************************************************************************//*!
*   @brief Configures GPIO pin as an analog input.
*	@par Include 
*					SIUL.h
* 	@par Description 
*					Function configures specified IO pin as an analog input.
* 	@param[in] pad
*					Pad to be configured - use either MSCR pad number or some 
*						predefined name.
*	@param[in] config
*					Additional configuration of the specified pad. Use number 
*						0 if no additional configuration is desired.
*	@remarks 	Predefined values can be combined using logical | operator.
*	@par Code sample
*			SIUL_AnalogInput(SIUL_PB7, 0);
*			- Command sets PB7 pin as an analog input with no additional 
*			configuration.
********************************************************************************/
void SIUL_AnalogInput(uint32_t pad, uint32_t config){
	SIUL2.MSCR[pad].R = SIUL_ANALOG | config;
}

/***************************************************************************//*!
*   @brief Configures GPIO pin as a digital output.
*	@par Include 
*					SIUL.h
* 	@par Description 
*					Function configures specified IO pin as a digital output.
* 	@param[in] pad
*					Pad to be configured - use either MSCR pad number or some 
*					predefined name.
*	@param[in] config
*					Additional configuration of the specified pad. Use number 
*					0 if no additional configuration is desired.
*	@remarks 	Predefined values can be combined using logical | operator.
*	@par Code sample
*			SIUL_DigitalOutput(SIUL_PD8, SIUL_ALT1);
*			- Command sets pad PD8 as a digital output with its first alternate 
*			function.
********************************************************************************/
void SIUL_DigitalOutput(uint32_t pad, uint32_t config){
	SIUL2.MSCR[pad].R = SIUL_OUTPUT | config;
}

/***************************************************************************//*!
*   @brief Configures GPIO pin as an Input/Output.
*	@par Include 
*					SIUL.h
* 	@par Description 
*					Function configures specified IO pin as a digital Input/Output.
* 	@param[in] pad
*					Pad to be configured - use either MSCR pad number or some 
*					predefined name.
*	@param[in] config
*					Additional configuration of the specified pad. Use number 
*					0 if no additional configuration is desired.
*	@remarks 	Predefined values can be combined using logical | operator.
*	@par Code sample
*			SIUL_DigitalIO(SIUL_PD2,0);
*			- Command sets pad PD2 as a digital Input/Output (both input and 
*			output buffers are enabled).
********************************************************************************/
void SIUL_DigitalIO(uint32_t pad, uint32_t config){
	SIUL2.MSCR[pad].R = SIUL_BIDIR | config;
}

/***************************************************************************//*!
*   @brief Toggles GPIO pin.
*	@par Include 
*					SIUL.h
* 	@par Description 
*					Function toggles actual state  of the specified pad.
* 	@param[in] pad
*					Pad to be toggled - use either MSCR pad number or some 
*					predefined name.
*	@remarks 	Predefined values can be combined using logical | operator.
*	@par Code sample
*															</param>
*			SIUL_DigitalIO(SIUL_PD2,0);						</param>
*			SIUL_ToggleIO(SIUL_PD2);						</param>
*			- First command sets pad PD2 as a digital Input/Output. Second 
*			command inverts actual state of the PD2 pad.
********************************************************************************/
void SIUL_ToggleIO(uint32_t pad){
	SIUL2.GPDO[pad].R = ~SIUL2.GPDI[pad].R;
}

/***************************************************************************//*!
*   @brief Enables interrupt capability for a GPIO pin.
*	@par Include 
*					SIUL.h
* 	@par Description 
*					Function enables external interrupt for the specified 
*					EIRQ - External Interrupt Request Input specified by its 
*					mask. This is done in the IRER - Interrupt Request Enable 
*					Register.
* 	@param[in] maskEIRQ
*					Mask to specify which EIRQs will be activated.
*					Predefined values:
*					- "SIUL_EIRQ0"
*					- "SIUL_EIRQ1"
*					- "SIUL_EIRQ2"
*					- "SIUL_EIRQ3"
*					- "SIUL_EIRQ4"
*					- "SIUL_EIRQ5"
*					- "SIUL_EIRQ6"
*					- "SIUL_EIRQ7"
*					- "SIUL_EIRQ8"
*					- "SIUL_EIRQ9"
*					- "SIUL_EIRQ10"
*					- "SIUL_EIRQ11"
*					- "SIUL_EIRQ12"
*					- "SIUL_EIRQ13"
*					- "SIUL_EIRQ14"
*					- "SIUL_EIRQ15"
*					- "SIUL_EIRQ16"
*					- "SIUL_EIRQ17"
*					- "SIUL_EIRQ18"
*					- "SIUL_EIRQ19"
*					- "SIUL_EIRQ20"
*					- "SIUL_EIRQ21"
*					- "SIUL_EIRQ22"
*					- "SIUL_EIRQ23"
*					- "SIUL_EIRQ24"
*					- "SIUL_EIRQ25"
*					- "SIUL_EIRQ26"
*					- "SIUL_EIRQ27"
*					- "SIUL_EIRQ28"
*					- "SIUL_EIRQ29"
*					- "SIUL_EIRQ30"
*					- "SIUL_EIRQ31"
*	@remarks 	Predefined values can be combined using logical | operator.
*	@par Code sample
*			SIUL_EnableExtInt(SIUL_EIRQ31);
*			- Command enables external interrupt for the PF13 pad.
********************************************************************************/
void SIUL_EnableExtInt(uint32_t maskEIRQ){
		
	SIUL2.DIRER0.R = SIUL2.DIRER0.R | maskEIRQ;				//ext INT enable	
}

/***************************************************************************//*!
*   @brief Enables interrupt capability on a rising edge.
*	@par Include 
*					SIUL.h
* 	@par Description 
*					Function enables an input filter and interrupt launched on 
*					the rising edge. Filter enables e.g. to filter out switching 
*					noise, when a button is connected directly to the input 
*					without using any special external circuitry.
* 	@param[in] maskEIRQ
*					Mask to specify for which EIRQs the filter and 
*					interrupt will be enabled.
*	@remarks 	Predefined values can be combined using logical | operator.
*	@par Code sample
*			SIUL_EnableExtIntRisingEdge(64);
*			- Command enables filter and an interrupt on the rising edge for 
*			the EIRQ6 - pad PA6.
********************************************************************************/
void SIUL_EnableExtIntRisingEdge(uint32_t maskEIRQ){

	SIUL2.IFER0.R = SIUL2.IFER0.R | maskEIRQ;			//filter enable
	SIUL2.IREER0.R = SIUL2.IREER0.R | maskEIRQ;			//rising edge INT enabled
	
}

/***************************************************************************//*!
*   @brief Enables interrupt capability on a falling edge.
*	@par Include 
*					SIUL.h
* 	@par Description 
*					Function enables an input filter and interrupt launched on 
*					the falling edge. Filter enables e.g. to filter out switching 
*					noise, when a button is connected directly to the input 
*					without using any special external circuitry.
* 	@param[in] maskEIRQ
*					Mask to specify for which EIRQs the filter and 
*					interrupt will be enabled.
*	@remarks 	Predefined values can be combined using logical | operator.
*	@par Code sample
*			SIUL_EnableExtIntFallingEdge(64);
*			- Command enables filter and an interrupt on the falling edge for 
*			the EIRQ6 - pad PA6.
********************************************************************************/
void SIUL_EnableExtIntFallingEdge(uint32_t maskEIRQ){

	SIUL2.IFER0.R = SIUL2.IFER0.R | maskEIRQ;			//filter enable
	SIUL2.IFEER0.R = SIUL2.IFEER0.R | maskEIRQ;			//falling edge INT enabled	
}

/***************************************************************************//*!
*   @brief Returns interrupt flag.
*	@par Include 
*					SIUL.h
* 	@par Description 
*					Function returns interrupt flag for the specified 
*					EIRQ
*						External Interrupt Request Input.
* 	@param[in] maskEIRQ
*						Mask to specify for which EIRQs the filter and 
*						interrupt will be enabled.
*	@return	Returns interrupt flag from the ISR - Interrupt Status Flag Register.
*	@remarks 	Predefined values can be combined using logical | operator.
*	@par Code sample
*			int_flag = SIUL_GetExtIntFlag(SIUL_EIRQ31);
*			- Command returns and stores interrupt flag for EIRQ31 - pad PF13.
********************************************************************************/
uint32_t SIUL_GetExtIntFlag(uint32_t maskEIRQ){
	if((SIUL2.DISR0.R & maskEIRQ) > 0){		//test EIF flag
		return 1;
	}
	else{
		return 0;
	}
}

/***************************************************************************//*!
*   @brief Clears interrupt flag.
*	@par Include 
*					SIUL.h
* 	@par Description 
*					Function clears interrupt flag in the Interrupt Status 
*					Register (ISR) for the specified External Interrupt 
*					Request Input (EIRQ).
* 	@param[in] maskEIRQ
*					Mask to specify which EIRQ's flag will be cleared.
*	@remarks 	Predefined values can be combined using logical | operator.
*	@par Code sample
*			SIUL_ClearExtIntFlag(SIUL_EIRQ31);
*			- Command clears interrupt flag for EIRQ31 - pad PF13.
********************************************************************************/
void SIUL_ClearExtIntFlag(uint32_t maskEIRQ){
	SIUL2.DISR0.R = maskEIRQ;											//clear EIF flag
}

/***************************************************************************//*!
*   @brief Sets GPIO pin state (logical HIGH).
*	@par Include 
*					SIUL.h
* 	@par Description 
*					Function sets state of the specified pad to the logical HIGH 
*					state.
* 	@param[in] pad
*					Pad to be set - use either MSCR pad number or some 
*					predefined name.
*	@remarks 	Pad should be previously configured as a digital output 
*				(see SIUL_DigitalOutput function).
*	@par Code sample
*																	</param>
*			SIUL_DigitalOutput(SIUL_PD8);							</param>
*			SIUL_SetPad(SIUL_PD8);									</param>
*			- First command configures pad PD8 as a digital output and 
*			the second command sets its state to logical HIGH.
********************************************************************************/
void SIUL_SetPad(uint32_t pad){				
	SIUL2.GPDO[pad].R = 1;
}

/***************************************************************************//*!
*   @brief Clears GPIO pin state (logical LOW).
*	@par Include 
*					SIUL.h
* 	@par Description 
*					Function clears state of the specified pad (logical LOW state).
* 	@param[in] pad
*					Pad to be cleared - use either MSCR pad number or some 
*					predefined name.
*	@remarks 	Predefined values can be combined using logical | operator.
*	@par Code sample
*																	</param>
*			SIUL_DigitalOutput(SIUL_PD8);							</param>
*			SIUL_ClearPad(SIUL_PD8);								</param>
*			- First command configures pad PD8 as a digital output and the second 
*			command clears its state (logical LOW).
********************************************************************************/
void SIUL_ClearPad(uint32_t pad){						
	SIUL2.GPDO[pad].R = 0;
}

/***************************************************************************//*!
*   @brief sets GPIO pin state (logical LOW or HIGH).
*	@par Include 
*					SIUL.h
* 	@par Description 
*					Function sets or clears state of the specified pad.
* 	@param[in] pad
*					Pad to be set/cleared - use either MSCR pad number or some 
*					predefined name.
*	@remarks 	Predefined values can be combined using logical | operator.
*	@par Code sample
*																	</param>
*			SIUL_DigitalOutput(SIUL_PD8,1);							</param>
*			SIUL_WritePad(SIUL_PD8,1);								</param>
*			- First command configures pad PD8 as a digital output and the second 
*			command sets its state to logical HIGH..
********************************************************************************/
void SIUL_WritePad(uint32_t pad, uint8_t val){						
	SIUL2.GPDO[pad].R = val & 0x1;
}

/***************************************************************************//*!
*   @brief Gets GPIO pin state.
*	@par Include 
*					SIUL.h
* 	@par Description 
*					Function gets state of the specified pad.
* 	@param[in] pad
*					Pad that will be read.
*	@remarks 	Specified pas has to be configured as input (its input buffer 
*				has to be previously enabled).
*	@par Code sample
*			SIUL_GetPadState(SIUL_PD8);							</param>
*			- Command gets and returns actual state on the pin PD8.
********************************************************************************/
uint32_t SIUL_GetPadState(uint32_t pad){
	return SIUL2.GPDI[pad].R;
}
 






