/*******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2006-2012 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
* (c) Copyright 2015 MicroSys GmbH
*
********************************************************************************
*
* $File Name:       PIT.c$
* @file             PIT.c
*
* $Date:            Jan-30-2015$
* @date             Jan-30-2015
*
* $Version:         1.0$
* @version          1.0
*
* Description       PIT driver source file
* @brief            PIT driver source file
*
* --------------------------------------------------------------------
* $Name:  $
*******************************************************************************/
/****************************************************************************//*!
*
*  @mainpage PIT driver for MPC5744
*
*  @section Intro Introduction
*
*	This package contains PIT driver for MPC5744 allowing to 
*	initialize and configure PIT module.
*
*  The key features of this package are the following:
*  - Initialize PIT for the desired time period
*  - Launch and stop PIT for specified channel
*  For more information about the functions and configuration items see these documents: 
*
*******************************************************************************
*
* @attention 
*            
*******************************************************************************/
/*==================================================================================================
*   Project              : SBC5744
*   Platform             : MPC5744
*   Dependencies         : MPC5744 - Basic SW drivers.
*   All Rights Reserved.
==================================================================================================*/

/*==================================================================================================
Revision History:
                             Modification     Function
Author (core ID)              Date D/M/Y       Name		  Description of Changes
B35993		 				  01/03/2012 	   ALL		  Driver created
MicroSys                      30/01/2015       ALL        Modifications for MPC5744

---------------------------   ----------    ------------  ------------------------------------------
==================================================================================================*/

#include "hal_def.h"
#include "pit.h"

/***************************************************************************//*!
*   @brief Initializes Periodic Interrupt Timer.
*	@par Include 
*					PIT.h
* 	@par Description 
*					Function turns on clock for the PIT module. 
*	@remarks 	In debug mode is the PIT module frozen by default.
*	@par Code sample
*			PIT_Init();
*			- Command activates clock for the PIT module.
********************************************************************************/
void PIT_Init(void)
{
	volatile struct PIT_tag *p_PIT;
	//pointer settings
	p_PIT = &PIT_0;
	p_PIT->MCR.B.MDIS=0; 	// Module clock on
	p_PIT->MCR.B.FRZ=1;  	// Freeze timer in debug	
}

/***************************************************************************//*!
*   @brief Computes and sets up period for a specified channel.
*	@par Include 
*					PIT.h
* 	@par Description 
*					Function computes register settings from clock frequency and expected time base.
*	@param[in] Channel 
*					Number of the PIT's channel (from 0 till 3).
*	@param[in] Clock_Freq
*					System clock frequency.
*	@param[in] ExpectedTimeBase
*					Desired time period for the specified channel.
*	@remarks 	Maximal setable time is given by the maximal value of the 32bit LDVAL register divided 
*				by system the clock frequency ((2^32)/fsys). PIT module should be initialized before 
*				(see PIT_Init function).
*	@par Code sample
*			PIT_Setup(0, 16000000, 0.010);
*			- Command sets up channel no. 0 of the timer with expected time 
*			base 10 ms assuming clock frequency 16 MHz.
********************************************************************************/
void PIT_Setup(int8_t Channel, uint32_t Clock_Freq, double ExpectedTimeBase)
{
	volatile struct PIT_tag *p_PIT;
	int32_t val = (int32_t)(Clock_Freq*ExpectedTimeBase)-1;

	p_PIT = &PIT_0;	
	p_PIT->TIMER[Channel].LDVAL.R=val;
}

/* Port for S32-Studio Issue */
/* created extra function to avoid multiplying floating point */
void PIT_Setup_Debug(int8_t Channel, uint32_t Clock_Freq, uint16_t ExpectedTimeBase)
{
	volatile struct PIT_tag *p_PIT;
	int32_t val = (int32_t)(Clock_Freq/ExpectedTimeBase)-1;

	p_PIT = &PIT_0;
	p_PIT->TIMER[Channel].LDVAL.R=val;
}

/***************************************************************************//*!
*   @brief Enables interrupt for a specified channel.
*	@par Include 
*					PIT.h
* 	@par Description 
*					Function enables interrupt for a specified channel 
*					(sets TIE bit) and clears 
*					interrupt flag (TIF) in the same time.
*	@param[in] Channel
*					Number of the PIT's channel (from 0 till 3).
*	@remarks 	PIT module should be initialized before (see PIT_Init function).
*	@par Code sample
*			PIT_EnableInt(0);
*			- Command enables interruption for the first channel of the PIT.
********************************************************************************/
void PIT_EnableInt(int8_t Channel)
{
	volatile struct PIT_tag *p_PIT;

	p_PIT = &PIT_0;
    p_PIT->TIMER[Channel].TCTRL.B.TIE=1; 
    p_PIT->TIMER[Channel].TFLG.R=1;
}

/***************************************************************************//*!
*   @brief Enables channel of the PIT.
*	@par Include 
*					PIT.h
* 	@par Description 
*					Function enables specified channel by setting TEN bit.
*	@param[in] Channel
*					Number of the PIT's channel (from 0 till 3).
*	@remarks 	PIT module should be initialized before (see PIT_Init function).
*	@par Code sample
*			PIT_EnableChannel(2);
*			- Command enables second channel of the PIT.
********************************************************************************/
void PIT_EnableChannel(int8_t Channel)
{
	volatile struct PIT_tag *p_PIT;
	
	p_PIT = &PIT_0;
    p_PIT->TIMER[Channel].TCTRL.B.TEN=1;
}

/***************************************************************************//*!
*   @brief Disables channel of the PIT (including interrupts).
*	@par Include 
*					PIT.h
* 	@par Description 
*					For the specified channel function clears interrupt flag, disables 
*					interruption and disables the whole channel. 
*	@param[in] Channel
*					Number of the PIT's channel (from 0 till 3).
*	@remarks 	PIT module should be initialized before (see PIT_Init function).
*	@par Code sample
*			PIT_Disable(3);
*			- Command disables channel no. 3 of the PIT.
********************************************************************************/
void PIT_Disable(int8_t Channel)
{
	volatile struct PIT_tag *p_PIT;

	p_PIT = &PIT_0;
    p_PIT->TIMER[Channel].TCTRL.B.TIE=0;
    p_PIT->TIMER[Channel].TCTRL.B.TEN=0;
    p_PIT->TIMER[Channel].TFLG.R=1;
}

/***************************************************************************//*!
*   @brief Returns flag of channel.
*	@par Include 
*					PIT.h
* 	@par Description 
*					Function returns interrupt flag (TIF) of the specified channel.
*	@param[in] Channel
*					Number of the PIT's channel (from 0 till 3).
*	@return Time Interrupt Flag.
*	@remarks 	PIT module should be initialized before (see PIT_Init function).
*	@par Code sample
*			while(PIT_GetFlag(3)==0);
*			- Command waits until the PIT channel no. 3 will overflow (Simple 
*			waiting without any interruption).
********************************************************************************/
int8_t PIT_GetFlag(int8_t Channel)
{
	volatile struct PIT_tag *p_PIT;

	p_PIT = &PIT_0;	
	return (int8_t)p_PIT->TIMER[Channel].TFLG.B.TIF;
}

/***************************************************************************//*!
*   @brief Clears flag of channel.
*	@par Include 
*					PIT.h
* 	@par Description 
*					Function clears Time Interrupt Flag (TIF) for the specified channel.
*	@param[in] Channel
*					Number of the PIT's channel (from 0 till 3).
*	@remarks 	PIT module should be initialized before (see PIT_Init function).
*	@par Code sample
*			PIT_ClearFlag(0);
*			- Command clears TIF flag of the channel no. 0.
********************************************************************************/
void PIT_ClearFlag(int8_t Channel)
{
	volatile struct PIT_tag *p_PIT;

	p_PIT = &PIT_0;
    p_PIT->TIMER[Channel].TFLG.R=1;
}

/***************************************************************************//*!
*   @brief Disables PIT channel.
*	@par Include 
*					PIT.h
* 	@par Description 
*					Function clears Timer Enable Bit (TEN) for the specified PIT channel.
*	@param[in] Channel
*					Number of the PIT's channel (from 0 till 3).
*	@remarks 	PIT module should be initialized before (see PIT_Init function).
*	@par Code sample
*			PIT_DisableChannel(1);
*			- Command disables first PIT channel.
********************************************************************************/
void PIT_DisableChannel(int8_t Channel)
{
	volatile struct PIT_tag *p_PIT;
	
	p_PIT = &PIT_0;
    p_PIT->TIMER[Channel].TCTRL.B.TEN=0;
}

/***************************************************************************//*!
*   @brief Checks if a specified PIT channel is enabled.
*	@par Include 
*					PIT.h
* 	@par Description 
*					Function returns content of the Timer Enable Bit (TEN) for the specified PIT channel.
*	@param[in] Channel
*					Number of the PIT's channel (from 0 till 3).
*	@return Number of the PIT's channel (from 0 till 3).
*	@remarks 	PIT module should be initialized before (see PIT_Init function).
*	@par Code sample
*			res = PIT_IsChannelEnabled(0);
*			- Command returns state of PIT channel number 0.
********************************************************************************/
uint32_t PIT_IsChannelEnabled(int8_t Channel)
{
	volatile struct PIT_tag *p_PIT;

	p_PIT = &PIT_0;	
	return p_PIT->TIMER[Channel].TCTRL.B.TEN;			//returns 1 if PIT channel is enabled or 0
}






