/* main.c - Out-of-Box example for MPC5744P */
/* Description:  Tests various I/O and communication interfaces */
/* Rev 1.0 Dec 6 2017 MicroSys - production version */
/* Copyright NXP Semiconductor, Inc 2016 All rights reserved. */

/*******************************************************************************
* NXP Semiconductor Inc.
* (c) Copyright 2016 NXP Semiconductor, Inc.
* ALL RIGHTS RESERVED.
********************************************************************************
Services performed by NXP in this matter are performed AS IS and without
any warranty. CUSTOMER retains the final decision relative to the total design
and functionality of the end product. NXP neither guarantees nor will be
held liable by CUSTOMER for the success of this project.
NXP DISCLAIMS ALL WARRANTIES, EXPRESSED, IMPLIED OR STATUTORY INCLUDING,
BUT NOT LIMITED TO, IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR
A PARTICULAR PURPOSE ON ANY HARDWARE, SOFTWARE ORE ADVISE SUPPLIED TO THE PROJECT
BY NXP, AND OR NAY PRODUCT RESULTING FROM NXP SERVICES. IN NO EVENT
SHALL NXP BE LIABLE FOR INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING OUT OF
THIS AGREEMENT.

CUSTOMER agrees to hold NXP harmless against any and all claims demands or
actions by anyone on account of any damage, or injury, whether commercial,
contractual, or tortuous, rising directly or indirectly as a result of the advise
or assistance supplied CUSTOMER in connection with product, services or goods
supplied under this Agreement.
********************************************************************************
* File              main.c
* Owner             MicroSys
* Version           1.0
* Date              Dec-5-2017
* Classification    General Business Information
* Brief             Tests various I/O and communication interfaces
********************************************************************************
* Detailed Description:
* This is the Miriac(TM)-EK5744 out-of-box demo.
*
* This demo tests various I/O and communications in the system.  You need to connect
* the board to a DHCP-capable network such as a LAN. The demo sets up an IP address
* and sends feedback over UART.  It also tests the CAN loopback communication.
* The Miriac(TM)-EK5744 functional safety kit supports two CAN ports, one directly
* from the MCU and another through the SBC. This code example tests the loopback
* between the two modules. Finally, digital and analog inputs are read and displayed
* on the UART terminal. The test repeats indefinitely.
*
* ------------------------------------------------------------------------------
* Test HW:         Miriac(TM)-EK5744
* MCU:             MPC5744P
* Terminal:        115200, 8N1, None
* Fsys:            200 MHz PLL with 40 MHz crystal reference
* Debugger:        USB Multilink
* Target:          FLASH
* EVB connection:  Follow the steps shown in the Miriac(TM)-EK5744 Quick
* 					Start Guide.
*
********************************************************************************
Revision History:
Version  Date         Author  			Description of Changes
1.0      Dec-5-2017   MicroSys	  		Initial version

*******************************************************************************/

#define _MAIN_


#include "project.h"
#include "fnet.h"

extern void xcptn_xmpl(void);
extern void MC_MODE_INIT(void);
extern void McalAdc_vInit(void);
void can_demo(void);
void configure_port_pins(void);
void InitFlexCAN_0(void);
void InitFlexCAN_2(void);
void io_test_init(void);
void io_test(void);

#define LED_ON 1
#define LED_OFF 0
#define LED1 SIUL2.GPDO[58].R
#define LED2 SIUL2.GPDO[120].R
#define LED3 SIUL2.GPDO[12].R
#define LED4 SIUL2.GPDO[14].R
#define BTN1 (SIUL2.GPDI[77].R == 0)
#define BTN2 (SIUL2.GPDI[123].R == 0)
#define BTN3 (SIUL2.GPDI[121].R == 0)
#define BTN4 (SIUL2.GPDI[59].R == 0)

uint32_t i = 0;

/* This is the IP address UDP packets are sent to */
#define IP_TO_SEND_UDP      FNET_GEN_ADDR(192,168,0,38);

/**********************************************************************************************
 * External objects
 **********************************************************************************************/
extern const struct fnet_fs_rom_image fnet_fs_image;
extern volatile uint32_t ui32_1ms_counter;

/**********************************************************************************************
 * Global variables
 **********************************************************************************************/
#define FAPP_HTTP_POST_BUFFER_SIZE    (50)	/* POST message buffer limited to 50 characters */

struct fnet_http_params params;
struct fnet_http_cgi cgihandler[3];
struct fnet_http_post posthandler[2];

struct sockaddr_in UDP_Destination;
SOCKET UDP_Client_Socket;
char TxBuffer[5] = {0, 0, 0, 0, 0};
fnet_ip_addr_t add;

static char fapp_http_post_buffer[FAPP_HTTP_POST_BUFFER_SIZE + 1];	/* For zero-termination */
static char fapp_http_postLED_buffer[16];	/* Only need 2 bytes but no harm */
static char fapp_http_cgi_buffer[100]; 		/* CGI Temporary buffer. */

/**********************************************************************************************
 * Local function prototypes
 *********************************************************************************************/
void fapp_dhcp_handler_updated (fnet_netif_desc_t netif, void *param);


/************************************************************************
* NAME: fapp_http_cgi_post_handle
*
* DESCRIPTION:
*************************************************************************/
static int fapp_http_cgi_post_handle(char * query, long *cookie)
{
    (void)query;

    *cookie = (uint32_t)fapp_http_post_buffer; /* Save fapp_http_post_buffer as cookie.*/                        
    
    return FNET_OK;
}

/************************************************************************
* NAME: fapp_http_string_buffer_respond
*
* DESCRIPTION:
*************************************************************************/
static unsigned long fapp_http_string_buffer_respond(char *buffer, unsigned long buffer_size, char *eof, long *cookie)
{
    unsigned long result = 0;
   
    char *string_buffer_ptr = (char *) *cookie;
    
    unsigned long send_size = fnet_strlen(string_buffer_ptr);
    
    *eof = 1; /* No aditional send by default. */
    
    if(buffer && buffer_size)
    {
	    if(send_size>buffer_size)
		{
	        send_size = buffer_size;
		}
	 
	    fnet_memcpy(buffer, string_buffer_ptr, send_size);

	    string_buffer_ptr += send_size;
	    if(*string_buffer_ptr!='\0') /* If any data for sending.*/ 
		{
			*eof = 0;    /* Need more itteration. */
		}

	    result = send_size;
	    
	    *cookie = (long)string_buffer_ptr; /* Save cgi_buffer_ptr as cookie.*/
    }
    
    return result;    
}

/************************************************************************
* NAME: fapp_http_post_receive
*
* DESCRIPTION:
*************************************************************************/
static int fapp_http_post_receive (char *buffer, unsigned long buffer_size, long *cookie)
{
    long post_buffer_index = *cookie;
    long post_buffer_free_size = FAPP_HTTP_POST_BUFFER_SIZE - post_buffer_index;
    long receive_size = (long)buffer_size;
    
    if(post_buffer_free_size)
    {
        if(receive_size > post_buffer_free_size)
	        receive_size = post_buffer_free_size; 
	 
	    fnet_memcpy(&fapp_http_post_buffer[post_buffer_index], buffer, (unsigned)receive_size);
	    post_buffer_index += receive_size;
	    fapp_http_post_buffer[post_buffer_index] = '\0';
    
		fnet_printf("Server> %s\n",fapp_http_post_buffer);		
	
        *cookie = post_buffer_index; /* Save buffer index as cookie.*/
    }
    
    return FNET_OK;
}


/************************************************************************
* NAME: fapp_http_cgi_postLED_handle
*
* DESCRIPTION:
*************************************************************************/
static int fapp_http_cgi_postLED_handle(char * query, long *cookie)
{
    (void)query;

    *cookie = (uint32_t)fapp_http_postLED_buffer; /* Save fapp_http_postLED_buffer as cookie.*/                        
    
    return FNET_OK;
}

/************************************************************************
* NAME: fapp_http_postLED_receive
*
* DESCRIPTION:
*************************************************************************/
static int fapp_http_postLED_receive (char *buffer, unsigned long buffer_size, long *cookie)
{
    long post_buffer_index = *cookie;
    long post_buffer_free_size = 16 - post_buffer_index;
    long receive_size = (long)buffer_size;
    
    if(post_buffer_free_size)
    {
        if(receive_size > post_buffer_free_size)
	        receive_size = post_buffer_free_size; 
	 
	    fnet_memcpy(&fapp_http_postLED_buffer[post_buffer_index], buffer, (unsigned)receive_size);
	    post_buffer_index += receive_size;
	    fapp_http_postLED_buffer[post_buffer_index] = '\0';
    
        *cookie = post_buffer_index; /* Save buffer index as cookie.*/
		
		switch (fapp_http_postLED_buffer[0])
		{
			case 0x31:				
				if(fapp_http_postLED_buffer[1] & 0x1)
				{
					LED1 = LED_OFF;
				}
				else
				{
					LED1 = LED_ON;
				}
				break;
			case 0x32:
				if(fapp_http_postLED_buffer[1] & 0x1)
				{
					LED2 = LED_OFF;
				}
				else
				{
					LED2 = LED_ON;
				}
				break;
			case 0x33:
				if(fapp_http_postLED_buffer[1] & 0x1)
				{
					LED3 = LED_OFF;
				}
				else
				{
					LED3 = LED_ON;
				}
				break;
			case 0x34:
				if(fapp_http_postLED_buffer[1] & 0x1)
				{
					LED4 = LED_OFF;
				}
				else
				{
					LED4 = LED_ON;
				}
				break;
		}
		
		
		
    }
    
    return FNET_OK;
}

/************************************************************************
* NAME: fapp_http_cgi_periodic_handle
*
* DESCRIPTION:
*************************************************************************/
static int fapp_http_cgi_periodic_handle(char * query, long *cookie)
{	
	/* Read the status of the buttons */
	int btn1 = BTN1;
	int btn2 = BTN2;
	int btn3 = BTN3;
	int btn4 = BTN4;	
  	
    (void)query;

	/* Write to the temprorary buffer. */
    fnet_snprintf(fapp_http_cgi_buffer, sizeof(fapp_http_cgi_buffer), 
                        "({\"btn1\":%d,\"btn2\":%d,\"btn3\":%d,\"btn4\":%d})", 
                        btn1, btn2, btn3, btn4);

    *cookie = (long)fapp_http_cgi_buffer; /* Save fapp_http_cgi_buffer as cookie.*/                        
    
    return FNET_OK;
}

void GPIO_Init(void)
{
#if 0
	SIUL2.MSCR[42].B.OBE = 1;		      /* port C10 output for PWENA */
	SIUL2.MSCR[45].B.OBE = 1;		      /* port C13 output for PWENB */

	SIUL2.GPDO[42].B.PDO = 0;			/* PWENA */
	SIUL2.GPDO[45].B.PDO = 0;			/* PWENB */

	/* PD10 */
	LED1 = LED_OFF;		
	SIUL2.MSCR[58].B.SSS = 0;
	SIUL2.MSCR[58].B.OBE = 1;

	/* PH8 */
	LED2 = LED_OFF;		
	SIUL2.MSCR[120].B.SSS = 0;
	SIUL2.MSCR[120].B.OBE = 1;

	/* PA12 */
	LED3 = LED_OFF;		
	SIUL2.MSCR[12].B.SSS = 0;
	SIUL2.MSCR[12].B.OBE = 1;

	/* PA14 */
	LED4 = LED_OFF;		
	SIUL2.MSCR[14].B.SSS = 0;
	SIUL2.MSCR[14].B.OBE = 1;

	/* PA8 */
	SIUL2.MSCR[77].B.SSS = 0;
	SIUL2.MSCR[77].B.IBE = 1;

	/* PA9 */
	SIUL2.MSCR[123].B.SSS = 0;
	SIUL2.MSCR[123].B.IBE = 1;

	/* PA12 */
	SIUL2.MSCR[121].B.SSS = 0;
	SIUL2.MSCR[121].B.IBE = 1;

	/* PA13 */
	SIUL2.MSCR[59].B.SSS = 0;
	SIUL2.MSCR[59].B.IBE = 1;
#else
	configure_port_pins();
#endif	
}

/************************************ Main ***********************************/
int main()
{
	uint32_t udp_tx_count = 0;
    MC_MODE_INIT();                           /* Setup the MCU clocks and modes */
    xcptn_xmpl ();                           /* Configure Interrupts */

    GPIO_Init();
	
/* Initialize UART */	
    fnet_cpu_serial_init(0, 115200);	/* Use FlexTimer 0 at 115200 bauds */
    fnet_printf("\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n");	/* Scroll a page on the serial terminal */
    fnet_printf("Welcome to the MPC5744P Ethernet Demo debug console\n");

	io_test_init();
  	InitFlexCAN_2();
  	InitFlexCAN_0();
	
    if (fnet_init_static() != FNET_ERR) 
    {
	fnet_printf("Initalized Stack...\n");

	/* DHCP Initialization */      
	fnet_dhcp_init(fnet_netif_get_default(), 0);		
	fnet_dhcp_handler_updated_set(fapp_dhcp_handler_updated, 0);
	fnet_printf("Started DHCP service \n");

	/* Create client socket */
	fnet_memset(&UDP_Destination, 0, sizeof(UDP_Destination));
	UDP_Destination.sin_family = AF_INET;
	UDP_Destination.sin_port = 2004;
	UDP_Destination.sin_addr.s_addr = IP_TO_SEND_UDP;

	UDP_Client_Socket = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
	bind(UDP_Client_Socket, (struct sockaddr *) &UDP_Destination, sizeof(UDP_Destination));
	connect(UDP_Client_Socket, (struct sockaddr *) &UDP_Destination, sizeof(UDP_Destination));


	/* Initialize the FileSystem */		
	if( fnet_fs_init( ) == FNET_OK)
	{
	    /* Register the FNET ROM FS. */
	    fnet_fs_rom_register( );

	    /* Mount the FNET ROM FS image. */
	    if( fnet_fs_mount( FNET_FS_ROM_NAME, FNET_FS_ROM_NAME, (void *)&fnet_fs_image ) != FNET_ERR )
	    {
			fnet_printf("Mounted FileSystem\n");	
	    }
	    else
	    {
			fnet_printf("Failed to mount FileSystem\n");	
	    }		        		        		         
	}
	else
	{
	    fnet_printf("Failed to init FNET FS\n");
	} 	        

	/* Configure CGI handlers for dynamic web generation */
	/* Handler to receive/trasnmit the data from/to the POST/GET textareas */
	cgihandler[0].name = "post.cgi";
	cgihandler[0].handle = fapp_http_cgi_post_handle;
	cgihandler[0].send = fapp_http_string_buffer_respond;
	
	/* Handler to periodically report the status of the push-buttons */
	cgihandler[1].name = "periodic.cgi";
	cgihandler[1].handle = fapp_http_cgi_periodic_handle;
	cgihandler[1].send = fapp_http_string_buffer_respond;	

	/* Handler to receive LED control commands */
	cgihandler[2].name = "postLED.cgi";
	cgihandler[2].handle = fapp_http_cgi_postLED_handle;
	cgihandler[2].send = 0;
	
	/* PostHandler to receive data from the POST textarea  */
	posthandler[0].name = "post.cgi";
	posthandler[0].handle = 0;
	posthandler[0].receive = fapp_http_post_receive;
	posthandler[0].send = 0;

	/* PostHandler to receive data from the LED controller */
	posthandler[1].name = "postLED.cgi";
	posthandler[1].handle = 0;
	posthandler[1].receive = fapp_http_postLED_receive;
	posthandler[1].send = 0;
	
	/* Configure the HTTP server parameters */		
	params.root_path = FNET_FS_ROM_NAME;    	/* Root directory path */
	params.index_path = "index.html";   		/* Index file path, relative to the root_path */
	params.port = 80;
	params.ip_address = INADDR_ANY;
	params.cgi_table = cgihandler;
	params.post_table = posthandler;	


	/* Enable HTTP server */
	(void)fnet_http_init(&params);	        
	fnet_printf("HTTP Server Initiated\n");

	
	fnet_printf("Waiting for DHCP server to assign IP...\n");
	for (;;)
	{
	    if(fnet_dhcp_state() == FNET_DHCP_STATE_BOUND)		/* If DHCP assigned address */
	    {
			udp_tx_count++;
			if (udp_tx_count == 1000)
			{
				send(UDP_Client_Socket, (char *)&TxBuffer, sizeof(TxBuffer), 0);				

				TxBuffer[0] = (char)((add>>24)&0xFF);		/* Send IP address */	            						
				TxBuffer[1] = (char)((add>>16)&0xFF);	            						
				TxBuffer[2] = (char)( (add>>8)&0xFF);	            						
				TxBuffer[3] = (char)(      add&0xFF);	    
				TxBuffer[4]++;	            			/* Send a data value */
				udp_tx_count = 0;
			}
			if ((ui32_1ms_counter % 1000) == 0)
			{
				io_test();
				can_demo();
			}
	    }

	    fnet_poll();  	        			/* Attend services as DHCP or HTTP server */
	}

    }
    else
    {
		fnet_printf("ERROR: FNET stack initialization is failed!\n");			
    }		

    return 0;
}


/***********************************************************************************************
 *
 * @brief    fapp_dhcp_handler_updated - Callback for the DHCP status update
 * @param    fnet_netif_desc_t netif - network interface whose status was updated
 * @return   none
 *
 ************************************************************************************************/    
void fapp_dhcp_handler_updated (fnet_netif_desc_t netif, void *param)
{	
    (void)param;
    add = fnet_netif_get_address(netif);
    fnet_printf("DHCP assigned IP: %u.%u.%u.%u\n",(add>>24)&0xFF, (add>>16)&0xFF, (add>>8)&0xFF, add&0xFF);    	
    fnet_printf("WebServer is accessible via web browser. Use the assigned IP as URL\n\n");	
}



