/**************************************************************************
* 
* Copyright 2005-2011 by Andrey Butok. Freescale Semiconductor, Inc.
*
**********************************************************************/ /*!
*
* @file fnet_socket_prv.h
*
* @author Andrey Butok
*
* @date May-24-2011
*
* @version 0.1.15.0
*
* @brief Private. Socket API.
*
***************************************************************************/

#ifndef _FNET_SOCKET_PRV_H_

#define _FNET_SOCKET_PRV_H_

#include "fnet_error.h"
#include "fnet_socket.h"

#if FNET_CFG_TCP

#include "fnet_tcp.h"

#endif

#include "fnet_netbuf.h"
#include "fnet_ip_prv.h"


/************************************************************************
*  Definitions.
*************************************************************************/

/* A allocated port will always be
 * FNET_SOCKET_PORT_RESERVED < local_port <= FNET_SOCKET_PORT_USERRESERVED (ephemeral port)
 */
#define FNET_SOCKET_PORT_RESERVED       (1024)  /* In host byte order.*/
#define FNET_SOCKET_PORT_USERRESERVED   (5000)  /* In host byte order.*/

#define FNET_SOCKET_DESC_RESERVED       (-1)    /* The descriptor is reserved.*/

extern int fnet_enabled;


/**************************************************************************/ /*!
 * @internal
 * @brief    Structure of socket buffer.
 ******************************************************************************/
typedef struct
{
    unsigned long count;   /**< Aactual chars in buffer.*/
    unsigned long count_max;        /**< Max actual char count (9*1024).*/
    fnet_netbuf_t *net_buf_chain;   /**< The net_buf chain.*/
#if FNET_CFG_SOCKET_BLOCKING_SUPPORT     
    unsigned short timeout;         /**< Timeout for read/write (int timer ticks).*/    
#endif    
    unsigned char is_shutdown;      /**< The socket has been shut down for read/write.*/    
} fnet_socket_buffer_t;

/**************************************************************************/ /*!
 * @internal
 * @brief    Structure contains parameter of receive datagram 
 * (only for SOCK_DGRAM).
 ******************************************************************************/
typedef struct
{
    fnet_ip_addr_t addr;        /**< IP address (in network byte order).*/
    unsigned short port;        /**< Port (in network byte order).*/
    fnet_netbuf_t *ip_options;  /**< Pointer to IP options.*/
} fnet_socket_buffer_addr_t;

/**************************************************************************/ /*!
 * @internal
 * @brief    Socket options.
 ******************************************************************************/
typedef struct
{
    fnet_ip_sockopt_t ip_opt;   /**< IP options.*/

#if FNET_CFG_TCP

    fnet_tcp_sockopt_t tcp_opt; /**< TCP options.*/

#endif

    int error;                  /**< Socket last error.*/
    int local_error;            /**< Socket local error (ICMP, on timeout).*/
    int flags;                  /**< Socket flags.*/
    int linger;                 /**< Lingers on close if unsent data is present (in timer ticks).*/
} fnet_socket_option_t;

/************************************************************************
*    Structure per socket.
*************************************************************************/
typedef struct _socket
{
    struct _socket *next;                    /**< Pointer to the next socket structure.*/
    struct _socket *prev;                    /**< Pointer to the previous socket structure.*/
    SOCKET descriptor;                       /**< Socket descriptor.*/

    fnet_socket_state_t state;      /**< Socket state.*/
    struct fnet_prot_if *protocol_interface; /**< Points to protocol specific functions (interface).*/
    void *protocol_control;                  /**< Points to protocol control structure (optional).*/

    /* For sockets with SO_ACCEPTCONN.*/
    struct _socket *partial_con;            /**< Queue of partial connections.*/
    int partial_con_len;                    /**< Number of connections on partial_con.*/
    struct _socket *incoming_con;           /**< Queue of incoming connections.*/
    int incoming_con_len;                   /**< Number of connections on incoming_con.*/
    int con_limit;                          /**< Max number queued connections (specified  by "listen").*/
    struct _socket *head_con;               /**< Back pointer to accept socket.*/

    fnet_socket_buffer_t receive_buffer; /**< Socket buffer for incoming data.*/
    fnet_socket_buffer_t send_buffer;    /**< Socket buffer for outgoing data.*/

    /* Common protocol options.*/
    fnet_ip_addr_t foreign_addr;         /**< Foreign IP address (in network byte order).*/
    unsigned short foreign_port;         /**< Foreign port (in network byte order).*/ 
    fnet_ip_addr_t local_addr;           /**< Local IP address (in network byte order).*/
    unsigned short local_port;           /**< Local port (in network byte order).*/
    fnet_socket_option_t options;        /**< Collection of socket options.*/
} fnet_socket_t;

/************************************************************************
*     Function Prototypes
*************************************************************************/
int fnet_stack_init( void );
void fnet_stack_release( void );

void fnet_socket_init( void );
void fnet_socket_list_add( fnet_socket_t ** head, fnet_socket_t *s );
void fnet_socket_list_del( fnet_socket_t ** head, fnet_socket_t *s );
void fnet_socket_set_error( fnet_socket_t *sock, int error );
fnet_socket_t *fnet_socket_lookup( fnet_socket_t *head,       fnet_ip_addr_t local_addr,
                                   unsigned short local_port, fnet_ip_addr_t foreign_addr,
                                   unsigned short foreign_port );
unsigned short fnet_socket_uniqueport( fnet_socket_t *head, fnet_ip_addr_t local_addr );
int fnet_socket_conflict( fnet_socket_t *head,         fnet_ip_addr_t local_addr,   unsigned short local_port,
                          fnet_ip_addr_t foreign_addr, unsigned short foreign_port, int wildcard );
fnet_socket_t *fnet_socket_copy( fnet_socket_t *sock );
void fnet_socket_release( fnet_socket_t ** head, fnet_socket_t *sock );
int fnet_socket_buffer_append_address( fnet_socket_buffer_t *sb, fnet_netbuf_t *nb, fnet_ip_addr_t addr,
                                       unsigned short port,      fnet_netbuf_t *opt );
int fnet_socket_buffer_append_record( fnet_socket_buffer_t *sb, fnet_netbuf_t *nb );
int fnet_socket_buffer_read_address( fnet_socket_buffer_t *sb, char *buf, int len, fnet_ip_addr_t *addr,
                                     unsigned short *port,     int remove );
int fnet_socket_buffer_read_record( fnet_socket_buffer_t *sb, char *buf, int len, int remove );
void fnet_socket_buffer_release( fnet_socket_buffer_t *sb );

int fnet_ip_setsockopt( fnet_socket_t *sock, int level, int optname, char *optval, int optlen );
int fnet_ip_getsockopt( fnet_socket_t *sock, int level, int optname, char *optval, int *optlen );


#endif
