#define LINPROT_C
/******************************************************************************
*                                                       
*       Copyright (C) 2005 Freescale Semiconductor, Inc.
*       All Rights Reserved								              
*														                            
* Filename:     linprot.c                
*														                            
* Revision:      										                    
*														                            
* Functions:    LIN Protocol module.  Master and Slave State Machines
*												                            
* Description:  
*
* Notes:        
*
******************************************************************************/

#include "linbase.h"

#define LIN_SYNC_FIELD                  0x55    /* Sync byte */
#define LIN_REQUEST_COMMAND_FRAME       0x3C    /* Master Request Command Frame ID */
#define LIN_RESPONSE_COMMAND_FRAME      0x3D    /* Slave Response Command Frame ID */
#define LIN_COMMAND_SLEEP               0x00    /* Command code for Go to Sleep Mode */

#pragma MESSAGE DISABLE C4200 /* WARNING C4200: other segment than in previous declaration */

/****************************************************************************
 * All common-purpose RAM variables shall be declared here.
 ***************************************************************************/
/****************************************************************************
 * It require to perform zero page RAM placement optimisation.
 ***************************************************************************/


 LIN_BYTE      LIN_StateFlags;             /* State machine status  */

 LIN_BYTE      LIN_ProtBuf[LIN_PROT_BUF_LEN];   /* Protocol buffer */
 LIN_BYTE      LIN_TmpSCIByte;             /* byte to receive byte from SCI                   */
                                           /* it shoud be used only under disabled interrupts */
 LIN_BYTE      LIN_ChkSum;                 /* byte to calculate checksum                      */
 LIN_BYTE      LIN_MsgLength;              /* Message length [2, 4, 8]   */

 LIN_BYTE      LIN_SaveIdx;                /* Current processed message index */

 LIN_BYTE      LIN_BufCounter;             /* ProtBuf counter point to sended byte for FLAG_DATA_SEND         */
                                                    /* or to first free byte for FLAG_DATA_RECEIVE or FLAG_DATA_IGNORE */




/****************************************************************************
 * Functions 
 ***************************************************************************/

 /***************************************************************************
 * Function :   LIN_CopyToBuf
 *
 * Description: Copy data from src to LIN_ProtBuf
 *
 * Returns:     none
 *
 * Notes:       OPTIM
 **************************************************************************/
void LIN_CopyToBuf( LIN_BYTE * src )
{
    LIN_BYTE i;

#if defined(LINAPI_1_0)

    /* reverse copy for LIN API */
        
    for( i = 0; i < LIN_MsgLength; i++ )
    {
        LIN_ProtBuf[i] = src[LIN_MsgLength - i - 1];
    }

#else /* !defined(LINAPI_1_0) */

    /* normal copy for Freescale API */
        
    for( i = 0; i < LIN_MsgLength; i++ )
    {
        LIN_ProtBuf[i] = src[i];
    }

#endif
}


/***************************************************************************
 * Function :   LIN_CopyFromBuf
 *
 * Description: Copy data from LIN_ProtBuf to dst 
 *
 * Returns:     none
 *
 * Notes:       OPTIM
 **************************************************************************/
void LIN_CopyFromBuf( LIN_BYTE * dst )
{
    LIN_BYTE i;

#if defined(LINAPI_1_0)

    /* reverse copy for LIN API */
        
    for( i = 0; i < LIN_MsgLength; i++ )
    {
        dst[i] = LIN_ProtBuf[LIN_MsgLength - i - 1];
    }

#else /* !defined(LINAPI_1_0) */

    /* normal copy for Freescale API */
        
    for( i = 0; i < LIN_MsgLength; i++ )
    {
        dst[i] = LIN_ProtBuf[i];
    }

#endif
}


/***************************************************************************
 * Function :   LIN_ChkAddByte
 *
 * Description: Add byte with carry to LIN_ChkSum variable 
 *                       
 * Returns:     none
 *
 * Notes:       
 *
 **************************************************************************/
#if defined(CW12)
#pragma OPTION ADD handle_register "-Or"
#endif /* defined(CW12) */

void LIN_ChkAddByte(LIN_BYTE byte)
{
#if defined(CW12)
    asm   
    {
        ADDB  LIN_ChkSum;
        ADCB  #0;
        STAB  LIN_ChkSum;
    }

#endif /* defined(CW12) */

#if defined(COSMIC12)
    {   
        _asm ("ADDB	_LIN_ChkSum");
        _asm ("ADCB	#0");
        _asm ("STAB	_LIN_ChkSum");
    }   
#endif /* defined(COSMIC12) */
}

#if defined(CW12)
#pragma OPTION DEL handle_register 
#endif /* defined(CW12) */


/***************************************************************************
 * Function :   LIN_Error
 *
 * Description: Set error and go to LIN_FLAG_IGNORE state
 * 
 * Returns:     none
 *
 * Notes:       Can be called only from states:
 *              MASTER: LIN_FLAG_SEND_HEADER,
 *                      LIN_FLAG_SEND_DATA,
 *                      LIN_FLAG_IGNORE_DATA,
 *                      LIN_FLAG_RECEIVE_DATA
 *              SLAVE:  LIN_FLAG_RECEIVE_SYNC,
 *                      LIN_FLAG_RECEIVE_ID,
 *                      LIN_FLAG_SEND_DATA,
 *                      LIN_FLAG_RECEIVE_DATA
 *
 **************************************************************************/
void LIN_Error( void )
{
#if defined(MASTER)
    LIN_CancelTimeout();        /* Cancel MaxFrame timeout */
#endif /* defined(MASTER) */

    /* Check what type of the error occure -- Tx or Rx */

#if defined(MASTER)
    if ( LIN_StateFlags & ( LIN_FLAG_SEND_HEADER | LIN_FLAG_SEND_DATA ))
#endif /* defined(MASTER) */
#if defined(SLAVE)
    if ( LIN_StateFlags & LIN_FLAG_SEND_DATA )
#endif /* defined(SLAVE) */
    {
        LIN_SetTxErr();
    }
    else
    {
        LIN_SetRxErr();
    }

    LIN_StateFlags = LIN_FLAG_IGNORE;
}


/***************************************************************************
 * Function :   LIN_EndOfFrame
 *
 * Description: Non-error end of frame
 * 
 * Returns:     none
 *
 * Notes:       Can be called only from states:
 *              MASTER: 
 *                      LIN_FLAG_SEND_DATA,
 *                      LIN_FLAG_IGNORE_DATA,
 *                      LIN_FLAG_RECEIVE_DATA
 *              SLAVE:  
 *                      LIN_FLAG_RECEIVE_ID,
 *                      LIN_FLAG_SEND_DATA,
 *                      LIN_FLAG_RECEIVE_DATA
 *
 **************************************************************************/
void LIN_EndOfFrame( void )
{
#if defined(MASTER)
    LIN_CancelTimeout();        /* Cancel MaxFrame timeout */
#endif /* defined(MASTER) */

    /* Check what type of frame ended -- Tx or Rx */
    if ( LIN_StateFlags & LIN_FLAG_SEND_DATA )
    {
        LIN_ShiftTxErr();
    }
    else
    {
        LIN_ShiftRxErr();
    }

#if !defined(LINAPI_1_0)

    /* Check Command Frame -- only for Freescale API -- OPTIM */

    if ( 
            /* We process Command Frame only if node configured to Receive or send data for it */        
            ( LIN_StateFlags & (LIN_FLAG_SEND_DATA | LIN_FLAG_RECEIVE_DATA) ) &&
        
            /* AND check frame's ID that we have just processed */

#if defined(MASTER)
            /* For Master: LIN_SaveIdentifier -- just processed frame's identifier */
            ( LIN_SaveIdentifier == LIN_REQUEST_COMMAND_FRAME ) 
#endif /* defined(MASTER) */

#if defined(SLAVE)
        /* For Slave: LinMsgId[LIN_SaveIdx] & LIN_MSG_INDEX_MASK -- just processed frame's identifier */
        /* NB: LIN_SaveIdx exists becouse we already check that we are not ignore this message */
            ( (LinMsgId[LIN_SaveIdx] & LIN_MSG_INDEX_MASK) == LIN_REQUEST_COMMAND_FRAME )
#endif /* defined(SLAVE) */
       )
    {
        /* yes, we have just processed Master Command Request Frame */

        /* Check Command code */
        if ( LIN_ProtBuf[0] == LIN_COMMAND_SLEEP )
        {
            /* Go to Sleep Mode command */

            LIN_StateFlags = LIN_FLAG_IGNORE | LIN_FLAG_SLEEP;
        }

        /* ... any other commands ... */

        else
        {
            /* not specific command */
            LIN_StateFlags = LIN_FLAG_IGNORE;
        }

        LIN_Command();        /* user hook after any Command */
    }
    else

#endif /* !defined(LINAPI_1_0) */

    {   /* no any commands */

        LIN_StateFlags = LIN_FLAG_IGNORE;
    }
}


/***************************************************************************
 * Function :   LIN_TxCompleted
 *
 * Description: Tx interrupt 
 * 
 * Returns:     none
 *
 * Notes:       
 *
 **************************************************************************/

void LIN_TxCompleted( void )
{
    /**********************************************************************/
    /***       MASTER:   LIN_FLAG_SEND_BREAK                            ***/
    /**********************************************************************/
#if defined(MASTER)
    if (LIN_StateFlags & LIN_FLAG_SEND_BREAK)           
    {
        LIN_SetIdleTimeout();           /* valid break transmited -> set idle timeout */

        /* Set normal SCI baud rate,  restore after send break */
        LIN_SCISetBaudRate(LIN_CfgConst.LIN_BaudRate);

        /* Set MaxFrame timeout depended on the message size */
        LIN_SetTimeout(LIN_MsgLength);

        /* Send Synchronization field */
        LIN_SCDRPut(LIN_SYNC_FIELD);

        LIN_StateFlags = LIN_FLAG_SEND_HEADER;

        /* LIN_ProtBuf[1] and BufCounter has been prepared in LIN_RequestMsg() */
    } 
#endif /* defined(MASTER) */

    /**********************************************************************/
    /***       SLAVE:   LIN_FLAG_WAKEUP_TIMEOUT                         ***/
    /**********************************************************************/
#if defined(SLAVE)
    if (LIN_StateFlags & LIN_FLAG_WAKEUP_TIMEOUT)
    {
        LIN_SetIdleTimeout();                /* Set idle timeout again */

        LIN_StateFlags = LIN_FLAG_IGNORE;
    }
#endif /* defined(SLAVE) */

#if defined(LIN_DBG_CHECK_INTERNAL_ERROR)
   else
   {
        LIN_InternalError   = LIN_ERROR_7;
        while (1)
        {}
   }
#endif /* defined(LIN_DBG_CHECK_INTERNAL_ERROR) */
}

/***************************************************************************
 * Function :   LIN_RxCompleted
 *
 * Description: Rx interrupt 
 *              LIN_TmpSCIByte   = SCI Data Register 
 * 
 * Returns:     none
 *
 * Notes:       cases are in order of frequency of occuring
 *
 **************************************************************************/

void LIN_RxCompleted( void )
{
	LIN_BYTE        tmp;      /* temporary variable: counter or message direction flags */

    /**********************************************************************/
    /***       MASTER & SLAVE : LIN_FLAG_RECEIVE_DATA                   ***/
    /**********************************************************************/
    if (LIN_StateFlags & LIN_FLAG_RECEIVE_DATA )
    {   
        if (LIN_BufCounter < LIN_MsgLength)        /* NB: LIN_MsgLength = 1..8 */
        {   
            /* not all bytes received */

            /* receive message data */ 
            LIN_ProtBuf[LIN_BufCounter] = LIN_TmpSCIByte;

            ++LIN_BufCounter;

            /* calculate checksum from received byte */
            LIN_ChkAddByte(LIN_TmpSCIByte);
        }
        else
        {   /* all data and checksum has been received */
            
            if ( (LIN_ChkSum + LIN_TmpSCIByte) != 0xff )
            {   
                /* Checksum error  */
                LIN_Error();	        /* Rx error */
            }
            else
            {   /* no error */
                
                /* Message received succesfuly so copy it to proper place and update flags */

                LIN_CopyFromBuf(LinMsgBuf[LIN_SaveIdx]);

#if defined(LINAPI_1_0)
                /* set flags for all signals placed into received message */

            	for( tmp = LinFlagLookup[LIN_SaveIdx]; tmp < LinFlagLookup[LIN_SaveIdx + 1]; tmp++ )
	            {
		            LinSigFlags[tmp] = 0xFFu;
	            }

#else /* defined(LINAPI_1_0) */
                /* set appropriate status for received message */

                if (( LinMsgStatus[LIN_SaveIdx] & ((LIN_BYTE)LIN_MSG_OK | (LIN_BYTE)LIN_MSG_OVERRUN )) != 0 )
                {
                    LinMsgStatus[LIN_SaveIdx] = LIN_MSG_OVERRUN; 
                }
                else  
                {
                    LinMsgStatus[LIN_SaveIdx] = LIN_MSG_OK; 
                } 
#endif /* defined(LINAPI_1_0) */

                LIN_EndOfFrame();
            }
        }     
    }

    /**********************************************************************/
    /***       MASTER & SLAVE:   LIN_FLAG_SEND_DATA                     ***/
    /**********************************************************************/
    else if (LIN_StateFlags & LIN_FLAG_SEND_DATA)
    {
        if (LIN_BufCounter < LIN_MsgLength)
        {   
            /* not all bytes sended */

            if (LIN_ProtBuf[LIN_BufCounter] != LIN_TmpSCIByte )
            { 
                /* bit error in data */
                LIN_Error();		        /* Tx error */
            }
            else
            {   /* send next byte */

                ++LIN_BufCounter;

                if (LIN_BufCounter == LIN_MsgLength)
                {   
                    /* send check sum */
                    LIN_ChkSum = ~LIN_ChkSum;

                    LIN_SCDRPut(LIN_ChkSum);
                }
                else
                {
                    LIN_ChkAddByte(LIN_ProtBuf[LIN_BufCounter]);    /* OPTIM */

                    LIN_SCDRPut(LIN_ProtBuf[LIN_BufCounter]);
                }
            }
        }
        else
        {   /* all data has been transfered so check biterror in checksum  */

            if (LIN_ChkSum != LIN_TmpSCIByte )
            { 
                /* bit error in checksum */
                LIN_Error();		            /* Tx error */
            }
            else
            {
                /* Message transfered succesfuly so update flags -- for Freescale API only */

#if !defined(LINAPI_1_0)
                /* NB: LIN_MSG_UPDATED set only with LIN_MSG_NOCHANGE */
                
                if ( ( LinMsgStatus[LIN_SaveIdx] & LIN_MSG_UPDATED ) == 0 )
                {
                    /* msg has not been changed since start of transmission */
                    LinMsgStatus[LIN_SaveIdx] = LIN_MSG_OK;
                }

                /* if msg has been changed since start of transmission then we
                   keep current state (LIN_MSG_UPDATED | LIN_MSG_NOCHANGE)      */

#endif /* !defined(LINAPI_1_0) */

                /* For LIN API: message state is already LIN_MSG_OK -- for any type of message */

                LIN_EndOfFrame();
            }
        }
    }

    /**********************************************************************/
    /***       MASTER & SLAVE :   LIN_FLAG_SLEEP                ***/
    /**********************************************************************/ 
    else if (LIN_StateFlags & LIN_FLAG_SLEEP)
    {
        if ((LIN_TmpSCIByte == WAKEUP_BYTE) || (LIN_TmpSCIByte == 0xC0) || (LIN_TmpSCIByte == 0x00))
        {
            LIN_SetIdleTimeout();                /* Set idle timeout again */

            LIN_StateFlags = LIN_FLAG_IGNORE;

#if defined(MASTER)
            LIN_SetTimeout(LIN_CfgConst.LIN_to_wakeup_delimiter);                    
            LIN_StateFlags = LIN_FLAG_WAKEUP_TIMEOUT;
#endif /* defined(MASTER) */
        }  
    }


#if defined(MASTER)
    /**********************************************************************/
    /***       MASTER : LIN_FLAG_IGNORE_DATA                            ***/
    /**********************************************************************/
    else if ( LIN_StateFlags & LIN_FLAG_IGNORE_DATA )
    {
        ++LIN_BufCounter;
        
        if (LIN_BufCounter > LIN_MsgLength)         /* NB: LIN_MsgLength = 1..8 */
        {   
            LIN_EndOfFrame();
        }
    }

    
    /**********************************************************************/
    /***       MASTER:   LIN_FLAG_SEND_HEADER                           ***/
    /**********************************************************************/
    else if (LIN_StateFlags & LIN_FLAG_SEND_HEADER)
    {
        if ( LIN_BufCounter == 0 )
        {   
            /* after SYNC byte sended */
        
            if ( LIN_TmpSCIByte == LIN_SYNC_FIELD )
            {
                ++LIN_BufCounter;
                /* send ID byte */
                LIN_SCDRPut(LIN_ProtBuf[LIN_BufCounter]);
            }
            else if ( LIN_TmpSCIByte != 0 )
            { 
                /* bit error in SYNC byte */
                LIN_Error();                /* Tx error */
            }
            /* If LIN_TmpSCIByte == 0 it may be Rx interrupt after break,
               so we must ignore it */
        }
        else
        {
            /* after ID byte sended */

            if ( LIN_TmpSCIByte != LIN_ProtBuf[LIN_BufCounter] )
            { 
                /* bit error in ID byte */
                LIN_Error();                /* Tx error */
            }
            else
            {   
                /* header has been transfered so process message ID */
    
                LIN_DBG_SET_PORT_6;

                /* LIN_SaveIdentifier was set in LIN_RequestMsg() */

                /* We must count bytes in IGNORE_DATA state also,
                   so we always need LIN_BufCounter */
                LIN_BufCounter  = 0;

                /* LIN_MsgLength is already set -- in LIN_RequestMsg */

                /* Check what we should do with this message */
                tmp = LinLookupTable[LIN_SaveIdentifier];
                /* Now tmp contains ID and msg flags */

                LIN_SaveIdx = tmp & LIN_MSG_INDEX_MASK;

                if ( !(tmp & LIN_MSG_NOT_IGNORED) )
                {
                    /* We did not find this id */
                    LIN_StateFlags = LIN_FLAG_IGNORE_DATA;
                }
                else if ( tmp & LIN_MSG_SEND ) /* if we probably send the message (SEND or SEND_UPDATED) */
                {   
                    /* check do we really should send data */

                    if ( (   ( LinMsgStatus[LIN_SaveIdx] & LIN_MSG_NODATA  ) != 0      ) ||
                         ( ( ( tmp & LIN_MSG_SEND_UPDATED                  ) != 0 ) &&
                           ( ( LinMsgStatus[LIN_SaveIdx] & LIN_MSG_UPDATED ) == 0 )    )   )
                    {
                        /* the data is not initialized  or
                           we should send only updated data but the data has not updated */
                        /* -> we ignore this message and don't change msg state */
                        /* NB: Tmaxframe (Rx) error should occure */

                        LIN_StateFlags = LIN_FLAG_IGNORE_DATA;
                    }
                    else
                    {   /* we should send data for this message */

                        /* new state of the msg */
                        if ( ( tmp & LIN_MSG_SEND_UPDATED ) != 0 )
                        {
                            /* if we are sending only updated data then we should reset
                               updated state of msg at all */
                            LinMsgStatus[LIN_SaveIdx] = LIN_MSG_OK;
                        }
                        else
                        {     
                            /* if we are sending msg always then we should clear updated flag,
                               so LIN_MSG_NOCHANGE flag can be keep */
                            LinMsgStatus[LIN_SaveIdx] &= ~(LINStatusType)LIN_MSG_UPDATED;
                        }
                        
                        LIN_CopyToBuf(LinMsgBuf[LIN_SaveIdx]);
                
                        LIN_SCDRPut(LIN_ProtBuf[0]);   /* Send first byte of buffer */

                        LIN_StateFlags  = LIN_FLAG_SEND_DATA;
                        LIN_ChkSum      = LIN_ProtBuf[0];
                    }
                }
                else    
                {   /* we receive data from this message */

                    LIN_StateFlags = LIN_FLAG_RECEIVE_DATA;        
                    LIN_ChkSum = 0;
                }

                LIN_DBG_CLR_PORT_6;
            }
        }
    }
#endif /* defined(MASTER) */

    
#if defined(SLAVE)
    /**********************************************************************/
    /***       SLAVE:   LIN_FLAG_RECEIVE_SYNC                           ***/
    /**********************************************************************/
    else if (LIN_StateFlags & LIN_FLAG_RECEIVE_SYNC)
    {
        if (LIN_TmpSCIByte != LIN_SYNC_FIELD )
        {
            LIN_Error();		/* Rx */
        }
        else
        {
            LIN_StateFlags = LIN_FLAG_RECEIVE_ID;
        }
    }


    /**********************************************************************/
    /***       SLAVE:   LIN_FLAG_RECEIVE_ID                             ***/
    /**********************************************************************/
    else if (LIN_StateFlags & LIN_FLAG_RECEIVE_ID)
    {
        /* header has been received so process message ID */

        LIN_DBG_SET_PORT_6;

        /* Cut the parity bits off and search message */
        tmp = LinLookupTable[LIN_TmpSCIByte & LIN_MSG_INDEX_MASK];
        /* Now tmp contains ID and msg flags */

        LIN_SaveIdx = tmp & LIN_MSG_INDEX_MASK;
        
        if ( ( !(tmp & LIN_MSG_NOT_IGNORED) ) || ( LinMsgId[LIN_SaveIdx] != LIN_TmpSCIByte ) )
        {
            /* We did not find this id or parity did not match. */
            /* NB: we don't store information about Parity Error ! */
            LIN_EndOfFrame();
        }
        else 
        {   
            LIN_BufCounter  = 0;
            LIN_MsgLength = LinMsgLen[LIN_SaveIdx];

            if ( tmp & LIN_MSG_SEND ) /* if we probably send the message (SEND or SEND_UPDATED) */
            {   
                /* check do we really should send data */

                if ( (   ( LinMsgStatus[LIN_SaveIdx] & LIN_MSG_NODATA  ) != 0      ) ||
                     ( ( ( tmp & LIN_MSG_SEND_UPDATED                  ) != 0 ) &&
                       ( ( LinMsgStatus[LIN_SaveIdx] & LIN_MSG_UPDATED ) == 0 )    )   )
                {
                    /* the data is not initialized  or
                       we should send only updated data but the data has not updated */
                    /* -> we ignore this message and don't change msg state */
                    /* NB: Tmaxframe (Rx) error should occure */

                    LIN_EndOfFrame();
                }
                else
                {   /* we should send data for this message */

                    /* set new state of the msg */

#if defined(LINAPI_1_0)

                    /* for LIN API: any transmitted message always go to state LIN_MSG_OK */
                    LinMsgStatus[LIN_SaveIdx] = LIN_MSG_OK;

#else   /* !defined(LINAPI_1_0) */

                    /* for Freescale API: LIN_SEND and LIN_SEND_UPDATED messages processed differently */

                    if ( ( tmp & LIN_MSG_SEND_UPDATED ) != 0 )
                    {
                        /* if we are sending only updated data then we should reset
                           updated state of msg at all */
                        LinMsgStatus[LIN_SaveIdx] = LIN_MSG_OK;
                    }
                    else
                    {     
                        /* if we are sending msg always then we should clear updated flag,
                           so LIN_MSG_NOCHANGE flag can be keep */
                        LinMsgStatus[LIN_SaveIdx] &= ~(LINStatusType)LIN_MSG_UPDATED;
                    }

#endif  /* !defined(LINAPI_1_0) */

                    LIN_CopyToBuf(LinMsgBuf[LIN_SaveIdx]);
        
                    LIN_SCDRPut(LIN_ProtBuf[0]);   /* Send first byte of buffer */

                    LIN_StateFlags  = LIN_FLAG_SEND_DATA;
                    LIN_ChkSum      = LIN_ProtBuf[0];
                }
            }

            else    
            {   /* we receive data from this message */

                LIN_StateFlags = LIN_FLAG_RECEIVE_DATA;        
                LIN_ChkSum = 0;
            }
        }

        LIN_DBG_CLR_PORT_6;
    }
#endif /* defined(SLAVE) */

    /* Ignored: */
    /**********************************************************************/
    /***       MASTER & SLAVE : LIN_FLAG_IGNORE,                        ***/
    /***                        LIN_FLAG_SLEEP, LIN_FLAG_WAKEUP_TIMEOUT ***/
    /**********************************************************************/
    /***       MASTER  :        LIN_FLAG_SEND_BREAK                     ***/
    /***                        Wait for Tx completed interrupt.        ***/
    /***                        Rx interrupt ignored for S12 MASTER     ***/
    /**********************************************************************/
    /***       LIN_API :        LIN_FLAG_DISCONNECT, LIN_FLAG_NOINIT    ***/
    /**********************************************************************/
    /* NB: in Sleep mode LIN_FLAG_IGNORE and LIN_FLAG_SLEEP flags are set */
    /* For LIN API -- no LIN_FLAG_SLEEP                                   */
    /**********************************************************************/
}


/***************************************************************************
 * Function :   LIN_FrameError
 *
 * Description: Frame Error interrupt 
 * 
 * Returns:     none
 *
 * Notes:       MASTER: no parameters
 *              SLAVE:  input parameter
 *                      LIN_FRAMEERROR  -- frame error like break;
 *                      LIN_NORMALBREAK -- normal break received.
 *              Slave should ignore break in sleep mode
 **************************************************************************/

#if defined(MASTER)
void LIN_FrameError( void )
{
    if (LIN_StateFlags & ( LIN_FLAG_SEND_HEADER | LIN_FLAG_SEND_DATA | 
                           LIN_FLAG_RECEIVE_DATA | LIN_FLAG_IGNORE_DATA ))
    {
        LIN_Error();        /* set Tx or Rx error for previous message */
    }
    /* in all other states -- ignoring */
}
#endif /* defined(MASTER) */

#if defined(SLAVE)
void LIN_FrameError( LIN_BYTE breakType )
{
        LIN_SetIdleTimeout();           /* break received -> set idle timeout */

        if (LIN_StateFlags & ( LIN_FLAG_RECEIVE_SYNC | LIN_FLAG_RECEIVE_ID | 
                               LIN_FLAG_RECEIVE_DATA | LIN_FLAG_SEND_DATA ))
        {
    
            if ( ( LIN_StateFlags & (LIN_FLAG_RECEIVE_SYNC | LIN_FLAG_RECEIVE_ID) ) && 
                 ( breakType == LIN_FRAMEERROR ) )
            {   /* we receiving message header and frame error (not normal break!) occure */

                LIN_Error();            /* set Rx error for previous message and goto IGNORE */
                return;                 /* ... and stay in IGNORE state */
            }

            LIN_Error();                /* set Tx or Rx error for previous message and goto IGNORE */
        }    

        /* IGNORE or WAKEUP states -> goto RECEIVE_55 */
        LIN_StateFlags = LIN_FLAG_RECEIVE_SYNC;
}
#endif /* defined(SLAVE) */


#if defined(MASTER)
/***************************************************************************
 * Function :   LIN_Timeout
 *
 * Description: Timeout interrupt 
 * 
 * Returns:     none
 *
 * Notes:       only for Master node
 *
 **************************************************************************/
void LIN_Timeout( void )
{
    if ( LIN_StateFlags & ( LIN_FLAG_SEND_HEADER | LIN_FLAG_SEND_DATA | 
                            LIN_FLAG_RECEIVE_DATA | LIN_FLAG_IGNORE_DATA))
    {
        LIN_Error();        /* Rx or Tx error */
    }

    else if (LIN_StateFlags & LIN_FLAG_WAKEUP_TIMEOUT )
    {
                            /* Wakeup delimiter timeout expired */
        LIN_StateFlags = LIN_FLAG_IGNORE;
    } 

#if defined(LIN_DBG_CHECK_INTERNAL_ERROR)
    else
    {
        /* for S12 while SEND_BREAK it is unreachable code. Driver waits for tx completed interrupt in this state */
        LIN_InternalError   = LIN_ERROR_6;
        while (1)
        {}
    }
#endif /* defined(LIN_DBG_CHECK_INTERNAL_ERROR) */

}
#endif /* defined(MASTER) */


