/*
 * The Clear BSD License
 * Copyright (c) 2017, NXP Semiconductor, Inc.
 * All rights reserved.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer below) provided
 * that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of copyright holder nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * NO EXPRESS OR IMPLIED LICENSES TO ANY PARTY'S PATENT RIGHTS ARE GRANTED BY THIS LICENSE.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 * usb_host_test.c
 *
 */

#include "main.h"
#include "host_mouse.h"

#define MSG_STRING_LEN 80
static char msgString[MSG_STRING_LEN];

extern bool done;

/*!
 * @brief host callback function.
 *
 * device attach/detach callback function.
 *
 * @param deviceHandle          device handle.
 * @param configurationHandle   attached device's configuration descriptor information.
 * @param eventCode             callback event code, please reference to enumeration host_event_t.
 *
 * @retval kStatus_USB_Success              The host is initialized successfully.
 * @retval kStatus_USB_NotSupported         The application don't support the configuration.
 */
static usb_status_t USB_HostEvent(usb_device_handle deviceHandle,
		usb_host_configuration_handle configurationHandle,
		uint32_t eventCode);

/*!
 * @brief application initialization.
 */
static void USB_HostApplicationInit(void);

extern void USB_HostClockInit(void);
extern void USB_HostIsrEnable(void);
extern void USB_HostTaskFn(void *param);
void BOARD_InitHardware(void);

/*******************************************************************************
 * Variables
 ******************************************************************************/
/*! @brief USB host mouse instance global variable */
extern usb_host_mouse_instance_t g_HostHidMouse;
usb_host_handle g_HostHandle;

/*******************************************************************************
 * Code
 ******************************************************************************/
#if (defined(USB_HOST_CONFIG_OHCI) && (USB_HOST_CONFIG_OHCI > 0U))
void USB0_IRQHandler(void)
{
	USB_HostOhciIsrFunction(g_HostHandle);
}
#endif /* USB_HOST_CONFIG_OHCI */
#if (defined(USB_HOST_CONFIG_IP3516HS) && (USB_HOST_CONFIG_IP3516HS > 0U))
void USB1_IRQHandler(void)
{
	USB_HostIp3516HsIsrFunction(g_HostHandle);
}
#endif /* USB_HOST_CONFIG_IP3516HS */

void USB_HostClockInit(void)
{
#if ((defined USB_HOST_CONFIG_OHCI) && (USB_HOST_CONFIG_OHCI > 0U))
	CLOCK_EnableUsbfs0HostClock(kCLOCK_UsbSrcFro, CLOCK_GetFroHfFreq());
#if ((defined FSL_FEATURE_USBFSH_USB_RAM) && (FSL_FEATURE_USBFSH_USB_RAM > 0U))
	for (int i = 0; i < (FSL_FEATURE_USBFSH_USB_RAM >> 2); i++)
	{
		((uint32_t *)FSL_FEATURE_USBFSH_USB_RAM_BASE_ADDRESS)[i] = 0U;
	}
#endif
#endif /* USB_HOST_CONFIG_OHCI */

#if ((defined USB_HOST_CONFIG_IP3516HS) && (USB_HOST_CONFIG_IP3516HS > 0U))
	CLOCK_EnableUsbhs0HostClock(kCLOCK_UsbSrcUsbPll, 48000000U);
#if ((defined FSL_FEATURE_USBHSH_USB_RAM) && (FSL_FEATURE_USBHSH_USB_RAM > 0U))
	for (int i = 0; i < (FSL_FEATURE_USBHSH_USB_RAM >> 2); i++)
	{
		((uint32_t *)FSL_FEATURE_USBHSH_USB_RAM_BASE_ADDRESS)[i] = 0U;
	}
#endif
#endif /* USB_HOST_CONFIG_IP3511HS */
}

void USB_HostIsrEnable(void)
{
	uint8_t irqNumber;
#if ((defined USB_HOST_CONFIG_OHCI) && (USB_HOST_CONFIG_OHCI > 0U))
	IRQn_Type usbHsIrqs[] = {(IRQn_Type)USB0_IRQn};
	irqNumber = usbHsIrqs[CONTROLLER_ID - kUSB_ControllerOhci0];
#endif /* USB_HOST_CONFIG_OHCI */
#if ((defined USB_HOST_CONFIG_IP3516HS) && (USB_HOST_CONFIG_IP3516HS > 0U))
	IRQn_Type usbHsIrqs[] = {(IRQn_Type)USB1_IRQn};
	irqNumber = usbHsIrqs[CONTROLLER_ID - kUSB_ControllerIp3516Hs0];
#endif /* USB_HOST_CONFIG_IP3511HS */

	/* Install isr, set priority, and enable IRQ. */
#if defined(__GIC_PRIO_BITS)
	GIC_SetPriority((IRQn_Type)irqNumber, USB_HOST_INTERRUPT_PRIORITY);
#else
	NVIC_SetPriority((IRQn_Type)irqNumber, USB_HOST_INTERRUPT_PRIORITY);
#endif
	EnableIRQ((IRQn_Type)irqNumber);
}

void USB_HostTaskFn(void *param)
{
#if ((defined USB_HOST_CONFIG_OHCI) && (USB_HOST_CONFIG_OHCI > 0U))
	USB_HostOhciTaskFunction(param);
#endif /* USB_HOST_CONFIG_OHCI */
#if ((defined USB_HOST_CONFIG_IP3516HS) && (USB_HOST_CONFIG_IP3516HS > 0U))
	USB_HostIp3516HsTaskFunction(param);
#endif /* USB_HOST_CONFIG_IP3516HS */
}

/*!
 * @brief USB isr function.
 */

static usb_status_t USB_HostEvent(usb_device_handle deviceHandle,
		usb_host_configuration_handle configurationHandle,
		uint32_t eventCode)
{
	usb_status_t status = kStatus_USB_Success;

	switch (eventCode)
	{
	case kUSB_HostEventAttach:
		status = USB_HostHidMouseEvent(deviceHandle, configurationHandle, eventCode);
		break;

	case kUSB_HostEventNotSupported:
		usb_echo("device not supported.\r\n");
		break;

	case kUSB_HostEventEnumerationDone:
		status = USB_HostHidMouseEvent(deviceHandle, configurationHandle, eventCode);
		break;

	case kUSB_HostEventDetach:
		status = USB_HostHidMouseEvent(deviceHandle, configurationHandle, eventCode);
		break;

	default:
		break;
	}
	return status;
}

static void USB_HostApplicationInit(void)
{
	uint32_t usbHostVersion;
	usb_status_t status = kStatus_USB_Success;

	USB_HostClockInit();

#if ((defined FSL_FEATURE_SOC_SYSMPU_COUNT) && (FSL_FEATURE_SOC_SYSMPU_COUNT))
	SYSMPU_Enable(SYSMPU, 0);
#endif /* FSL_FEATURE_SOC_SYSMPU_COUNT */

	status = USB_HostInit(CONTROLLER_ID, &g_HostHandle, USB_HostEvent);
	if (status != kStatus_USB_Success)
	{
		snprintf(msgString,MSG_STRING_LEN,"\nhost init error\r\n");
		GUI_DispString(msgString);
		usb_echo(msgString);
		return;
	}
	USB_HostIsrEnable();

	USB_HostGetVersion(&usbHostVersion);

	snprintf(msgString,MSG_STRING_LEN,"\nhost init done, the host stack version is %d.%d.%d.\r\n", ((uint8_t)(usbHostVersion >> 16)),
			((uint8_t)(usbHostVersion >> 8)), ((uint8_t)(usbHostVersion)));
	GUI_DispString(msgString);
	usb_echo(msgString);

}

void usb_host_test_cmd (uint32_t argc, char *argv[])
{
	GUI_Clear();

	snprintf(msgString,MSG_STRING_LEN,"\r\nLPC54S018M USB (HS) Host Test\r\n");
	GUI_DispString(msgString);
	usb_echo(msgString);
	snprintf(msgString,MSG_STRING_LEN,"Make sure Jumper JP9 is set to 1-2\r\n");
	GUI_DispString(msgString);
	usb_echo(msgString);
	snprintf(msgString,MSG_STRING_LEN,"Make sure Jumpers JP11-12-13 are set to 2-3\r\n");
	GUI_DispString(msgString);
	usb_echo(msgString);
	snprintf(msgString,MSG_STRING_LEN,"Connect a USB mouse to HS USB connector (J2)\r\n");
	GUI_DispString(msgString);
	usb_echo(msgString);
	snprintf(msgString,MSG_STRING_LEN,"and USB power connected to USB +5V connector (J1)\r\n");
	GUI_DispString(msgString);
	usb_echo(msgString);
	snprintf(msgString,MSG_STRING_LEN,"Mouse connection status, coordinates and buttons will be reported.\r\n");
	GUI_DispString(msgString);
	usb_echo(msgString);
	snprintf(msgString,MSG_STRING_LEN,"\r\nPress User button to end test and go to Self-Test main menu\r\n");
	GUI_DispString(msgString);
	usb_echo(msgString);

	USB_HostApplicationInit();

	while (1)
	{
		USB_HostTaskFn(g_HostHandle);
		USB_HostHidMouseTask(&g_HostHidMouse);

		if (done)
			break;
	}

	done = false;
	GUI_Display_Prompt();
	cli_display_prompt();

}

