/*
 * The Clear BSD License
 * Copyright (c) 2017, NXP Semiconductor, Inc.
 * All rights reserved.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer below) provided
 * that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of copyright holder nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * NO EXPRESS OR IMPLIED LICENSES TO ANY PARTY'S PATENT RIGHTS ARE GRANTED BY THIS LICENSE.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 * sdram_test.c
 *
 */

#include "main.h"
#include "emwin_support.h"
extern bool done;
extern status_t APP_LCDC_Init(void);

#define SDRAM_BASE_ADDR 0xa0000000
#define SDRAM_SIZE_BYTES (8 * 1024 * 1024)
#define SDRAM_EXAMPLE_DATALEN (SDRAM_SIZE_BYTES / 4)
#define SDRAM_TEST_PATTERN (2)

status_t SDRAM_DataBusCheck(volatile uint32_t *address)
{
    uint32_t data = 0;

    /* Write the walking 1's data test. */
    for (data = 1; data != 0; data <<= 1)
    {
        *address = data;

        /* Read the data out of the address and check. */
        if (*address != data)
        {
            return kStatus_Fail;
        }
    }
    return kStatus_Success;
}

status_t SDRAM_AddressBusCheck(volatile uint32_t *address, uint32_t bytes)
{
    uint32_t pattern = 0x55555555;
    uint32_t size = bytes / 4;
    uint32_t offset;
    uint32_t checkOffset;

    /* write the pattern to the power-of-two address. */
    for (offset = 1; offset < size; offset <<= 1)
    {
        address[offset] = pattern;
    }
    address[0] = ~pattern;

    /* Read and check. */
    for (offset = 1; offset < size; offset <<= 1)
    {
        if (address[offset] != pattern)
        {
            return kStatus_Fail;
        }
    }

    if (address[0] != ~pattern)
    {
        return kStatus_Fail;
    }

    /* Change the data to the revert one address each time
     * and check there is no effect to other address. */
    for (offset = 1; offset < size; offset <<= 1)
    {
        address[offset] = ~pattern;
        for (checkOffset = 1; checkOffset < size; checkOffset <<= 1)
        {
            if ((checkOffset != offset) && (address[checkOffset] != pattern))
            {
                return kStatus_Fail;
            }
        }
        address[offset] = pattern;
    }
    return kStatus_Success;
}

void config_pins_for_emc (void)
{
	// EMC pins may have been set for ENET, so config them for EMC use
    IOCON->PIO[1][23] = ((IOCON->PIO[1][23] &
                          /* Mask bits to zero which are setting */
                          (~(IOCON_PIO_FUNC_MASK | IOCON_PIO_DIGIMODE_MASK)))

                         /* Selects pin function.
                          * : PORT123 (pin M10) is configured as EMC_A(11). */
                         | IOCON_PIO_FUNC(PIO123_FUNC_ALT6)

                         /* Select Analog/Digital mode.
                          * : Digital mode. */
                         | IOCON_PIO_DIGIMODE(PIO123_DIGIMODE_DIGITAL));

    IOCON->PIO[1][16] = ((IOCON->PIO[1][16] &
                          /* Mask bits to zero which are setting */
                          (~(IOCON_PIO_FUNC_MASK | IOCON_PIO_DIGIMODE_MASK)))

                         /* Selects pin function.
                          * : PORT116 (pin B7) is configured as EMC_A(10). */
                         | IOCON_PIO_FUNC(PIO116_FUNC_ALT6)

                         /* Select Analog/Digital mode.
                          * : Digital mode. */
                         | IOCON_PIO_DIGIMODE(PIO116_DIGIMODE_DIGITAL));

    // Turn off P1_1 pull up
    IOCON->PIO[1][1] = ((IOCON->PIO[1][1] &
                          /* Mask bits to zero which are setting */
                          (~(IOCON_PIO_FUNC_MASK | IOCON_PIO_DIGIMODE_MASK)))
 					    | IOCON_PIO_MODE(0x0)                                   /* Turn off p1_1 pull-up function */
    );

}

void sdram_test_cmd (uint32_t argc, char *argv[])
{
    uint32_t index;
    uint32_t *sdram = (uint32_t *)SDRAM_BASE_ADDR; /* SDRAM start address. */
    uint8_t test_fail = 0;

    BOARD_InitPWM(0);
    config_pins_for_emc();

    PRINTF("\r\n SDRAM Test starting...\r\n");
    PRINTF("\r\n JP14 and JP15 must be in SD position (JP14 2-3, JP15 2-3) to pass\r\n");

    if (SDRAM_DataBusCheck(sdram) != kStatus_Success)
    {
        PRINTF(" SDRAM data bus check failure.\r\n");
        test_fail = 1;
    }

    if (SDRAM_AddressBusCheck(sdram, SDRAM_SIZE_BYTES) != kStatus_Success)
    {
        PRINTF(" SDRAM address bus check failure.\r\n");
        PRINTF(" Check JP14 and JP15.\r\n");
        test_fail = 1;
    }

    PRINTF("\r\n SDRAM Write Start, Start Address 0x%x, Data Length %d !\r\n", sdram, SDRAM_EXAMPLE_DATALEN);
    /* Prepare data and write to SDRAM. */
    for (index = 0; index < SDRAM_EXAMPLE_DATALEN; index++)
    {
        *(uint32_t *)(sdram + index) = index;
    }
    PRINTF(" SDRAM Write finished\r\n");

    PRINTF(" SDRAM Read/Check Start, Start Address 0x%x, Data Length %d !\r\n", sdram, SDRAM_EXAMPLE_DATALEN);
    /* Read data from the SDRAM. */
    for (index = 0; index < SDRAM_EXAMPLE_DATALEN; index++)
    {
        if (*(uint32_t *)(sdram + index) != index)
        {
            PRINTF(" SDRAM Write Data and Read Data Check Error!\r\n");
            test_fail = 1;
            break;
        }
    }

    if (test_fail == 0)
    {
    	PRINTF("\r\n SDRAM Write Data and Read Data TEST PASSED.\r\n");
    }
    else
    {
    	PRINTF("\r\n SDRAM Write Data and Read Data TEST FAIL.\r\n");
    }

    PRINTF("\r\n SDRAM test complete.\r\n");

    done = false;
    GUI_Display_Prompt();
    BOARD_InitPWM(5);
    cli_display_prompt();
}
