/*
 * The Clear BSD License
 * Copyright (c) 2017, NXP Semiconductor, Inc.
 * All rights reserved.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer below) provided
 * that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of copyright holder nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * NO EXPRESS OR IMPLIED LICENSES TO ANY PARTY'S PATENT RIGHTS ARE GRANTED BY THIS LICENSE.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 * rtc_test.c
 *
 */

#include "main.h"

#define MSG_STRING_LEN 80
static char msgString[MSG_STRING_LEN];

extern bool done;
volatile bool busyWait;

/*!
* @brief ISR for Alarm interrupt
*
* This function changes the state of busyWait.
*/
void RTC_IRQHandler(void)
{
    if (RTC_GetStatusFlags(RTC) & kRTC_AlarmFlag)
    {
        busyWait = false;

        /* Clear alarm flag */
        RTC_ClearStatusFlags(RTC, kRTC_AlarmFlag);
    }
    /* Add for ARM errata 838869, affects Cortex-M4, Cortex-M4F Store immediate overlapping
      exception return operation might vector to incorrect interrupt */
#if defined __CORTEX_M && (__CORTEX_M == 4U)
    __DSB();
#endif
}

void rtc_test_cmd (uint32_t argc, char *argv[])
{
	uint32_t sec;
	uint32_t currSeconds;
	uint8_t index;
	rtc_datetime_t date;

    GUI_Clear();

    snprintf(msgString,MSG_STRING_LEN,"\r\nLPC54018 RTC Test\r\n");
    GUI_DispString(msgString);
    usb_echo(msgString);


    /* Set a start date time and start RT */
    date.year = 2014U;
    date.month = 12U;
    date.day = 25U;
    date.hour = 19U;
    date.minute = 0;
    date.second = 0;

    /* RTC time counter has to be stopped before setting the date & time in the TSR register */
    RTC_StopTimer(RTC);

    /* Set RTC time to default */
    RTC_SetDatetime(RTC, &date);

    /* Enable RTC alarm interrupt */
    RTC_EnableInterrupts(RTC, kRTC_AlarmInterruptEnable);

    /* Enable at the NVIC */
    EnableIRQ(RTC_IRQn);

    /* Start the RTC time counter */
    RTC_StartTimer(RTC);

    /* This loop will set the RTC alarm */
    while (1)
    {
        busyWait = true;
        index = 0;
        sec = 0;
        /* Get date time */
        RTC_GetDatetime(RTC, &date);

        /* print default time */
        snprintf(msgString,MSG_STRING_LEN,"Current datetime: %04d-%02d-%02d %02d:%02d:%02d\r",
               date.year,
               date.month,
               date.day,
               date.hour,
               date.minute,
               date.second);
        GUI_DispString(msgString);
        usb_echo(msgString);


        /* Get alarm time from user */
        snprintf(msgString,MSG_STRING_LEN,"Please input the number of second to wait for alarm \r\n");
        GUI_DispString(msgString);
        usb_echo(msgString);

        snprintf(msgString,MSG_STRING_LEN,"The second must be positive value\r\n");
        GUI_DispString(msgString);
        usb_echo(msgString);

        while (index != 0x0D)
        {
            index = GETCHAR();
            if((index >= '0') && (index <= '9'))
            {
//                PUTCHAR(index);
                sec = sec * 10 + (index - 0x30U);
            }
        }
        PRINTF("\r\n");
        snprintf(msgString,MSG_STRING_LEN,"Wait %d seconds\r\n",sec);
        GUI_DispString(msgString);

        /* Read the RTC seconds register to get current time in seconds */
        currSeconds = RTC->COUNT;

        /* Add alarm seconds to current time */
        currSeconds += sec;

        /* Set alarm time in seconds */
        RTC->MATCH = currSeconds;

        /* Get alarm time */
        RTC_GetAlarm(RTC, &date);

        /* Print alarm time */
        snprintf(msgString,MSG_STRING_LEN,"Alarm will occur at: %04d-%02d-%02d %02d:%02d:%02d\r\n",
               date.year,
               date.month,
               date.day,
               date.hour,
               date.minute,
               date.second);
        GUI_DispString(msgString);
        usb_echo(msgString);

        /* Wait until alarm occurs */
        while (busyWait)
        {
        }
        snprintf(msgString,MSG_STRING_LEN,"\r\n Alarm occurs !!!! ");
        GUI_DispString(msgString);
        usb_echo(msgString);
       	break;
    }
    done = false;
    GUI_Display_Prompt();
    cli_display_prompt();
}

