/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of the copyright holder nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*  Standard C Included Files */
#include "main.h"
#include "host_mouse.h"
/*******************************************************************************
 * Definitions
 ******************************************************************************/


extern void level_test_cmd (uint32_t argc, char *argv[]);
extern  void sdcard_test_cmd (uint32_t argc, char *argv[]);
extern void sdram_test_cmd (uint32_t argc, char *argv[]);
extern void led_button_test_cmd (uint32_t argc, char *argv[]);
extern void ether_test_cmd (uint32_t argc, char *argv[]);
extern void lcd_touch_test_cmd (uint32_t argc, char *argv[]);
extern void spifi_test_cmd (uint32_t argc, char *argv[]);
extern void usb_host_test_cmd (uint32_t argc, char *argv[]);
extern void rtc_test_cmd (uint32_t argc, char *argv[]);
extern void usb_device_audio_test_cmd (uint32_t argc, char *argv[]);

/*
 * directory record
 */
extern cli_record_t cli_root_element;

/*
 * allocate command records
 */
static cli_record_t level_test_record;
static cli_record_t sdram_test_record;
static cli_record_t ether_test_record;
static cli_record_t sdcard_test_record;
static cli_record_t led_button_test_record;
static cli_record_t lcd_touch_test_record;
static cli_record_t spifi_test_record;
static cli_record_t usb_host_test_record;
static cli_record_t rtc_test_record;
static cli_record_t usb_device_audio_test_record;

#define DEMO_PINT_PIN_INT0_SRC kINPUTMUX_GpioPort0Pin6ToPintsel
#define DEMO_PINT_PIN_INT1_SRC kINPUTMUX_GpioPort0Pin5ToPintsel
#define DEMO_PINT_PIN_INT2_SRC kINPUTMUX_GpioPort0Pin4ToPintsel
#define DEMO_PINT_PIN_INT3_SRC kINPUTMUX_GpioPort1Pin1ToPintsel

#if (defined(__CC_ARM) || defined(__GNUC__))
__attribute__((aligned(8)))
#elif defined(__ICCARM__)
#pragma data_alignment = 8
#else
#error Toolchain not support.
#endif


#if (defined(__CC_ARM) || defined(__GNUC__))
__attribute__((aligned(8)))
#elif defined(__ICCARM__)
#pragma data_alignment = 8
#else
#error Toolchain not support.
#endif

__attribute__((aligned(8)))


volatile bool g_MasterCompletionFlag = false;

bool done = false;

/*******************************************************************************
 * Prototypes
 ******************************************************************************/


/*******************************************************************************
 * Variables
 ******************************************************************************/

void BOARD_InitPWM(uint8_t dutyCycle)
{
    sctimer_config_t config;
    sctimer_pwm_signal_param_t pwmParam;
    uint32_t event;

    CLOCK_AttachClk(kMAIN_CLK_to_SCT_CLK);

    CLOCK_SetClkDiv(kCLOCK_DivSctClk, 2, true);

    SCTIMER_GetDefaultConfig(&config);

    SCTIMER_Init(SCT0, &config);

    pwmParam.output = kSCTIMER_Out_5;
    pwmParam.level = kSCTIMER_HighTrue;
    pwmParam.dutyCyclePercent = dutyCycle;

    SCTIMER_SetupPwm(SCT0, &pwmParam, kSCTIMER_CenterAlignedPwm, 1000U, CLOCK_GetFreq(kCLOCK_Sct), &event);
}

void BOARD_InitHardware(void)
{
    gpio_pin_config_t config;

	CLOCK_EnableClock(kCLOCK_InputMux);

	/* attach 12 MHz clock to FLEXCOMM0 (debug console) */
    CLOCK_AttachClk(BOARD_DEBUG_UART_CLK_ATTACH);

    /* Route Main clock to LCD. */
    CLOCK_AttachClk(kMAIN_CLK_to_LCD_CLK);

    /* attach 12 MHz clock to FLEXCOMM2 (I2C master for accelerometer) */
    CLOCK_AttachClk(kFRO12M_to_FLEXCOMM2);

    /* attach 12 MHz clock to SPI3 */
    CLOCK_AttachClk(kFRO12M_to_FLEXCOMM9);

    /* attach main clock to SDIF */
    CLOCK_AttachClk(BOARD_SDIF_CLK_ATTACH);
    CLOCK_SetClkDiv(kCLOCK_DivLcdClk, 1, true);

    CLOCK_EnableClock(kCLOCK_Iocon);
    CLOCK_EnableClock(kCLOCK_Gpio2);

    /* Enable the RTC 32K Oscillator */
    SYSCON->RTCOSCCTRL |= SYSCON_RTCOSCCTRL_EN_MASK;

    /* If this case runs in RAM, the debuger reset must
       be selected as Core in case not resetting SRAM after downloading,
       so the peripherals used will also not be reset by IDE tool.
       In this case these peripherals used should be reset manually by software
    */
    RESET_PeripheralReset(kSPIFI_RST_SHIFT_RSTn);

    /* reset USB0 and USB1 device */
    RESET_PeripheralReset(kUSB0D_RST_SHIFT_RSTn);
    RESET_PeripheralReset(kUSB1D_RST_SHIFT_RSTn);
    RESET_PeripheralReset(kUSB0HMR_RST_SHIFT_RSTn);
    RESET_PeripheralReset(kUSB0HSL_RST_SHIFT_RSTn);
    RESET_PeripheralReset(kUSB1H_RST_SHIFT_RSTn);

    NVIC_ClearPendingIRQ(USB0_IRQn);
    NVIC_ClearPendingIRQ(USB0_NEEDCLK_IRQn);
    NVIC_ClearPendingIRQ(USB1_IRQn);
    NVIC_ClearPendingIRQ(USB1_NEEDCLK_IRQn);

    BOARD_InitPins();
    BOARD_BootClockFROHF96M();
    /* need call this function to clear the halt bit in clock divider register */
    CLOCK_SetClkDiv(kCLOCK_DivSdioClk, (uint32_t)(SystemCoreClock / FSL_FEATURE_SDIF_MAX_SOURCE_CLOCK + 1U), true);
//    BOARD_BootClockPLL180M();
    BOARD_InitDebugConsole();

    BOARD_InitSDRAM();
    /* Set the back light PWM. */
    BOARD_InitPWM(5);

    POWER_DisablePD(kPDRUNCFG_PD_USB0_PHY); /*< Turn on USB Phy */
    POWER_DisablePD(kPDRUNCFG_PD_USB1_PHY); /*< Turn on USB Phy */

    CLOCK_SetClkDiv(kCLOCK_DivUsb0Clk, 1, false);
    CLOCK_AttachClk(kFRO_HF_to_USB0_CLK);
    /* enable usb0 host clock */
    CLOCK_EnableClock(kCLOCK_Usbhsl0);
    /*According to reference mannual, device mode setting has to be set by access usb host register */
    *((uint32_t *)(USBFSH_BASE + 0x5C)) |= USBFSH_PORTMODE_DEV_ENABLE_MASK;
    /* disable usb0 host clock */
    CLOCK_DisableClock(kCLOCK_Usbhsl0);
    /* enable USB IP clock */
    CLOCK_EnableUsbfs0DeviceClock(kCLOCK_UsbSrcFro, CLOCK_GetFreq(kCLOCK_FroHf));

    CLOCK_EnableClock(kCLOCK_Gpio4);
    /* Enable USB Host VBUS */
    config.outputLogic = 0;
    config.pinDirection = kGPIO_DigitalOutput;

    GPIO_PinInit(GPIO, 4, 9, &config);

    /* Init RTC */
    RTC_Init(RTC);

    /* enable USB IP clock */
    CLOCK_EnableUsbfs0DeviceClock(kCLOCK_UsbSrcFro, CLOCK_GetFreq(kCLOCK_FroHf));
    SystemCoreClockUpdate();
    SysTick_Init();
}

void GUI_Display_Help()
{
	GUI_Clear();
	GUI_DispString("LPC54S018M-EVK Self-Test Application commands \r\n");
	GUI_DispString("    h          print this help \r\n");
	GUI_DispString("    level      Spirit level test - Test the Accelerometer\r\n");
	GUI_DispString("    sdram      Test SDRAM\r\n");
	GUI_DispString("    sdcard     Test SD Card\r\n");
	GUI_DispString("    lcd        Test the LCD and Touch controller\r\n");
	GUI_DispString("    led        Test ISP buttons and user leds\r\n");
    GUI_DispString("    ether      Test the Ethernet PHY\r\n");
    GUI_DispString("    spifi      Test the QSPI/SPIFI\r\n");
    GUI_DispString("    usb        Test USB as Host on HS\r\n");
    GUI_DispString("    audio      Test audio lines and codec as well as USB device on FS\r\n");
    GUI_DispString("    rtc        Test the RTC\r\n");
    GUI_DispString("\r\nAll commands followed by <enter> \r\n");
	GUI_DispString("\r\nPress User button to end test and go to Self-Test main menu\r\n");
}

void GUI_Display_Prompt()
{
	GUI_Clear();
	GUI_DispString("LPC54S018M-EVK Self-Test Application \r\n");
	GUI_DispString("    h          to print list of commands \r\n");
    GUI_DispString("\r\nAll commands followed by <enter> \r\n");
}

void pint_intr_callback(pint_pin_int_t pintr, uint32_t pmatch_status)
{
//    PRINTF("\f\r\nPINT Pin Interrupt %d event detected.", pintr);
    if (pintr == 0)
    	done = true;
}

/*
 * User button is used to end some tests.
 * This function sets up the button in case other tests have altered settings
 */
void SetUp_UserButton(void)
{
    IOCON->PIO[1][1] = ((IOCON->PIO[1][1] &
                         /* Mask bits to zero which are setting */
                         (~(IOCON_PIO_FUNC_MASK | IOCON_PIO_DIGIMODE_MASK)))

                        /* Selects pin function.
                         * : PORT11 (pin K12) is configured as PIO1_1. */
                        | IOCON_PIO_FUNC(PIO11_FUNC_ALT0)

					    | IOCON_PIO_MODE(0x2)                                   /* Selects pull-up function */

                        /* Select Analog/Digital mode.
                         * : Digital mode. */
                        | IOCON_PIO_DIGIMODE(PIO11_DIGIMODE_DIGITAL));

}

int main(void)
{
    /* Initialize Board pins, clock and debug console unit*/
    BOARD_InitHardware();

    /* emWin start */
    GUI_Init();

  	WM_SetSize(WM_HBKWIN, LCD_WIDTH, LCD_HEIGHT);

    /* Solid color display */
    GUI_SetBkColor(GUI_WHITE);
    GUI_Clear();

    WM_Exec();

    GUI_SetColor(GUI_BLACK);

    GUI_SetFont(GUI_FONT_8X16);

#define MAXMESG  ( 200U )
    RC_CLI_t rc;

    cli_init_data_t init_data;
    unsigned char input[MAXMESG];

    strcpy(init_data.prefix, "Enter test name to run");
    cli_init(&init_data);


    rc = cli_mkcmd("level", level_test_cmd, &cli_root_element, &level_test_record);
    rc = cli_mkcmd("sdram", sdram_test_cmd, &cli_root_element, &sdram_test_record);
    rc = cli_mkcmd("ether", ether_test_cmd, &cli_root_element, &ether_test_record);
    rc = cli_mkcmd("sdcard", sdcard_test_cmd, &cli_root_element, &sdcard_test_record);
    rc = cli_mkcmd("led", led_button_test_cmd, &cli_root_element, &led_button_test_record);
    rc = cli_mkcmd("lcd", lcd_touch_test_cmd, &cli_root_element, &lcd_touch_test_record);
    rc = cli_mkcmd("spifi", spifi_test_cmd, &cli_root_element, &spifi_test_record);
    rc = cli_mkcmd("usb", usb_host_test_cmd, &cli_root_element, &usb_host_test_record);
    rc = cli_mkcmd("rtc", rtc_test_cmd, &cli_root_element, &rtc_test_record);
    rc = cli_mkcmd("audio", usb_device_audio_test_cmd, &cli_root_element, &usb_device_audio_test_record);

     /*
      * make sure we can tolerate a null string
      */
     cli_engine("");

     ///////////////////////////////////////////////////////////////////
//     PRINTF("\f\r\nPINT Pin interrupt example\r\n");

 #if defined(FSL_FEATURE_SYSCON_HAS_PINT_SEL_REGISTER) && FSL_FEATURE_SYSCON_HAS_PINT_SEL_REGISTER
     /* Connect trigger sources to PINT */
     SYSCON_AttachSignal(SYSCON, kPINT_PinInt0, DEMO_PINT_PIN_INT0_SRC);
     SYSCON_AttachSignal(SYSCON, kPINT_PinInt1, DEMO_PINT_PIN_INT1_SRC);
 #else
     /* Connect trigger sources to PINT */
     INPUTMUX_Init(INPUTMUX);
//     INPUTMUX_AttachSignal(INPUTMUX, kPINT_PinInt0, DEMO_PINT_PIN_INT0_SRC);
//     INPUTMUX_AttachSignal(INPUTMUX, kPINT_PinInt1, DEMO_PINT_PIN_INT1_SRC);
//     INPUTMUX_AttachSignal(INPUTMUX, kPINT_PinInt2, DEMO_PINT_PIN_INT2_SRC);
     INPUTMUX_AttachSignal(INPUTMUX, kPINT_PinInt0, DEMO_PINT_PIN_INT3_SRC);

     /* Turnoff clock to inputmux to save power. Clock is only needed to make changes */
     INPUTMUX_Deinit(INPUTMUX);
 #endif /* FSL_FEATURE_SYSCON_HAS_PINT_SEL_REGISTER */

     /* Initialize PINT */
     PINT_Init(PINT);

     /* Setup Pin Interrupt 0 for rising edge */
     PINT_PinInterruptConfig(PINT, kPINT_PinInt0, kPINT_PinIntEnableRiseEdge, pint_intr_callback);
     /* Enable callbacks for PINT0 by Index */
     PINT_EnableCallbackByIndex(PINT,kPINT_PinInt0);

 #if (FSL_FEATURE_PINT_NUMBER_OF_CONNECTED_OUTPUTS > 1U)
     /* Setup Pin Interrupt 1 for falling edge */
     PINT_PinInterruptConfig(PINT, kPINT_PinInt1, kPINT_PinIntEnableFallEdge, pint_intr_callback);
     /* Enable callbacks for PINT1 by Index */
     PINT_EnableCallbackByIndex(PINT,kPINT_PinInt1);
 #endif

 #if (FSL_FEATURE_PINT_NUMBER_OF_CONNECTED_OUTPUTS > 2U)
     /* Setup Pin Interrupt 2 for falling edge */
     PINT_PinInterruptConfig(PINT, kPINT_PinInt2, kPINT_PinIntEnableFallEdge, pint_intr_callback);
     /* Enable callbacks for PINT2 by Index */
     PINT_EnableCallbackByIndex(PINT,kPINT_PinInt2);
 #endif

//     PRINTF("\r\nPINT Pin Interrupt events are configured\r\n");
//     PRINTF("\r\nPress corresponding switches to generate events\r\n");

     level_test_cmd(0,NULL);
     cli_print_prompt();
     ////////////////////////////////////////////////////////////////////////////
     //
     // Stay in an infinite loop prompting for input to pass to the cli facility.
     //
     ////////////////////////////////////////////////////////////////////////////
     for(;;)
     {
    	SetUp_UserButton();
    	DbgConsole_ReadLine(input, MAXMESG);
        cli_engine((char *)&input);
     }

}
