/******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2011-2012 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
***************************************************************************//*!
*
* @file mma8451.c
*
* @author B37811
*
* @version 0.0.1
*
* @date Sep 19, 2011
*
* @brief application entry point which performs application specific tasks. 
*
*******************************************************************************
*
* provide a demo for how to initialize the PT60, output messages via SCI, 
* flash operations, etc.
* NOTE:
*	printf call may occupy a lot of memory (around 1924 bytes), so please
*	consider your code size before using printf.
******************************************************************************/
#include "mma8451.h"

/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: MMA8451_Standby
   *
   * @brief description: configure the MMA8451 as standby mode
   *        
   * @parameter:  none
   *
   * @return: none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/
void MMA8451_Standby (void)
{
    unsigned int n;
/*
** Read current value of System Control 1 Register.
** Put sensor into Standby Mode by clearing the Active bit
** Return with previous value of System Control 1 Register.
*/
    n = u8MMA8451ReadRegister(CTRL_REG1);
    MMA8451WriteRegister(CTRL_REG1, n & ~ACTIVE_MASK);
}


/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: MMA8451_Active
   *
   * @brief description: configure the MMA8451 as Active mode
   *        
   * @parameter:  none
   *
   * @return: none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/
void MMA8451_Active (void)
{
/*
** Set the Active bit in CTRL Reg 1
*/
    MMA8451WriteRegister(CTRL_REG1, (u8MMA8451ReadRegister(CTRL_REG1) | ACTIVE_MASK));
}


/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: MMA8451_2g_mode_Select
   *
   * @brief description: configure the MMA8451 as 2g mode
   *        
   * @parameter:  none
   *
   * @return: none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/
void MMA8451_2g_mode_Select(void)
{
/*
**Put the part in Standby Mode
*/
    MMA8451_Standby();
/*
**Write the 4g dynamic range value into register 0x0E
*/
    MMA8451WriteRegister(XYZ_DATA_CFG_REG, (u8MMA8451ReadRegister(XYZ_DATA_CFG_REG) & ~FS_MASK));
    MMA8451WriteRegister(XYZ_DATA_CFG_REG, (u8MMA8451ReadRegister(XYZ_DATA_CFG_REG) | FULL_SCALE_2G)); // 2g mode
    //MMA8451WriteRegister(XYZ_DATA_CFG_REG, (u8MMA8451ReadRegister(XYZ_DATA_CFG_REG) | FULL_SCALE_4G)); // 4g mode   
    //MMA8451WriteRegister(XYZ_DATA_CFG_REG, (u8MMA8451ReadRegister(XYZ_DATA_CFG_REG) | FULL_SCALE_8G)); // 8g mode
    /*
**Put the part back into the Active Mode
*/
    MMA8451_Active();
}


/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: MMA8451_Output_Rate_Set
   *
   * @brief description: configure the MMA8451's output data rate 
   *        
   * @parameter:  DataRateValue: data rate value to be written to the date rate register
   *
   * @return: none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/
void MMA8451_Output_Rate_Set(unsigned char DataRateValue)
{
	/*
	** Adjust the desired Output Data Rate value as needed.
	*/
	DataRateValue &= 0x07;
	DataRateValue <<= 3;
/*
** Put the device into Standby Mode
*/
    MMA8451_Standby();
/*
** Write in the Data Rate value into Ctrl Reg 1
*/
    MMA8451WriteRegister(CTRL_REG1,u8MMA8451ReadRegister(CTRL_REG1) & ~DR_MASK);
    MMA8451WriteRegister(CTRL_REG1,u8MMA8451ReadRegister(CTRL_REG1)| DataRateValue);
/*
** Put the device into the active mode again.
*/
    MMA8451_Active();
}

/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: MMA8451_HPFilter_Cutofffrequency_Set
   *
   * @brief description: Select the desired cut-off frequency.
   *        
   * @parameter:  CutOffValue: cup off value to be written to  register
   *
   * @return: none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/
void MMA8451_HPFilter_Cutofffrequency_Set(unsigned char CutOffValue)  // CutOffValue :a value from 0x00 to 0x03
{
/*
** Put the device in Standby Mode
*/
    MMA8451_Standby();
/*
** Write in the cut off value
*/
    MMA8451WriteRegister(HP_FILTER_CUTOFF_REG,u8MMA8451ReadRegister(HP_FILTER_CUTOFF_REG)& ~SEL_MASK);
    MMA8451WriteRegister(HP_FILTER_CUTOFF_REG,u8MMA8451ReadRegister(HP_FILTER_CUTOFF_REG)| CutOffValue);
/*
** Put the device back in the active mode
*/
    MMA8451_Active();
}

/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: MMA8451_Highpss_Filter_Set
   *
   * @brief description: configure the MMA8451 as high pass filter mode.
   *        
   * @parameter:  none
   *
   * @return: none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/
void MMA8451_Highpss_Filter_Set(void)
{
    MMA8451_Standby();
/*
** Set the HPF_OUT Bit to enable the HPF Data Out
*/
    MMA8451WriteRegister(XYZ_DATA_CFG_REG, (u8MMA8451ReadRegister(XYZ_DATA_CFG_REG) | HPF_OUT_MASK));
    MMA8451_Active();
}

 
/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: MMA8451_Initial
   *
   * @brief description: MMA8451 Initial as  Motion Detection mode.
   *        
   * @parameter:  none
   *
   * @return: none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/
void MMA8451_Initial (void)
{
	MMA8451_Standby ();
	 // MMA8451WriteRegister(CTRL_REG1,u8MMA8451ReadRegister(CTRL_REG1)& ~FREAD_MASK); 
	 // MMA8451WriteRegister(CTRL_REG2,u8MMA8451ReadRegister(CTRL_REG2)| ST_MASK); // self test enable
	
	//Set the device in 100 Hz ODR, Standby
	MMA8451WriteRegister(CTRL_REG1, 0x18); 
	
	// Set Configuration Register for Motion Detection by setting the OR condition OAE = 1
	// enabling X, Y, and the latch
	MMA8451WriteRegister(FF_MT_CFG_REG, 0xD8);
	
	// Threshold Setting Value for the Motion detection of > 3g
	// Note: The step count is 0.063g/count, 3g/0.063g = 47.6, Round up to 48
	MMA8451WriteRegister(FF_MT_THS_REG, 0x30);
	//MMA8451WriteRegister(FF_MT_THS_REG, 0x02);
	
	// Set the debounce counter to eliminate false readings for 100 Hz sample rate with a requirement
	// of 100 ms timer.Note: 100 ms/10 ms (steps) = 10 counts
	MMA8451WriteRegister(FF_MT_COUNT_REG, 0x0A);
	
	// Enable Motion/Freefall Interrupt Function in the System (CTRL_REG4)
	MMA8451WriteRegister(CTRL_REG4, 0x04);
	
	// Route the Motion/Freefall Interrupt Function to INT1 hardware pin (CTRL_REG5)
	MMA8451WriteRegister(CTRL_REG5, 0x04);
		 
	MMA8451_Active();
}



