/******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2011-2012 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
***************************************************************************//*!
*
* @file iic.c
*
* @author B37811
*
* @version 0.0.1
*
* @date Sep 19, 2011
*
* @brief application entry point which performs application specific tasks. 
*
*******************************************************************************
*
* provide a demo for how to initialize the PT60, output messages via SCI, 
* flash operations, etc.
* NOTE:
*	printf call may occupy a lot of memory (around 1924 bytes), so please
*	consider your code size before using printf.
******************************************************************************/
#include "iic.h"

unsigned int  resultx, resulty, resultz;
unsigned char SlaveID;


/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: init_I2C
   *
   * @brief description: I2C Initialization, Set Baud Rate and turn on I2C
   *        
   * @parameter:  none 
   *
   * @return: none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/
void init_I2C(void)
{
    /* Configure GPIO for I2C function */
	//SYS_SOPT1_IICPS = 0; // default, IIC SCL and SDA are mapped on PTA3 and PTA2 or PTB7 and PTB6
	I2C_F  = 0x1D;   // 100K baud rate
    I2C_C1 = I2C_C1_IICEN_MASK |    // enable IIC 
             I2C_C1_MST_MASK   |    // master
             I2C_C1_TX_MASK;           
}

/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: Pause
   *
   * @brief description: delay 50 nops
   *        
   * @parameter:  none 
   *
   * @return: none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/
void Pause(void){
    int n;
    for(n=1;n<50;n++) {
      asm(nop);
    }
}

/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: IIC_StartTransmission
   *
   * @brief description: Start I2C Transmission
   *        
   * @parameter:  SlaveID, Mode
   *
   * @return: none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/
void IIC_StartTransmission (unsigned char SlaveID, unsigned char Mode)
{
  unsigned char MasterTransmission;
  if(Mode == MWSR)
  {
	  /* Master write  */
    MasterTransmission = MWSR;  
  }
  else
  {
	  /* Master read */
    MasterTransmission = MRSW;  
  }

  /* shift ID in right possition */
  SlaveID = (unsigned char) MMA8451_I2C_ADDRESS << 1;

  /* Set R/W bit at end of Slave Address */
  SlaveID |= (unsigned char)MasterTransmission;

  /* send start signal */
  i2c_Start();

  /* send ID with W/R bit */
  i2c_write_byte(SlaveID);
}


/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: u8MMA8451ReadRegister
   *
   * @brief description: Read a register from the MMA8451
   *        
   * @parameter:  u8RegisterAddress
   *
   * @return: unsigned int 
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/
unsigned int u8MMA8451ReadRegister(unsigned char u8RegisterAddress)
{
  unsigned int dummy,result;

  /* Send Slave Address */
  IIC_StartTransmission(SlaveID,MWSR);
  i2c_Wait();

  /* Write Register Address */
  I2C_D = u8RegisterAddress;
  i2c_Wait();

  /* Do a repeated start */
  I2C_C1 |= I2C_C1_RSTA_MASK;

  /* Send Slave Address */
  I2C_D = (MMA8451_I2C_ADDRESS << 1) | 0x01; //read address
  i2c_Wait();

  /* Put in Rx Mode */
  I2C_C1 &= ~I2C_C1_TX_MASK;
  
  /* Turn off ACK ,send NACK*/
  i2c_DisableAck();

  /* Dummy read */
  dummy = I2C_D ;
  i2c_Wait();
  
  /* Read one byte */
  result = I2C_D;
  i2c_Wait();

  /* Send stop */
  i2c_Stop();
  return result;
}

/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: MMA8451_Read_3axis_data
   *
   * @brief description: Read first three registers from the MMA8451
   *        
   * @parameter:  none
   *
   * @return: none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/
void MMA8451_Read_3axis_data(void)
{
   // if((u8MMA8451ReadRegister(DATA_STATUS_REG) & 0x0f) != 0)  // polling
    {
        resultx   = u8MMA8451ReadRegister(0x01)<<8;
        resultx  |= u8MMA8451ReadRegister(0x02);
        resultx >>= 2;

        resulty   = u8MMA8451ReadRegister(0x03)<<8;
        resulty  |= u8MMA8451ReadRegister(0x04);
        resulty >>= 2;

        resultz   = u8MMA8451ReadRegister(0x05)<<8;
        resultz  |= u8MMA8451ReadRegister(0x06);
        resultz >>= 2;
    }
}


/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: MMA8451WriteRegister
   *
   * @brief description: Write a byte of Data to the MMA8451
   *        
   * @parameter:  u8RegisterAddress: register address to be written.
   *              u8Data: data to be written 
   *
   * @return: none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/
void MMA8451WriteRegister(unsigned char u8RegisterAddress, unsigned char u8Data)
{
  /* send data to slave */
  IIC_StartTransmission(SlaveID,MWSR);
  i2c_Wait();

  /* Send I2C address */
  I2C_D = u8RegisterAddress;
  i2c_Wait();

  /* Send data */
  I2C_D = u8Data;
  i2c_Wait();

  i2c_Stop();

  Pause();
}






