/*
 * PCA85063A_RTC.c
 *
 *  Created on: Jul 18, 2018
 *      Author: nxa07657
 */

#include "PCA85063A_RTC.h"
#include "cpu.h"

/*
 * convert the BCD code number to decimal number
 */
#define  BCD_to_DEC(x) ((x&0xF)+((x&0xF0)>>4)*10)
/*
 * convert the decimal number to BCD code number
 */
#define  DEC_to_BCD(x) ((x%10)+((x/10)<<4))

static unsigned short days[4][12] =
{
    {   0,  31,  60,  91, 121, 152, 182, 213, 244, 274, 305, 335},
    { 366, 397, 425, 456, 486, 517, 547, 578, 609, 639, 670, 700},
    { 731, 762, 790, 821, 851, 882, 912, 943, 974,1004,1035,1065},
    {1096,1127,1155,1186,1216,1247,1277,1308,1339,1369,1400,1430},
};


static unsigned int date_time_to_epoch(RTC_Calendar_t* date_time)
{
    unsigned int second = date_time->Second;  // 0-59
    unsigned int minute = date_time->Minute;  // 0-59
    unsigned int hour   = date_time->Hour;    // 0-23
    unsigned int day    = date_time->Day-1;   // 0-30
    unsigned int month  = date_time->Month-1; // 0-11
    unsigned int year   = date_time->Year-2000;    // 0-99
    return (((year/4*(365*4+1)+days[year%4][month]+day)*24+hour)*60+minute)*60+second+946656000;
}



/*
 * initialize the PCA85063A RTC module
 *
 * send command to soft reset the chip
 */
status_t PCA85063A_RTC_Init(void)
{
	status_t Status = STATUS_SUCCESS;
	uint8_t masterTxBuffer[10] = {0};/*buffer used to send the I2C message frame to PCA85063A*/


    masterTxBuffer[0] = 0x00;/*set the Register Control_1 address (0x00) */
    masterTxBuffer[1] = 0x58;/*configure Register Control_1 = 0x58 for soft reset */
    I2C_MasterSetSlaveAddress(&i2c1_instance,RTC_DEVICE_ADDR,false);
    Status = I2C_MasterSendDataBlocking(RTC_I2C, masterTxBuffer, 2, true, 0xFF);/*soft reset the RTC */

    return Status;
}
/*
 * set the PCA85063A RTC with current Calendar
 */
status_t PCA85063A_RTC_Set_Calendar(RTC_Calendar_t Calendar)
{
	status_t Status = STATUS_SUCCESS;
	uint32_t Buffer_Index = 0;
	uint8_t masterTxBuffer[10] = {0};/*buffer used to send the I2C message frame to PCA85063A*/

	 masterTxBuffer[Buffer_Index++] = 0x04;/*set the write start address as 0x04 */
	 masterTxBuffer[Buffer_Index++] = DEC_to_BCD(Calendar.Second);
	 masterTxBuffer[Buffer_Index++] = DEC_to_BCD(Calendar.Minute);
	 masterTxBuffer[Buffer_Index++] = DEC_to_BCD(Calendar.Hour);
	 masterTxBuffer[Buffer_Index++] = DEC_to_BCD(Calendar.Day);
	 masterTxBuffer[Buffer_Index++] = DEC_to_BCD(Calendar.Weekday);
	 masterTxBuffer[Buffer_Index++] = DEC_to_BCD(Calendar.Month);
	 masterTxBuffer[Buffer_Index++] = DEC_to_BCD(Calendar.Year%100);/*can only set the  offset with 100* year*/
	 I2C_MasterSetSlaveAddress(&i2c1_instance,RTC_DEVICE_ADDR,false);
	 Status = I2C_MasterSendDataBlocking(RTC_I2C, masterTxBuffer, Buffer_Index, true, 0xFF);/*transmit salve address + write and  0x04 as read start address as well as 8 Bytes configuration data*/

	 return Status;
}
status_t  PCA85063A_RTC_Get_TS(uint32_t* ts){
	status_t status=STATUS_SUCCESS;
	RTC_Calendar_t cal;
	status=PCA85063A_RTC_Get_Calendar(&cal);
	*ts=date_time_to_epoch(&cal);
	return status;
}
/*
 * get current Calendar from the PCA85063A RTC
 */
status_t PCA85063A_RTC_Get_Calendar(RTC_Calendar_t *Calendar)
{
	I2C_MasterSetSlaveAddress(&i2c1_instance,RTC_DEVICE_ADDR,false);
	status_t Status = STATUS_SUCCESS;
	uint32_t Buffer_Index = 0;

	uint8_t masterTxBuffer[10] = {0};/*buffer used to send the I2C message frame to PCA85063A*/
	uint8_t masterRxBuffer[10] = {0};/*buffer used to receive the I2C message frame to PCA85063A*/

	masterTxBuffer[0] = 0x04;/*set the read start address as 0x04 */
	I2C_MasterSendDataBlocking(RTC_I2C, masterTxBuffer, 1, true, 0xFF);/*transmit salve address + write and  0x04 as read start address*/

	I2C_MasterReceiveDataBlocking(RTC_I2C, masterRxBuffer, 7, true, 0xFF);/*transmit salve address + read and read back Time and date registers address from 0x04 to 0x0A*/

	Calendar->Second = BCD_to_DEC(masterRxBuffer[0]&0X7F);
	Calendar->Minute = BCD_to_DEC(masterRxBuffer[1]);
	Calendar->Hour = BCD_to_DEC(masterRxBuffer[2]);
	Calendar->Day = BCD_to_DEC(masterRxBuffer[3]);
	Calendar->Weekday = BCD_to_DEC(masterRxBuffer[4]);
	Calendar->Month = BCD_to_DEC(masterRxBuffer[5]);
	Calendar->Year = 2000+BCD_to_DEC(masterRxBuffer[6]);/*add the year base--2000 year*/

	return Status;
}

/*
 * set the PCA85063A RTC with Calendar_Alarm
 */
status_t PCA85063A_RTC_Set_Calendar_Alarm(RTC_Calendar_t Calendar_Alarm)
{
	status_t Status = STATUS_SUCCESS;
	uint32_t Buffer_Index = 0;

	uint8_t masterTxBuffer[10] = {0};/*buffer used to send the I2C message frame to PCA85063A*/

	 masterTxBuffer[Buffer_Index++] = 0x0B;/*set the write start address as 0x0B */
	 masterTxBuffer[Buffer_Index++] = DEC_to_BCD(Calendar_Alarm.Second);
	 masterTxBuffer[Buffer_Index++] = DEC_to_BCD(Calendar_Alarm.Minute);
	 masterTxBuffer[Buffer_Index++] = DEC_to_BCD(Calendar_Alarm.Hour);
	 masterTxBuffer[Buffer_Index++] = DEC_to_BCD(Calendar_Alarm.Day);
	 masterTxBuffer[Buffer_Index++] = DEC_to_BCD(Calendar_Alarm.Weekday);
	 I2C_MasterSetSlaveAddress(&i2c1_instance,RTC_DEVICE_ADDR,false);
	 Status = I2C_MasterSendDataBlocking(RTC_I2C,  masterTxBuffer, Buffer_Index++, true, 0xFF);/*transmit salve address + write and  0x04 as read start address as well as 8 Bytes configuration data*/

	 return Status;
}

/*
 * get the PCA85063A RTC current Calendar_Alarm configuration
 */
status_t PCA85063A_RTC_Get_Calendar_Alarm(RTC_Calendar_t *Calendar_Alarm)
{
	status_t Status = STATUS_SUCCESS;
	I2C_MasterSetSlaveAddress(&i2c1_instance,RTC_DEVICE_ADDR,false);
	uint8_t masterTxBuffer[10] = {0};/*buffer used to send the I2C message frame to PCA85063A*/
	uint8_t masterRxBuffer[10] = {0};/*buffer used to receive the I2C message frame to PCA85063A*/

	masterTxBuffer[0] = 0x0B;/*set the read start address as 0x0B */
	I2C_MasterSendDataBlocking(RTC_I2C,  masterTxBuffer, 1, true, 0xFF);/*transmit salve address + write and  0x04 as read start address*/

	I2C_MasterReceiveDataBlocking(RTC_I2C,  masterRxBuffer, 5, true, 0xFF);/*transmit salve address + read and read back Time and date registers address from 0x04 to 0x0A*/

	Calendar_Alarm->Second = BCD_to_DEC(masterRxBuffer[0]);
	Calendar_Alarm->Minute = BCD_to_DEC(masterRxBuffer[1]);
	Calendar_Alarm->Hour = BCD_to_DEC(masterRxBuffer[2]);
	Calendar_Alarm->Day = BCD_to_DEC(masterRxBuffer[3]);
	Calendar_Alarm->Weekday = BCD_to_DEC(masterRxBuffer[4]);

	return Status;
}


#if 0
status_t PCA85063A_RTC_Test(void)
{
	status_t status = STATUS_SUCCESS;

	RTC_Calendar_t Calendar_Init={
			.Second = 0,
			.Minute = 0,
			.Hour = 0,
			.Day = 19,
			.Weekday = 4,
			.Day = 19,
			.Month = 7,
			.Year = 2018
	};

	RTC_Calendar_t Calendar_Alarm_Set={
				.Second = 0,
				.Minute = 30,
				.Hour = 10,
				.Day = 19,
				.Weekday = 4
		};

	RTC_Calendar_t Calendar_Current;
	RTC_Calendar_t Calendar_Alarm_Get;

	status = PCA85063A_RTC_Init();

	status = PCA85063A_RTC_Set_Calendar(Calendar_Init);

	status = PCA85063A_RTC_Set_Calendar_Alarm(Calendar_Alarm_Set);
	status = PCA85063A_RTC_Get_Calendar_Alarm(&Calendar_Alarm_Get);
	while(1)
	{
		OSIF_TimeDelay(1000);/*delay 1000ms = 1 second*/
		PCA85063A_RTC_Get_Calendar(&Calendar_Current);/*get current calendar time*/
		PINS_DRV_TogglePins(LED_GPIO_PORT, (1<<LED_BLUE));
	}

	return status;

}
#endif
