/***********************************************************************************************\
* Freescale MMA8451,2,3Q Driver
*
* Filename: mma845x.c
*
*
* (c) Copyright 2010, Freescale, Inc.  All rights reserved.
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale Semiconductor.
*
\***********************************************************************************************/

#include "MMA845x.h"

#include "S32K148_TBOX_BSP.h"
#include "S32K148_TBOX_BSP_test_Config.h"

#include "S32K148_TBOX_BSP.h"

/***********************************************************************************************\
* Private macros
\***********************************************************************************************/

/***********************************************************************************************\
* Private type definitions
\***********************************************************************************************/

/***********************************************************************************************\
* Private prototypes
\***********************************************************************************************/

/***********************************************************************************************\
* Private memory declarations
\***********************************************************************************************/

/***********************************************************************************************\
* Public memory declarations
\***********************************************************************************************/



/***********************************************************************************************\
* Public functions
\***********************************************************************************************/

/*
 * write the register at RegAddr with value of RegVal
 */
void IIC_RegWrite(uint8_t RegAddr, uint8_t RegVal)
{
	uint8_t ACC_masterTxBuffer[8] = {0};

    ACC_masterTxBuffer[0] = RegAddr;/*set target register address*/
    ACC_masterTxBuffer[1] = RegVal;  /*set the register value to write*/
    I2C_MasterSendDataBlocking(ACC_I2C,  ACC_masterTxBuffer, 2, true, 0xFFFF);/*transmit slave address + write and register address */
}
/*
 * read back the register at RegAddr, and return one byte by *RegVal
 */
void IIC_RegRead(uint8_t RegAddr, uint8_t *RegVal)
{
	uint8_t ACC_masterTxBuffer[8] = {0};

	ACC_masterTxBuffer[0] = RegAddr;/*set target register address*/
	I2C_MasterSendDataBlocking(ACC_I2C,  ACC_masterTxBuffer, 1, false, 0xFFFF);/*transmit slave address + write and register address */

	I2C_MasterReceiveDataBlocking(ACC_I2C,  RegVal, 1, true, 0xFFFF);/*transmit slave address + read and read the register*/

}
/*
 * read back the register at RegAddr, and return Byte_Num byte by *RegVal
 */
void IIC_RegRead_MultiByte(uint8_t RegAddr, uint8_t *RegVal, uint32_t Byte_Num)
{
	uint8_t ACC_masterTxBuffer[8] = {0};

	ACC_masterTxBuffer[0] = RegAddr;/*set target register address*/
	I2C_MasterSendDataBlocking(ACC_I2C,  ACC_masterTxBuffer, 1, false, 0xFFFF);/*transmit slave address + write and register address */

	I2C_MasterReceiveDataBlocking(ACC_I2C,  RegVal, Byte_Num, true, 0xFFFF);/*transmit slave address + read and read the register*/

}
/*********************************************************\
* Put MMA845xQ into Active Mode
\*********************************************************/
void MMA845x_Active ()
{
	uint8_t RegVal_Read = 0;

	IIC_RegRead(CTRL_REG1, &RegVal_Read);
	IIC_RegWrite(CTRL_REG1, (RegVal_Read|ACTIVE_MASK));

}

/*********************************************************\
* Put MMA845xQ into Standby Mode
\*********************************************************/
void MMA845x_Standby (void)
{
  uint8_t n;
  /*
  **  Read current value of System Control 1 Register.
  **  Put sensor into Standby Mode.
  **  Return with previous value of System Control 1 Register.
  */

	IIC_RegRead(CTRL_REG1,&n);
	IIC_RegWrite(CTRL_REG1, n & ~ACTIVE_MASK);

}

/***********************************************************************************\
* Initialize MMA845xQ
* configure the I2C interface
* read the device ID and identify the chip, if correct do initialization
\***********************************************************************************/
void MMA845x_Init (void)
{
	uint8_t temp = 0;

  /*
   * Initialize the I2C module for ACC communication
   */

   /*initialize the I2C interface for MMA8452 accelerator communication*/
   I2C_MasterSetSlaveAddress(ACC_I2C,i2c1_MasterConfig1.slaveAddress, i2c1_MasterConfig1.is10bitAddr);/* set the slave address */

  IIC_RegRead(MMA845x_WHO_AM_I,&temp);/*read the device ID*/

  if(temp==MMA8452Q_ID)/*if it is matched, do initialization*/
  {

	  /* first step: Put the MMA8452 into Standby mode for configuration*/
	  MMA845x_Standby();
	  /*
	   * second step: configure the parameter as user requirements
	   *
	  **  Configure sensor for:
	  **    - Sleep Mode Poll Rate of 50Hz (20ms)
	  **    - System Output Data Rate of 200Hz (5ms)
	  **    - Full Scale of +/-8g
	  */
	  IIC_RegWrite(CTRL_REG1, ASLP_RATE_20MS+DATA_RATE_5MS);
	  IIC_RegWrite(XYZ_DATA_CFG_REG, FULL_SCALE_8G);

	  /* last step: Put the MMA8452 into Active mode for normal work*/
	  MMA845x_Active();
  }
}

/*********************************************************\
* get the 3-axis accelerator data with unit g from MMA845xQ
\*********************************************************/
void MMA845x_Get_accelerator (float *x, float *y, float *z)
{
	uint8_t Raw_Data[6] ={0};
	uint8_t Status_Flag = 0;

	uint32_t i = 0;

	signed int temp = 0;

	IIC_RegRead(MMA845x_STATUS_00,&Status_Flag);/*read back the status register*/

	if(Status_Flag&ZYXDR_MASK)/*if a new set of data is ready*/
	{

	//	IIC_RegRead_MultiByte(MMA845x_OUT_X_MSB,Raw_Data,6);/*read back the raw data*/

		for(i=0;i<6;i++)
		{
			 IIC_RegRead((i+1),&Raw_Data[i]);
		}


		/*
		 * get the x axis accelerated velocity
		 */
		temp = ((int16_t)(((uint16_t)Raw_Data[0]<<8) + Raw_Data[1]))>>4;

		*x = (float)temp/256; /*convert LSB count to g*/


		/*
		 * get the y axis accelerated velocity
		 */
		temp = ((int16_t)(((uint16_t)Raw_Data[2]<<8) + Raw_Data[3]))>>4;

		*y = (float)temp/256; /*convert LSB count to g*/


		/*
		 * get the z axis accelerated velocity
		 */
		temp = ((int16_t)(((uint16_t)Raw_Data[4]<<8) + Raw_Data[5]))>>4;

		*z = (float)temp/256; /*convert LSB count to g*/
	}
}

/*********************************************************\
* a simple test of MMA8452 3-axis accelerator
\*********************************************************/
void MMA8452_Test(void)
{
	float ACC_x,ACC_y,ACC_z;

	/********************************************************************************************
	 * if test PCA85063A_RTC at the same time, it's a must to re-configure the I2C slave address
	 * as they are using the same I2C but with different salve address
	 * *****************************************************************************************/
#ifdef PCA85063A_RTC_test
	/*initialize the I2C interface for MMA8452 accelerator communication*/
	I2C_MasterSetSlaveAddress(ACC_I2C,i2c1_MasterConfig1.slaveAddress, i2c1_MasterConfig1.is10bitAddr);/* set the slave address */
#endif

//		OSIF_TimeDelay(200);/*delay 200ms = 0.2 second*/

	MMA845x_Get_accelerator(&ACC_x,&ACC_y,&ACC_z);

	printf("the MMA8452A accelerator has completed the sample:\n\r x-axis = %f g;\n y-axis = %f g;\n z-axis = %f g;\n\r", ACC_x,ACC_y,ACC_z);

}


/***********************************************************************************************\
* Private functions
\***********************************************************************************************/
