/*
 * CAN_BSP_Test.c
 *
 *  Created on: Sep 28, 2018
 *      Author: nxa07657
 */

/*
 * include the BSP and test configuration header
 */
#include "S32K148_TBOX_BSP.h"
#include "S32K148_TBOX_BSP_test_Config.h"

  /* CAN 2.0 A/B Buffer configuration */
  can_buff_config_t CAN_buffConfig = {
   .enableFD = false,
   .enableBRS = false,
   .fdPadding = 0xCC,
   .idType = CAN_MSG_ID_STD,
   .isRemote = false
  };

  /* CAN-FD Buffer configuration */
  can_buff_config_t CAN_FD_buffConfig = {
   .enableFD = true,  /*enable CAN-FD feature*/
   .enableBRS = true, /*enable data filed bite rate switch in a CAN-FD frame*/
   .fdPadding = 0xCC,
   .idType = CAN_MSG_ID_STD,
   .isRemote = false
  };

/***************************************************************************************************************************
 * the RX ID filter table for CAN1 RX FIFO, user can set choose each element whether to receive or not the
 * 1) remote frame,
 * 2) extended frame,
 * 3) the acceptance ID
 * the filter elements number and format is determined by the CAN_PAL component configuration after enable Rx FIFO extension
 ****************************************************************************************************************************/
flexcan_id_table_t CAN1_RX_FIFO_Filter_Table[8] ={
		{  /*ID filter element 0*/
			  .isRemoteFrame = false,
			  .isExtendedFrame = false,
			  .id = 0x100
		},
		{  /*ID filter element 1*/
			   .isRemoteFrame = false,
			   .isExtendedFrame = false,
			   .id = 0x111
		},
		{  /*ID filter element 2*/
			  .isRemoteFrame = false,
			  .isExtendedFrame = false,
			  .id = 0x122
		},
		{  /*ID filter element 3*/
			   .isRemoteFrame = false,
			   .isExtendedFrame = false,
			.id = 0x133
		},
		{  /*ID filter element 4*/
			  .isRemoteFrame = false,
			  .isExtendedFrame = false,
			  .id = 0x144
		},
		{  /*ID filter element 5*/
			   .isRemoteFrame = false,
			   .isExtendedFrame = false,
			   .id = 0x155
		},
		{  /*ID filter element 6*/
			  .isRemoteFrame = false,
			  .isExtendedFrame = false,
			  .id = 0x166
		},
		{  /*ID filter element 7*/
			   .isRemoteFrame = false,
			   .isExtendedFrame = false,
			   .id = 0x333
		}

};

#define CAN0_MB_Rx_Buffer_MAX 2         /*the maximum buffer number for CAN0 interrupt event callback receive*/
#define CAN1_MB_Rx_Buffer_MAX 2         /*the maximum buffer number for CAN1 interrupt event callback receive*/
#define CAN2_MB_Rx_Buffer_MAX 2         /*the maximum buffer number for CAN2 interrupt event callback receive*/

 uint32_t CAN0_MB_RxBuffer_Index = 0;   /*buffer index for CAN0 interrupt event callback receive*/
 uint32_t CAN0_MB_RxComplete_Flag = 0;  /*RX complete flag for CAN0 interrupt event callback receive*/
 uint32_t CAN0_MB_TxComplete_Flag = 0;  /*TX complete flag for CAN0 interrupt event callback receive*/
 can_message_t recvMsg0_Buffer[CAN0_MB_Rx_Buffer_MAX]; /*buffer for CAN0 interrupt event callback receive*/

 uint32_t CAN1_MB_RxBuffer_Index = 0;   /*buffer index for CAN1 interrupt event callback receive*/
 uint32_t CAN1_MB_RxComplete_Flag = 0;  /*RX complete flag for CAN1 interrupt event callback receive*/
 uint32_t CAN1_MB_TxComplete_Flag = 0;  /*TX complete flag for CAN1 interrupt event callback receive*/
 can_message_t recvMsg1_Buffer[CAN1_MB_Rx_Buffer_MAX]; /*buffer for CAN1 interrupt event callback receive*/

 uint32_t CAN2_MB_RxBuffer_Index = 0;   /*buffer index for CAN2 interrupt event callback receive*/
 uint32_t CAN2_MB_RxComplete_Flag = 0;  /*RX complete flag for CAN2 interrupt event callback receive*/
 uint32_t CAN2_MB_TxComplete_Flag = 0;  /*TX complete flag for CAN2 interrupt event callback receive*/
 can_message_t recvMsg2_Buffer[CAN2_MB_Rx_Buffer_MAX]; /*buffer for CAN2 interrupt event callback receive*/

 uint32_t CAN0_BusOff_Recover_DoneCnt = 0;/*counter used to record the CAN0 bus off recover done*/
 uint32_t CAN1_BusOff_Recover_DoneCnt = 0;/*counter used to record the CAN1 bus off recover done*/
 uint32_t CAN2_BusOff_Recover_DoneCnt = 0;/*counter used to record the CAN2 bus off recover done*/

 uint32_t CAN0_BusOff_Cnt = 0;/*counter used to record the CAN0 bus off event*/
 uint32_t CAN1_BusOff_Cnt = 0;/*counter used to record the CAN1 bus off event*/
 uint32_t CAN2_BusOff_Cnt = 0;/*counter used to record the CAN2 bus off event*/

 /*!< Error IRQ handler callback */
 void CAN_Error_ISR_Callback(uint8_t instance,
                           flexcan_event_type_t eventType,
                           struct FlexCANState *driverState)
{

	uint32_t Can_Error_Status = 0;

	DEV_ASSERT(FLEXCAN_EVENT_ERROR == eventType);

	DEV_ASSERT(instance < CAN_INSTANCE_COUNT);

	Can_Error_Status = FLEXCAN_DRV_GetErrorStatus(instance);

	if(Can_Error_Status&CAN_ESR1_BOFFDONEINT_MASK)/*bus off done interrupt*/
	{
		switch(instance)
		{
			case 0: CAN0_BusOff_Recover_DoneCnt++;/*record the bus off recover done by add the counter*/
					break;
			case 1: CAN1_BusOff_Recover_DoneCnt++;/*record the bus off recover done by add the counter*/
					break;
			case 2: CAN2_BusOff_Recover_DoneCnt++;/*record the bus off recover done by add the counter*/
					break;
			default: break;
		}
	}

	if(Can_Error_Status&CAN_ESR1_BOFFINT_MASK)/*bus off interrupt*/
	{
		switch(instance)
		{
			case 0: CAN0_BusOff_Cnt++;/*record the bus off event by add the counter*/
					break;
			case 1: CAN1_BusOff_Cnt++;/*record the bus off event by add the counter*/
					break;
			case 2: CAN2_BusOff_Cnt++;/*record the bus off event by add the counter*/
					break;
			default: break;
		}
	}
}

 /*!<IRQ handler event callback */
 void CAN_IRQ_ISR_Callback(uint32_t instance,
						 can_event_t eventType,
						 uint32_t objIdx,
						 void *driverState)
{

	DEV_ASSERT(instance < CAN_INSTANCE_COUNT);

	 if(CAN_EVENT_RX_COMPLETE==eventType)
	 {
		 if(instance==0)
		 {
			 CAN0_MB_RxBuffer_Index++;/*increase the index to use next buffer for receive*/
			 /* Receive data using buffer configured for reception */

			 if(CAN0_MB_RxBuffer_Index>=CAN0_MB_Rx_Buffer_MAX)/*index check*/
				 CAN0_MB_RxBuffer_Index = 0;/*reset the index to zero*/
			 /*
			  * start a new receive
			  */
			 CAN_Receive(TBOX_CAN0, CAN0_RX_BUFF_IDX, &recvMsg0_Buffer[CAN0_MB_RxBuffer_Index]);

			 CAN0_MB_RxComplete_Flag = 1;/*set the flag*/
		 }
		 else if(instance==1)
		 {
			 CAN1_MB_RxBuffer_Index++;/*increase the index to use next buffer for receive*/
			 /* Receive data using buffer configured for reception */

			 if(CAN1_MB_RxBuffer_Index>=CAN1_MB_Rx_Buffer_MAX)/*index check*/
				 CAN1_MB_RxBuffer_Index = 0;/*reset the index to zero*/
			 /*
			  * start a new receive
			  */
			 CAN_Receive(TBOX_CAN1, CAN1_RX_BUFF_IDX, &recvMsg1_Buffer[CAN1_MB_RxBuffer_Index]);

			 CAN1_MB_RxComplete_Flag = 1;/*set the flag*/
		 }
		 else if(instance==2)
		 {
			 CAN2_MB_RxBuffer_Index++;/*increase the index to use next buffer for receive*/
			 /* Receive data using buffer configured for reception */

			 if(CAN2_MB_RxBuffer_Index>=CAN2_MB_Rx_Buffer_MAX)/*index check*/
				 CAN2_MB_RxBuffer_Index = 0;/*reset the index to zero*/
			 /*
			  * start a new receive
			  */
			 CAN_Receive(TBOX_CAN2, CAN2_RX_BUFF_IDX, &recvMsg2_Buffer[CAN2_MB_RxBuffer_Index]);

			 CAN2_MB_RxComplete_Flag = 1;/*set the flag*/
		 }
		 else
		 {
			 /*do nothing*/
		 }

	 }

	 if(CAN_EVENT_TX_COMPLETE==eventType)
	 {
		 if(instance==0)
		 {
			 CAN0_MB_TxComplete_Flag = 1;/*set the flag*/
		 }
		 else if(instance==1)
		 {
			 CAN1_MB_TxComplete_Flag = 1;/*set the flag*/
		 }
		 else if(instance==2)
		 {
			 CAN2_MB_TxComplete_Flag = 1;/*set the flag*/
		 }
		 else
		 {
			 /*do nothing*/
		 }

	 }

}
/*******************************************************************************************************************************
 * call CAN_PAL component initialization API to initialize CAN0~CAN2according to their processor expert configuration
 * and also configure the CAN transceiver to work in normal mode
 *
 * CAN0, use FlexCAN0, enable TX/RX CAN-FD with 64B data and data rate switch to 2Mbit/s while arbitration phrase use 500Kbit/s
 * 		ID filter configure to receive only standard ID=0x111 CAN-FD data message
 * CAN1, use FlexCAN1, configure to TX/RX 500Kbit/s CAN 2.0 A/B message with RX FIFO and 8 Format-A(32-bit) filter elements
 *     ID filter configure to receive only standard ID=0x100/0x111/0x122/0x133/0x144/0x155/0x165/0x177 CAN data message
 *  CAN2, use FlexCAN2, configure to TX/RX 500Kbit/s CAN 2.0 A/B message with RX FIFO disable
 *     ID filter configure to receive only standard ID=0x333 CAN data message
 ********************************************************************************************************************************/
 void CAN_BSP_Init(void)
 {

	 /*
	  * initialize the CAN1 RX FIFO filter table provided by the CAN PAL, which will be use in API--CAN_Init()
	  */
	 can_pal2_rx_fifo_ext0.idFilterTable = CAN1_RX_FIFO_Filter_Table;
	/*
	* initialize FlexCAN0/1/2 for SBC(UJA1132), TJA1044 and TJA1043 transceiver
	*/

	 LPSPI_DRV_MasterInit(LPSPICOM1,&lpspiCom1State,&lpspiCom1_MasterConfig0);
	 UJA113X_Init(0, &sbc_uja113x1_InitConfig0,&sbc_uja113x1State);

	CAN_Init(TBOX_CAN0, &can_pal1_Config0);
	CAN_Init(TBOX_CAN1, &can_pal2_Config0);
	CAN_Init(TBOX_CAN2, &can_pal3_Config0);

	/*enable and install the callback for FlexCAN error interrupt*/
	FLEXCAN_DRV_InstallErrorCallback(can_pal1_instance.instIdx,CAN_Error_ISR_Callback,NULL);
	FLEXCAN_DRV_InstallErrorCallback(can_pal2_instance.instIdx,CAN_Error_ISR_Callback,NULL);
	FLEXCAN_DRV_InstallErrorCallback(can_pal3_instance.instIdx,CAN_Error_ISR_Callback,NULL);

	/*
	* configure TJA1044 and TJA1043 transceiver for normal work
	*/
	PINS_DRV_ClearPins(TJA1044_CTRL_GPIO_PORT,(1<<TJA1044_CTRL_STB));/*set TJA1044 in normal mode*/
	PINS_DRV_SetPins(TJA1043_CTRL_GPIO_PORT,(1<<TJA1043_CTRL_EN));   /*enable TJA1043*/
	PINS_DRV_SetPins(TJA1043_CTRL_GPIO_PORT,(1<<TJA1043_CTRL_STB_N));/*set TJA1043 in normal mode*/

	/*
	 * CAN0 MB configuration for TX and RX, and support CAN-FD
	 */
	CAN_ConfigTxBuff(TBOX_CAN0, CAN0_TX_BUFF_IDX, &CAN_FD_buffConfig);		 /*configure the FlexCAN MB for transmit*/
	CAN_ConfigRxBuff(TBOX_CAN0, CAN0_RX_BUFF_IDX, &CAN_FD_buffConfig, 0x111);/*configure the FlexCAN MB for receive*/

	/*
	 * CAN1 MB configuration for TX and RX, and support only classical CAN(CAN 2.0 A/B)
	 */
	CAN_ConfigTxBuff(TBOX_CAN1, CAN1_TX_BUFF_IDX, &CAN_buffConfig);		 /*configure the FlexCAN MB for transmit*/
	/*
	 * when enable RX FIFO, MB0~MB5 is used by RX FIFO, MB6~MB8 is use for 8 RX ID filter table
	 * !!! user cannot use API --CAN_ConfigRxBuff() to configure the MB for RX
	 */

	/*
	 * CAN2 MB configuration for TX and RX and support only classical CAN(CAN 2.0 A/B)
	 */
	CAN_ConfigTxBuff(TBOX_CAN2, CAN2_TX_BUFF_IDX, &CAN_buffConfig);		 /*configure the FlexCAN MB for transmit*/
	CAN_ConfigRxBuff(TBOX_CAN2, CAN2_RX_BUFF_IDX, &CAN_buffConfig, 0x333);/*configure the FlexCAN MB for receive*/

#ifndef CAN0_TRX_Polling_Method  /*use polling/blocking */

	/*enable and install the callback for FlexCAN error interrupt*/
	CAN_InstallEventCallback(TBOX_CAN0,CAN_IRQ_ISR_Callback,NULL);/*install the callback*/

	/* Receive data using buffer configured for reception */
	CAN_Receive(TBOX_CAN0, CAN0_RX_BUFF_IDX, &recvMsg0_Buffer[CAN0_MB_RxBuffer_Index]);/*CAN0 RX buffer initialization*/

	CAN0_MB_TxComplete_Flag = 1;/*the MBx should be OK to transmit CAN message after initialization, set the flag*/

#endif /*end of CAN0_TRX_Polling_Method*/

#ifndef CAN1_TRX_Polling_Method  /*use polling/blocking */

	/*enable and install the callback for FlexCAN error interrupt*/
	CAN_InstallEventCallback(TBOX_CAN1,CAN_IRQ_ISR_Callback,NULL);/*install the callback*/

	/* Receive data using buffer configured for reception */
	CAN_Receive(TBOX_CAN1, CAN1_RX_BUFF_IDX, &recvMsg1_Buffer[CAN1_MB_RxBuffer_Index]);/*CAN0 RX buffer initialization*/

	CAN1_MB_TxComplete_Flag = 1;/*the MBx should be OK to transmit CAN message after initialization, set the flag*/

#endif /*end of CAN0_TRX_Polling_Method*/

#ifndef CAN2_TRX_Polling_Method  /*use polling/blocking */

	/*enable and install the callback for FlexCAN error interrupt*/
	CAN_InstallEventCallback(TBOX_CAN2,CAN_IRQ_ISR_Callback,NULL);/*install the callback*/

	/* Receive data using buffer configured for reception */
	CAN_Receive(TBOX_CAN2, CAN2_RX_BUFF_IDX, &recvMsg2_Buffer[CAN2_MB_RxBuffer_Index]);/*CAN0 RX buffer initialization*/

	CAN2_MB_TxComplete_Flag = 1;/*the MBx should be OK to transmit CAN message after initialization, set the flag*/

#endif /*end of CAN2_TRX_Polling_Method*/

 }
/*****************************************************************************************
 * CAN BSP API test sample function
 * transmit CAN/CAN-FD message
 * receive the CAN/CAN-FD message, increase the first data byte and then transmit it back
 *
 * user can test the polling/blocking TX/RX or interrupt callback TX/RX by configure
 *  the MACRA  CAN0_TRX_Polling_Method in header file--S32K148_TBOX_BSP_test_Config.h
 *
 *****************************************************************************************/
 void CAN_BSP_Test(void)
 {

#ifdef CAN0_test

	   /*CAN message structure used for data transmit and receive*/
#if 0  /*for CAN 2.0 A/B message fame*/
	   can_message_t sendMsg0 = {
		 .id = 0x100,
		 .length = 5U,
		 .data = {"Hello"}
	   };
#else
	   can_message_t sendMsg0 = {
		 .id = 0x100,
		 .length = 64U,
		 .data = {"1234567812345678123456781234567812345678123456781234567812345678"}
	   };
#endif
	   /*
		* CAN0 TX/RX test
		*/
#ifdef CAN0_TRX_Polling_Method  /*use polling/blocking */

	   can_message_t recvMsg0;/*used for CAN frame receive*/

	   /* Send data using buffer configured for transmission */
	   CAN_Send(TBOX_CAN0, CAN0_TX_BUFF_IDX, &sendMsg0);
	   while(CAN_GetTransferStatus(TBOX_CAN0, CAN0_TX_BUFF_IDX) == STATUS_BUSY);
	   /* Receive data using buffer configured for reception */
	   CAN_Receive(TBOX_CAN0, CAN0_RX_BUFF_IDX, &recvMsg0);
	   while(CAN_GetTransferStatus(TBOX_CAN0, CAN0_RX_BUFF_IDX) == STATUS_BUSY);

	   /* Send back the received data using buffer configured for transmission */
	   recvMsg0.data[0]++;/*increase the first data*/
	   CAN_Send(TBOX_CAN0, CAN0_TX_BUFF_IDX, &recvMsg0);
	   while(CAN_GetTransferStatus(TBOX_CAN0, CAN0_TX_BUFF_IDX) == STATUS_BUSY);

#else	/*use interrupt call back method to RX */

	   if(CAN0_MB_TxComplete_Flag&&CAN0_MB_RxComplete_Flag)
	   {

		   /* Send back the received data using buffer configured for transmission */

		   if(CAN0_MB_RxBuffer_Index==0)/*index roll back ?*/
		   {
			   recvMsg0_Buffer[CAN0_MB_Rx_Buffer_MAX-1].data[0]++;/*increase the first data*/
			   CAN_Send(TBOX_CAN0, CAN0_TX_BUFF_IDX, &recvMsg0_Buffer[CAN0_MB_Rx_Buffer_MAX-1]);
		   }
		   else
		   {
			   recvMsg0_Buffer[CAN0_MB_RxBuffer_Index-1].data[0]++;/*increase the first data*/
			   CAN_Send(TBOX_CAN0, CAN0_TX_BUFF_IDX, &recvMsg0_Buffer[CAN0_MB_RxBuffer_Index-1]);
		   }

		   CAN0_MB_TxComplete_Flag = 0;/*clean the flag*/
		   CAN0_MB_RxComplete_Flag = 0;/*clean the flag*/

	   }

	   if(CAN0_MB_TxComplete_Flag)/*check if TX completed ?*/
	   {
		   /* Send data using buffer configured for transmission */
		   CAN_Send(TBOX_CAN0, CAN0_TX_BUFF_IDX, &sendMsg0);

		   CAN0_MB_TxComplete_Flag = 0;/*clean the flag*/
	   }

#endif /*end of CAN0_TRX_Polling_Method*/

#endif /*end of CAN0_test*/

#ifdef CAN1_test
	   /*
		* CAN1 TX/RX test
		*/
	   can_message_t sendMsg1 = {
				 .id = 0x101,
				 .length = 5U,
				 .data = {"T-BOX"}
			   };

#ifdef CAN1_TRX_Polling_Method  /*use polling/blocking */

	   can_message_t recvMsg1;/*used for CAN frame receive*/

	   /* Send data using buffer configured for transmission */
	   CAN_Send(TBOX_CAN1, CAN1_TX_BUFF_IDX, &sendMsg1);
	   while(CAN_GetTransferStatus(TBOX_CAN1, CAN1_TX_BUFF_IDX) == STATUS_BUSY);
	   /* Receive data using buffer configured for reception */
	   CAN_Receive(TBOX_CAN1, CAN1_RX_BUFF_IDX, &recvMsg1);
	   while(CAN_GetTransferStatus(TBOX_CAN1, CAN1_RX_BUFF_IDX) == STATUS_BUSY);
	   /* Send back the received data using buffer configured for transmission */
	   recvMsg1.data[0]++;/*increase the first data*/
	   CAN_Send(TBOX_CAN1, CAN1_TX_BUFF_IDX, &recvMsg1);
	   while(CAN_GetTransferStatus(TBOX_CAN1, CAN1_TX_BUFF_IDX) == STATUS_BUSY);

#else	/*use interrupt call back method to RX */

	   if(CAN1_MB_TxComplete_Flag&&CAN1_MB_RxComplete_Flag)
	   {

		   /* Send back the received data using buffer configured for transmission */

		   if(CAN1_MB_RxBuffer_Index==0)/*index roll back ?*/
		   {
			   recvMsg1_Buffer[CAN1_MB_Rx_Buffer_MAX-1].data[0]++;/*increase the first data*/
			   CAN_Send(TBOX_CAN1, CAN1_TX_BUFF_IDX, &recvMsg1_Buffer[CAN1_MB_Rx_Buffer_MAX-1]);
		   }
		   else
		   {
			   recvMsg1_Buffer[CAN1_MB_RxBuffer_Index-1].data[0]++;/*increase the first data*/
			   CAN_Send(TBOX_CAN1, CAN1_TX_BUFF_IDX, &recvMsg1_Buffer[CAN1_MB_RxBuffer_Index-1]);
		   }

		   CAN1_MB_TxComplete_Flag = 0;/*clean the flag*/
		   CAN1_MB_RxComplete_Flag = 0;/*clean the flag*/

	   }

	   if(CAN1_MB_TxComplete_Flag)/*check if TX completed ?*/
	   {
		   /* Send data using buffer configured for transmission */
		   CAN_Send(TBOX_CAN1, CAN1_TX_BUFF_IDX, &sendMsg1);

		   CAN1_MB_TxComplete_Flag = 0;/*clean the flag*/
	   }

#endif /*end of CAN1_TRX_Polling_Method*/
#endif /*end of CAN1_test*/

#ifdef CAN2_test

	   /*
		* CAN2 TX/RX test
		*/
	   can_message_t sendMsg2 = {
				 .id = 0x333,
				 .length = 5U,
				 .data = {"S32K1"}
			   };

#ifdef CAN2_TRX_Polling_Method  /*use polling/blocking */

	   can_message_t recvMsg2;/*used for CAN frame receive*/

	   /* Send data using buffer configured for transmission */
	   CAN_Send(TBOX_CAN2, CAN2_TX_BUFF_IDX, &sendMsg2);
	   while(CAN_GetTransferStatus(TBOX_CAN2, CAN2_TX_BUFF_IDX) == STATUS_BUSY);
	   /* Receive data using buffer configured for reception */
	   CAN_Receive(TBOX_CAN2, CAN2_RX_BUFF_IDX, &recvMsg2);
	   while(CAN_GetTransferStatus(TBOX_CAN2, CAN2_RX_BUFF_IDX) == STATUS_BUSY);
	   /* Send back the received data using buffer configured for transmission */
	   recvMsg2.data[0]++;/*increase the first data*/
	   CAN_Send(TBOX_CAN2, CAN2_TX_BUFF_IDX, &recvMsg2);
	   while(CAN_GetTransferStatus(TBOX_CAN2, CAN2_TX_BUFF_IDX) == STATUS_BUSY);

#else	/*use interrupt call back method to RX */

	   if(CAN2_MB_TxComplete_Flag&&CAN2_MB_RxComplete_Flag)
	   {

		   /* Send back the received data using buffer configured for transmission */

		   if(CAN2_MB_RxBuffer_Index==0)/*index roll back ?*/
		   {
			   recvMsg2_Buffer[CAN2_MB_Rx_Buffer_MAX-1].data[0]++;/*increase the first data*/
			   CAN_Send(TBOX_CAN2, CAN2_TX_BUFF_IDX, &recvMsg2_Buffer[CAN2_MB_Rx_Buffer_MAX-1]);
		   }
		   else
		   {
			   recvMsg2_Buffer[CAN2_MB_RxBuffer_Index-1].data[0]++;/*increase the first data*/
			   CAN_Send(TBOX_CAN2, CAN2_TX_BUFF_IDX, &recvMsg2_Buffer[CAN2_MB_RxBuffer_Index-1]);
		   }

		   CAN2_MB_TxComplete_Flag = 0;/*clean the flag*/
		   CAN2_MB_RxComplete_Flag = 0;/*clean the flag*/

	   }

	   if(CAN2_MB_TxComplete_Flag)/*check if TX completed ?*/
	   {
		   /* Send data using buffer configured for transmission */
		   CAN_Send(TBOX_CAN2, CAN2_TX_BUFF_IDX, &sendMsg2);

		   CAN2_MB_TxComplete_Flag = 0;/*clean the flag*/
	   }

#endif /*end of CAN2_TRX_Polling_Method*/

#endif /*CAN2_test*/

	   printf("the bus off on CAN0 has occurred %d times, and been recovered %d times\n",CAN0_BusOff_Cnt,CAN0_BusOff_Recover_DoneCnt);
	   printf("the bus off on CAN1 has occurred %d times, and been recovered %d times\n",CAN1_BusOff_Cnt,CAN1_BusOff_Recover_DoneCnt);
	   printf("the bus off on CAN2 has occurred %d times, and been recovered %d times\n",CAN2_BusOff_Cnt,CAN2_BusOff_Recover_DoneCnt);
 }


