/*
 * Copyright 2017 NXP
 * All rights reserved.
 *
 * THIS SOFTWARE IS PROVIDED BY NXP "AS IS" AND ANY EXPRESSED OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL NXP OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef TSS_RAW_H
#define TSS_RAW_H

#include "adc_driver.h"
#include "lptmr_driver.h"
#include "interrupt_manager.h"
#include "pins_driver.h"

/*! @file tss_raw.h */

/*!
 * @addtogroup tss_mw
 * @{
 */

/*******************************************************************************
 * Definitions
 ******************************************************************************/

/*!
 * @brief Electrode configuration structure.
 * It holds data representing the ADC instance, ADC channel configurations
 * and GPIO pins/bases.
 * @note: Both CEXT and ELEC pins shall share the same GPIO and PORT
 */
typedef struct
{
	uint32_t adcInstance;                    /*!< ADC instance number */
	adc_chan_config_t const * channelConfig; /*!< ADC channel configuration */
	PORT_Type * portBase;			         /*!< PORT base for both CEXT and ELEC pins */
	GPIO_Type * gpioBase;                    /*!< GPIO base for both CEXT and ELEC pins */
	uint32_t pinNumberElec;                  /*!< ELEC pin number */
	uint32_t pinNumberCext;                  /*!< Cext pin number */
	uint32_t portMask;                       /*!< Port mask, example (1UL << pinNumberElec)|(1UL << pinNumberCext) */
} tss_raw_electrode_config_t;

/*!
 * @brief Configuration structure containing the required information to
 * initialize the raw acquisition layer of the Touch Sense solution.
 */
typedef struct
{
	tss_raw_electrode_config_t ** electrodes;  /*!< Pointer to the electrode array */
	uint8_t numberOfElectrodes;                /*!< Number of configured electrodes */
	adc_converter_config_t const * adcConfig;  /*!< Pointer to the ADC configuration */
	lptmr_config_t const * lptmrConfig;        /*!< Pointer to the LPTMR configuration which contains the sampling frequency */
	uint32_t lptmrInstance;                    /*!< LPTMR instance number */
	uint32_t chargeDistributionPeriod;         /*!< Charge distribution period */
	uint32_t sensingCycles;                    /*!< Number of following electrode cycles per single sample. Range [1,6] */
	uint32_t sensingPrecycles;				   /*!< Number of following electrode precycles per single sample. Range [1,2] */
	int16_t * sampleBuffer;					   /*!< Pointer to the sample buffer which is required for internal calculations */
	void (*timerCallback)(void *);			   /*!< Pointer to the timer callback */
} tss_raw_config_t;

/*!
 * @brief  State structure that is used to store electrode values.
 * @note: For middleware internal use only. The values contained should not
 * be used directly since their values might change without notice.
 */
typedef struct
{
	int16_t  dischargeRaw;
	int32_t  dischargeBuffer;
	uint16_t dischargeBufferCounter;
} tss_raw_electrode_state_t;

/*!
 * @brief Middleware state structure. Contains information that is necessary
 * in order to follow the internal state machine.
 * @note: For middleware internal use only. The values contained should not
 * be used directly since their values might change without notice.
 */
typedef struct
{
	tss_raw_electrode_config_t ** electrodes;   	/*!< Pointer to the electrode array */
	uint8_t numberOfElectrodes;                		/*!< Number of configured electrodes */
	uint32_t chargeDistributionPeriod;          	/*!< Charge distribution period */
	uint32_t sensingCycles;                     	/*!< Number of following electrode cycles per single sample. Range [1,6] */
	uint32_t sensingPrecycles;				    	/*!< Number of following electrode precycles per single sample. Range [1,2] */
	lptmr_config_t const * lptmrConfig;             /*!< Pointer to the LPTMR configuration which contains the sampling frequency */
	uint32_t lptmrInstance;                     	/*!< LPTMR instance number */
	tss_raw_electrode_state_t ** electrodeState; 	/*!< Electrode state vector */
	int16_t * sampleBuffer;					    	/*!< Pointer to the sample buffer which is required for internal calculations */
	void (*timerCallback)(void *);		            /*!< Pointer to the timer callback */
} tss_raw_state_t;

/*******************************************************************************
 * Function prototypes
 ******************************************************************************/
/*!
 * @name Touch Sense Solution Raw Acquisition Layer functions
 * @{
 */

#if defined(__cplusplus)
extern "C" {
#endif

/*!
 * @brief Initialization function for the sampling layer of the
 * Touch Sense solution.
 * It will initialize the internal state structures, ADC instance
 * which are configured for use with electrodes and the timer which
 * will provide an interrupt used for sampling.
 * The application must ensure the that clocks which used by
 * the allocated peripherals are enabled.
 * @param[in] configPtr         Configuration pointer
 * @param[in] statePtr          Pointer to the state structure used to store
 *                              internal data. The application must allocate
 *                              the storage required in such manner that it
 *                              will be not destroyed until the Raw layer
 *                              is deinitialized.
 * @param[in] electrodeStateArr Pointer to the array of electrode state structures
 * 								used by the middleware to store values that are
 * 								retrieved after conversions.
 */
void TSS_Raw_Init(tss_raw_config_t * configPtr,
		          tss_raw_state_t * statePtr,
     			  tss_raw_electrode_state_t ** electrodeStateArr);

/*!
 * @brief Deinitializes the TSS Raw layer by freeing the used structures
 * and peripherals.
 */
void TSS_Raw_Deinit(void);

/*!
 * @brief Convert electrode capacitance to equivalent voltage
 * @param[in] electrodeIndex Electrode index
 * @return Equivalent voltage of the electrode
 */
int16_t TSS_Raw_GetElectrodeValue(uint8_t electrodeIndex);

/*!
 * @brief Start allocated timer module with a compare value
 * defined by 'ticks' parameter.
 * @param[in] ticks Number of ticks that shall pass before a
 *                  compare event occurs
 */
void TSS_Raw_StartTimer(uint32_t ticks);

/*!
 * @brief Stop allocated timer module.
 */
void TSS_Raw_StopTimer(void);

/*!
 * @brief Installs a callback function that will be called
 * when the allocated timer raises an interrupt due to the
 * compare match event. Prior to this callback, timer compare
 * flags will be cleared.
 * The callback should have the following signature:
 *  void aTimerCallback(void * param);
 *  Where the raw layer state structure will be passed as the
 *  param argument.
 * @param[in] callback Pointer to the callback function.
 */
void TSS_Raw_InstallTimerCallback(void (*callback)(void *));

/*!
 * @brief This function will set the value of the electrode
 * to high impedance.
 * @param[in] electrodeIndex Electrode number
 */
void TSS_Raw_ElectrodeFloat(uint8_t electrodeIndex);

/*!
 * @brief This function will set the value of the electrode
 * to ground level.
 * @param electrodeIndex Electrode number
 */
void TSS_Raw_ElectrodeGnd(uint8_t electrodeIndex);

#if defined(__cplusplus)
}
#endif

/*! @}*/

/*! @}*/

#endif /* TSS_RAW_H */
/*******************************************************************************
 * EOF
 ******************************************************************************/
