/*
 * Copyright 2017 NXP
 * All rights reserved.
 *
 * THIS SOFTWARE IS PROVIDED BY NXP "AS IS" AND ANY EXPRESSED OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL NXP OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef TSS_FILTER_DECODE_H
#define TSS_FILTER_DECODE_H

#include "tss_raw.h"
#include <stdint.h>

/*! @file tss_filter_decode.h */

/*!
 * @addtogroup tss_mw
 * @{
 */

/*******************************************************************************
 * Definitions
 ******************************************************************************/
/*! Type casting - signed 32-bit fractional value cast to a signed 64-bit integer, the value placed at the lower
*   32-bits of the 64-bit result. */
#define F32TOINT64(x)       ((int64_t) (x))
/*! Constant representing the maximal negative value of a signed 16-bit fixed point fractional number, floating point representation. */
#define SFRACT_MIN          (-1.0)

/*! Constant representing the maximal positive value of a signed 16-bit fixed point fractional
 *  number,  floating point representation. */
#define SFRACT_MAX          (0.999969482421875)

/*! Macro converting a signed fractional [-1,1) number into a 16-bit fixed point number in format Q1.15.*/
#define FRAC16(x)           ((tFrac16) (((x) < SFRACT_MAX) ? (((x) >= SFRACT_MIN) ? ((x)*32768.0) : INT16_MIN) : INT16_MAX))

typedef int16_t tFrac16;
typedef int32_t tFrac32;

typedef struct
{
  tFrac16 f16B0; /*!< B0 coefficient of an IIR1 filter, fractional format normalized to fit into (-2\f$^{15}\f$, 2\f$^{15}\f$-1). */
  tFrac16 f16B1; /*!< B1 coefficient of an IIR1 filter, fractional format normalized to fit into (-2\f$^{15}\f$, 2\f$^{15}\f$-1). */
  tFrac16 f16A1; /*!< A1 coefficient of an IIR1 filter, fractional format normalized to fit into (-2\f$^{15}\f$, 2\f$^{15}\f$-1). */
}GDFLIB_FILTER_IIR1_COEFF_T_F16;

typedef struct
{
    GDFLIB_FILTER_IIR1_COEFF_T_F16 trFiltCoeff; /*!< Sub-structure containing filter coefficients. */
    tFrac16 f16FiltBufferX[1]; /*!< Input buffer of an IIR1 filter, fractional format normalized to fit into (-2\f$^{15}\f$, 2\f$^{15}\f$-1). */
    tFrac32 f32FiltBufferY[1]; /*!< Internal accumulator buffer, fractional format normalized to fit into (-2\f$^{15}\f$, 2\f$^{15}\f$-1). */
}GDFLIB_FILTER_IIR1_T_F16;

/*!
 * @brief This structure defines the filter coefficients
 * that will be used to filter the electrodes raw values
 */
typedef struct
{
	float a1; /*!< A1 coefficient */
	float b0; /*!< B0 coefficient */
	float b1; /*!< B1 coefficient */
} tss_filter_coeff_t;

/*!
 * @brief Configuration structure used to define
 * values specific for an electrode such as
 * thresholds and filter coefficients.
 */
typedef struct
{
	uint32_t touchThredsholdDelta;  /*!< Difference required between DC tracker and electrode value to detect an touch event */
	uint32_t touchReleaseDelta;     /*!< Difference required between DC tracker and electrode value to detect an release event */
	uint8_t dcTrackerFactor;        /*!< DC tracker factor */
	tss_filter_coeff_t * filterPtr; /*!< Pointer to the filter coefficients structure */
} tss_electrode_filter_config_t;

/*!
 * @brief State structure for an electrode containing data
 * describing thresholds, filtering and events.
 * @note: For middleware internal use only. The values contained should not
 * be used directly since their values might change without notice.
 */
typedef struct
{
	uint32_t touchThredsholdDelta;    /*!< Difference required between DC tracker and electrode value to detect an touch event */
	uint32_t touchReleaseDelta;       /*!< Difference required between DC tracker and electrode value to detect an release event */
	int32_t touchThredshold;          /*!< Calculated threshold for touch event */
	int32_t touchRelease;             /*!< Calculated threshold for release event */
	uint32_t dcTrackerFactor;         /*!< DC Tracker factor */
	GDFLIB_FILTER_IIR1_T_F16 filter;  /*!< IIR filter structure containing calculated coefficients and past values */
	tFrac16 lpFilterData;             /*!< Last LP filtered value */
	int32_t dcTrackerBufferRaw;       /*!< Raw DC tracker value */
	int16_t dcTrackerBuffer;          /*!< Normalize DC tracker value */
	uint8_t dcTrackerBufferShift;     /*!< Shift coefficient for DC tracker value calculation */
	uint8_t touchQualified;           /*!< Flag indicating that a touch or release event was qualified */
	uint8_t touched;                  /*!< Flag indicating the digital value of the electrode [0, 1] */
} tss_electrode_filter_state_t;

/*!
 * @brief Filter layer state structure containing data required for the layer's internal
 * state machine.
 * @note: For middleware internal use only. The values contained should not
 * be used directly since their values might change without notice.
 */
typedef struct
{
	bool selfTrimDone;							   /*!< Flag indicating that the self trim process is completed */
	tss_electrode_filter_state_t ** electrodeState; /*!< Pointer to the electrode state array */
	uint8_t numberOfElectrodes;                    /*!< Number of electrodes */
	bool    wakeupEnable;                          /*!< Enables or disables wakeup functionality */
	uint8_t wakeupElectrodeNumber;                 /*!< Index of the wakeup electrode */
	uint8_t electrodeTouchNumberPlusOne;           /*!< Variable containing the number of touched electrodes */
	uint8_t electrodeTouchQualifiedReport;         /*!< Flag indicating wheter the qualify process was completed */
	uint8_t activeElectrode;					   /*!< Electrode which is currently sampled */
	uint32_t wakeupActivateCnt;					   /*!< How much will the MCU be active after a wakeup event is detected
	                                                *   and no other activity is detected
	                                                */
	uint32_t wakeupActivateInitial;
	volatile bool powerModeEnable;
} tss_filter_state_t ;

/*!
 * @brief Configuration structure for filtering layer.
 */
typedef struct
{
	tss_electrode_filter_config_t ** electrodesConfig; /*!< Electrode configuration array */
	uint8_t  numberOfElectrodes;                       /*!< Number of electrodes */
	bool     wakeupEnable;                             /*!< Enables or disables wakeup functionality */
	uint8_t  wakeupElectrodeNumber;                    /*!< Index of the wakeup electrode */
	uint32_t wakeupActivateCnt;
} tss_filter_config_t;

/*******************************************************************************
 * Function prototypes
 ******************************************************************************/
/*!
 * @name Touch Sense Solution Filter and Decode functions
 * @{
 */

#if defined(__cplusplus)
extern "C" {
#endif

/*!
 * @brief Initializes the Filter and Decode layer.
 * The function will copy the configurations to the internal
 * state structures and it will install the internal timer
 * callback required to perform the internal state machine.
 * @note: The application must ensure that TSS_Raw_Init was called
 * prior to this functions call. Without it, this layer will not
 * work as expected.
 * @param[in] configPtr    Pointer to the configuration structure
 * @param[in] statePtr     Pointer to the state structure used to store
*                          internal data. The application must allocate
 *                         the storage required in such manner that it
 *                         will be not destroyed until the Raw layer
 *                         is deinitialized.
 * @param[in] electrodeArr Pointer to the array of electrode state
 * 						   structures
 */
void TSS_FilterDecode_Init(tss_filter_config_t * configPtr,
		                       tss_filter_state_t * statePtr,
							   tss_electrode_filter_state_t ** electrodeArr);

/*!
 * @brief Deinitializes the filtering and decode layer of the TSS.
 * After the call of TSS_FilterDecode_Deinit, the application must
 * call TSS_Raw_Deinit.
 */
void TSS_FilterDecode_Deinit(void);

/*!
 * @brief This function will return the result of the electrodes
 * sampling and filtering in the form of an array containing the
 * values of the electrodes.
 * @param[out] resultArr Pointer to the storage location where the
 * values can be found.
 */
void TSS_FilterDecode_GetResult(uint8_t * resultArr);

/*!
 * @brief This function will provide the information describing
 * if the TSS middleware has finished it's internal calculation and
 * it is not affected by entering is sleep mode.
 *
 * @return True if the middleware is ready to sleep, false if not
 */
bool TSS_FilterDecode_ReadyToSleep(void);
#if defined(__cplusplus)
}
#endif

/*! @}*/

/*! @}*/

#endif /* TSS_FILTER_DECODE_H */
/*******************************************************************************
 * EOF
 ******************************************************************************/
