/****************************************************************************************************/
/**
Copyright (c) 2009 Freescale Semiconductor

\file       Thermostat_LCD.c
\brief      Routines to write Temperature, Date Time and other messages to the Thermostat LCd
\author     Freescale Semiconductor
\author     Technical Information Center
\version    1.0
\date       February/2009
*/
/********************************************************************************************
* Freescale reserves the right to make changes without further notice to any product
* herein to improve reliability, function, or design. Freescale does not assume any
* liability arising out of the application or use of any product, circuit, or software
* described herein; neither does it convey any license under its patent rights nor the
* rights of others. Freescale products are not designed, intended, or authorized for
* use as components in systems intended for surgical implant into the body, or other
* applications intended to support life, or for any other application in which the
* failure of the Freescale product could create a situation where personal injury or
* death may occur. Should Buyer purchase or use Freescale products for any such
* intended or unauthorized application, Buyer shall indemnify and hold Freescale and
* its officers, employees, subsidiaries, affiliates, and distributors harmless against
* all claims, costs, damages, and expenses, and reasonable attorney fees arising out
* of, directly or indirectly, any claim of personal injury or death associated with
* such unintended or unauthorized use, even if such claim alleges that Freescale was
* negligent regarding the design or manufacture of the part.
*
* Freescale is a registered trademark of Freescale, Inc.
********************************************************************************************/	

/*****************************************************************************************************
* Include files
*****************************************************************************************************/
#include "Thermostat_LCD.h"

/*****************************************************************************************************
* Declaration of module wide FUNCTIONs - NOT for use in other modules
*****************************************************************************************************/


/*****************************************************************************************************
* Definition of module wide MACROs / #DEFINE-CONSTANTs - NOT for use in other modules
*****************************************************************************************************/

/*****************************************************************************************************
* Declaration of module wide TYPEs - NOT for use in other modules
*****************************************************************************************************/


/*****************************************************************************************************
* Definition of module wide VARIABLEs - NOT for use in other modules
*****************************************************************************************************/


/*****************************************************************************************************
* Definition of module wide (CONST-) CONSTANTs - NOT for use in other modules
*****************************************************************************************************/
#ifdef __LOWEND_TSTAT__
const UINT8* DaysOfWeekLCDWaveForms[DAYS_OF_WEEK] = 
{
  (UINT8*)&_LCD_SUNDAY_WF,
  (UINT8*)&_LCD_MONDAY_WF,
  (UINT8*)&_LCD_TUESDAY_WF,
  (UINT8*)&_LCD_WEDNESDAY_WF,
  (UINT8*)&_LCD_THURSDAY_WF,
  (UINT8*)&_LCD_FRIDAY_WF,
  (UINT8*)&_LCD_SATURDAY_WF,
};

const UINT8 DaysOfWeekWaveformsBitNumbers[DAYS_OF_WEEK] = 
{
  (UINT8)_LCD_SUNDAY_BIT_NUMBER, 
  (UINT8)_LCD_MONDAY_BIT_NUMBER, 
  (UINT8)_LCD_TUESDAY_BIT_NUMBER, 
  (UINT8)_LCD_WEDNESDAY_BIT_NUMBER, 
  (UINT8)_LCD_THURSDAY_BIT_NUMBER, 
  (UINT8)_LCD_FRIDAY_BIT_NUMBER, 
  (UINT8)_LCD_SATURDAY_BIT_NUMBER
};
#endif

/*****************************************************************************************************
* Code of project wide FUNCTIONS
*****************************************************************************************************/

/**
* \brief    Initializes the HW LCD module and clears the screen
* \author   
* \param    void
* \return   void
* \todo     
*/
void vfnThermostatLCD_Init(void)
{
  vfnLCD_Init();
  vfnLCD_All_Segments_OFF();
}

/**
* \brief    Update the Time and Date in the LCD. Takes the corresponding LCD definition and updates the contents
* \author   
* \param    LocalRTC* Pointer to the RTC Structure
* \return   void
* \todo     
*/
void vfnUpdateDayAndTime(RTC_Type* LocalRTC)
{
  UINT8 TemporalHour;
  UINT8 ASCII_Hour[HOUR_DIGITS];
  
  TemporalHour = LocalRTC->Hours;
  
  #ifdef __LOWEND_TSTAT__
    _LCD_PM_OFF;
    _LCD_AM_ON;
    if (TemporalHour >= 12)
    {
      
      TemporalHour -= 12;
      _LCD_PM_ON;
      _LCD_AM_OFF;
    }
    
    if (!TemporalHour)
    {
      TemporalHour = 12;
    }
  
    _LCD_ONE_OFF;
    if (TemporalHour >= 10)
    {
      _LCD_ONE_ON;
      TemporalHour -= 10;
    }
    ASCII_Hour[0] = TemporalHour + '0';
    Dec8bitsToASCIIBCD(LocalRTC->Minutes, &ASCII_Hour[1]);
  
  #else
    Dec8bitsToASCIIBCD(LocalRTC->Hours, &ASCII_Hour[0]);
    Dec8bitsToASCIIBCD(LocalRTC->Minutes, &ASCII_Hour[2]);
  #endif
  
  
  vfnLCD_Home();
  vfnLCD_Write_Msg(&ASCII_Hour[0], HOUR_DIGITS);
}

#ifdef __LOWEND_TSTAT__
/**
* \brief    Displays the Day of Week (works only for the Low End version)
* \author   
* \param    DayOfWeek
* \return   void
* \todo     
*/
  void vfnDisplayDayOfWeek(UINT8 DayOfWeek)
  {
    UINT8* WaveFormPointer;
    
    _LCD_SUNDAY_OFF;
    _LCD_MONDAY_OFF;
    _LCD_TUESDAY_OFF;
    _LCD_WEDNESDAY_OFF;
    _LCD_THURSDAY_OFF;
    _LCD_FRIDAY_OFF;
    _LCD_SATURDAY_OFF;
    
    if (DayOfWeek < DAYS_OF_WEEK)
    {
      /* Get the address of the waveform pointing to this Day of the Week*/
      WaveFormPointer = (UINT8*)DaysOfWeekLCDWaveForms[DayOfWeek];
      /* Turn on the bit of the DOW and the alternate message for the same day of the Week*/
      *WaveFormPointer  |= (1<<DaysOfWeekWaveformsBitNumbers[DayOfWeek]) | (1<<(DaysOfWeekWaveformsBitNumbers[DayOfWeek] + 4));
    }
  }
  
/**
* \brief    Writes the alternate message to make an automatic blink (in Low-End only)
* \author   
* \param    DayOfWeek
* \return   void
* \todo     
*/
  void vfnBlinkDayOfWeek(UINT8 DayOfWeek)
  {
    UINT8* WaveFormPointer;
    
    if (DayOfWeek < DAYS_OF_WEEK)
    {
      WaveFormPointer = (UINT8*)DaysOfWeekLCDWaveForms[DayOfWeek];
      *WaveFormPointer &= ~(1<<(DaysOfWeekWaveformsBitNumbers[DayOfWeek] + 4));
    }
  }
  
#else
  void vfnDisplayDayOfWeek(UINT8 DayOfWeek)
  {

  }
#endif

/**
* \brief    Displays the current Battery Level
* \author   
* \param    BatteryLevel
* \return   void
* \todo     
*/
void vfnDisplayBatteryLife(UINT16 BatteryLevel)
{
  _LCD_BATTERY_ON;
  _LCD_BAT1_ON;
  _LCD_BAT2_ON;
  _LCD_BAT3_ON;
  _LCD_BAT4_ON;
  
  if (BatteryLevel < BATT_4_LEVEL_INDICATOR)
  {
    _LCD_BAT1_OFF;
  }
  if (BatteryLevel < BATT_3_LEVEL_INDICATOR)
  {
    _LCD_BAT2_OFF;
  }
  if (BatteryLevel < BATT_2_LEVEL_INDICATOR)
  {
    _LCD_BAT3_OFF;
  }
  if (BatteryLevel < BATT_1_LEVEL_INDICATOR)
  {
    _LCD_BAT4_OFF;
  }
}

#ifdef __LOWEND_TSTAT__
/**
* \brief    Displays the Temperature. Two digits for the low end and 3 for the HighEnd version
* \author   
* \param    BatteryLevel
* \return   void
* \todo     
*/
  void vfnDisplayTemperature(UINT16 Temperature)
  {
    UINT8 Ascii_Temperature[2];
    UINT8 LocalTemp = (UINT8)(Temperature / 10);
    /* round up to next digit if decimal is greater than 0.5 */
    if ((Temperature % 10) > 5)
    {
      LocalTemp++;
    }
    Dec8bitsToASCIIBCD(LocalTemp, &Ascii_Temperature[0]);
    LCD_SET_CURSOR(LCD_TEMPERATURE_POS);
    vfnLCD_Write_Msg(&Ascii_Temperature[0], TEMPERATURE_DIGITS);
  }
  
#else
  void vfnDisplayTemperature(UINT16 Temperature)
  {
    UINT8 Ascii_Temperature[3];
    UINT8 LocalTemp = (UINT8)(Temperature / 10);
    UINT8 Modulo = Temperature % 10;
    /* round up to next digit if decimal is greater than 0.5 */
    Dec8bitsToASCIIBCD(LocalTemp, &Ascii_Temperature[0]);
    Ascii_Temperature[2] = Modulo + 0x30;
    
    LCD_SET_CURSOR(LCD_TEMPERATURE_POS);
    _LCD_DT8_ON;
    _LCD_COL4_ON;
    vfnLCD_Write_Msg(&Ascii_Temperature[0], TEMPERATURE_DIGITS);
  }
#endif
