/*****************************************************************************
 *
 * MODULE:             JN-AN-1238
 *
 * COMPONENT:          app_serial_commands.c
 *
 * DESCRIPTION:        Serial Command parser for CMD Module
 *
 ****************************************************************************
 *
 * This software is owned by NXP B.V. and/or its supplier and is protected
 * under applicable copyright laws. All rights are reserved. We grant You,
 * and any third parties, a license to use this software solely and
 * exclusively on NXP products [NXP Microcontrollers such as JN5168, JN5179].
 * You, and any third parties must reproduce the copyright and warranty notice
 * and any other legend of ownership on each copy or partial copy of the
 * software.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * Copyright NXP B.V. 2019. All rights reserved
 *
 ***************************************************************************/

/****************************************************************************/
/***        Include files                                                 ***/
/****************************************************************************/

#include <jendefs.h>
#include "dbg.h"
#include "app_serial_commands.h"
#include "string.h"
#include "stdlib.h"
#include <MMAC.h>
#include "MiniMac.h"
#include "Command.h"
#include "fsl_ctimer.h"
#include "fsl_debug_console.h"


/****************************************************************************/
/***        Macro Definitions                                             ***/
/****************************************************************************/
#ifdef DEBUG_SERIAL
    #define TRACE_SERIAL      TRUE
#else
    #define TRACE_SERIAL      FALSE
#endif
#define BUS_CLK_FREQ	CLOCK_GetFreq(kCLOCK_BusClk)
/****************************************************************************/
/***        Type Definitions                                              ***/
/****************************************************************************/
#define COMMAND_BUF_SIZE   80

typedef struct
{
    uint8  au8Buffer[COMMAND_BUF_SIZE];
    uint8  u8Pos;
}tsCommand;

#if (JENNIC_CHIP_FAMILY == JN517x)
#define stricmp strcasecmp
#endif

/****************************************************************************/
/***        Local Function Prototypes                                     ***/
/****************************************************************************/
PRIVATE void vProcessCommand(void);

/****************************************************************************/
/***        Exported Variables                                            ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Variables                                               ***/
/****************************************************************************/
PRIVATE tsCommand sCommand;
PRIVATE int8 i8Power = DEFAULT_TX_POWER_dBm;
PRIVATE uint8 u8lChannel = DEFAULT_CHANNEL;
PRIVATE uint8 u8rChannel = 0;
PRIVATE uint16 u16PingTimeOut=500;
static ctimer_match_config_t matchConfig1;
extern PUBLIC volatile bool_t bAckReceived;
PRIVATE bool_t bShowCursor=TRUE;
PUBLIC bool_t bFind = FALSE;
PRIVATE bool_t bSendingChannel = FALSE;

extern PUBLIC tsPhyFrame sFrame;


void delayMicroseconds(uint32 u32Count){
	volatile uint32 u32Total = 0;
	for(u32Total=0;u32Total<u32Count;u32Total++ ){}
}

void ctimer_match1_callback(uint32_t flags)
{

	if(bSendingChannel){
		bSendingChannel = FALSE;
		PRINTF("Set PING to %d\n\r",u8lChannel);
		vCommand_Transmit(E_CMD_ID_PING, 0, NULL);
		CTIMER_StartTimer(CTIMER1);
	}
	else{
		CTIMER_StopTimer(CTIMER1);

		if (bFind){
			if(bAckReceived){
				PRINTF("Moving to Channel %d\n\r>",u8rChannel);
				u8lChannel =u8rChannel;
				bFind = FALSE;
				bSendingChannel = FALSE;
				// tell DUT to the intended channel
				uint32 u32Mask = 1 << u8rChannel;
				vCommand_SendChannel(u32Mask);

				// then move CMD module to that channel
				vMMAC_SetChannel(u8lChannel);
			}
			else{
				PRINTF("NO ACK\n\r");
				if(++u8lChannel==27){
					PRINTF("NO DUT FOUND\n\r>");
					bFind=FALSE;
					bSendingChannel = FALSE;
					vCommand_SendChannel(u8lChannel);
				}
				else{
					vMMAC_SetChannel(u8lChannel);
					PRINTF("Set Channel to %d\n\r",u8lChannel);
					bSendingChannel = TRUE;
					uint32 u32Mask = 1 << u8lChannel;
					vCommand_SendChannel(u32Mask);
					CTIMER_StartTimer(CTIMER1);
				}
			}
		}
		else{
			if(!bAckReceived){
				PRINTF("NO ACK\n\r>");
			}
		}
		bAckReceived = FALSE;
	}
}


ctimer_callback_t ctimer_callback_table[] = {
    ctimer_match1_callback, NULL, NULL, NULL, NULL, NULL, NULL, NULL};

/****************************************************************************/
/***        Exported Functions                                            ***/
/****************************************************************************/
/******************************************************************************
 * NAME: vProcessRxChar
 *
 * DESCRIPTION:
 * Processes the received character
 *
 * PARAMETERS:      Name            Usage
 * uint8            u8Char          Character
 *
 * RETURNS:
 * None
 ****************************************************************************/
PUBLIC void vProcessRxChar(uint8 u8Char)
{
    if ((u8Char >= 'a' && u8Char <= 'z'))
    {
        u8Char -= ('a' - 'A');
    }
    if ((sCommand.u8Pos < COMMAND_BUF_SIZE)  && (u8Char != 0x0d))
    {
        sCommand.au8Buffer[sCommand.u8Pos++] = u8Char;
    }
    else if (sCommand.u8Pos >= COMMAND_BUF_SIZE)
    {
        DBG_vPrintf(TRACE_SERIAL, "OverFlow\n");
        memset(sCommand.au8Buffer, 0, COMMAND_BUF_SIZE);
        sCommand.u8Pos = 0;
    }

    if (u8Char == 0x0d)
    {
    	sCommand.au8Buffer[sCommand.u8Pos++] = u8Char;
    	PRINTF("\n");
        vProcessCommand();
    	if (bShowCursor) PRINTF(">");
    	bShowCursor=TRUE;
    }
}

//void delayMicroseconds(uint32 u32Count){
	//uint32 u32Total = u32AHI_TickTimerRead() + (u32Count<<4);
	//while (u32AHI_TickTimerRead() <u32Total);
//}

PRIVATE void Start_Timer(void){
	ctimer_config_t config;

	CTIMER_GetDefaultConfig(&config);

	CTIMER_Init(CTIMER1, &config);

	/* Configuration 0 */
	matchConfig1.enableCounterReset = true;
	matchConfig1.enableCounterStop = false;
	matchConfig1.matchValue = (BUS_CLK_FREQ / 1000) * u16PingTimeOut;
	matchConfig1.outControl = kCTIMER_Output_NoAction;
	matchConfig1.outPinInitState = false;
	matchConfig1.enableInterrupt = true;

	CTIMER_RegisterCallBack(CTIMER1, &ctimer_callback_table[0], kCTIMER_SingleCallback);
	CTIMER_SetupMatch(CTIMER1, kCTIMER_Match_0, &matchConfig1);
	CTIMER_StartTimer(CTIMER1);
}


int16 i16LocalPower = 10;

/******************************************************************************
 * NAME: vProcessCommand
 *
 * DESCRIPTION:
 * Processed the received command
 *
 * PARAMETERS:      Name            Usage
 *
 * RETURNS:
 * None
 ****************************************************************************/
PRIVATE void vProcessCommand(void)
{
    uint8 Delimiters[] = "\n";
    char *token = NULL;

    token = (char *)strtok((char *)sCommand.au8Buffer, (char *)Delimiters);
#if 0
    PRINTF( "Command = %s\n", token);
#endif


    if(0 == strncmp((char*)token, "CUSTOM LPINGTIMEOUT ", 20))
    {
    	token+=20;
    	u16PingTimeOut = atoi(token);
    	PRINTF("Timeout set to %d\n\r",u16PingTimeOut);
    }
    else if(0 == strncmp((char*)token, "CUSTOM LPOWER ", 14))
    {
    	token+=14;
    	i16LocalPower = atoi(token);
    	if (i16LocalPower<0){
    		PRINTF("Power set to -%d\n\r",i16LocalPower);
    	}
    	else
    	{
    		PRINTF("Power set to %d\n\r",i16LocalPower);
    	}
    }
    else if(0 == strncmp((char*)token, "CUSTOM LSEND ", 13))
   	{
   		token+=13;
   		uint16 u16SendPacketCount = atoi(token);
   		PRINTF("Sending %d Packets\n\r",u16SendPacketCount);
   		int i;
		/* Start transmission */
		tsMacFrame stFrame;
		stFrame.u8PayloadLength =20;
		stFrame.u16FCF= FCF_TYPE_MAC_DATA | FCF_PUT_SECURITY(1) |
				FCF_PUT_DST_ADDR_MODE(FCF_ADDR_MODE_EXT)|FCF_PUT_SRC_ADDR_MODE(FCF_ADDR_MODE_EXT);
		stFrame.u16DestPAN=0xCAFE;
		stFrame.u16SrcPAN=0xCAFE;
		stFrame.uDestAddr.sExt.u32L =0x0;
		stFrame.uDestAddr.sExt.u32H =0x0;
		stFrame.uSrcAddr.sExt.u32L =0x0;
		stFrame.uSrcAddr.sExt.u32H =0x0;
		stFrame.u16FCS=0xFF;

	    vMMAC_SetChannelAndPower(u8lChannel, i16LocalPower);

		for (i=0;i<20;i++)
			stFrame.uPayload.au8Byte[i]=i;

   		for (i=0;i<u16SendPacketCount;i++){
			stFrame.u8SequenceNum++;
   			vMMAC_StartMacTransmit(&stFrame, E_MMAC_TX_START_NOW);
   	   		/* Wait for TX to complete before going back to receive mode */
   	   		while (u32MMAC_PollInterruptSourceUntilFired(E_MMAC_INT_TX_COMPLETE) == 0);
   	   		//delayMicroseconds(10000);
   	   		delayMicroseconds(20000);
   		}
   		PRINTF("Complete\r\n");

   	    vMMAC_SetChannelAndPower(u8lChannel, i8Power);

   		/* Restart Reception */
   		vMMAC_StartPhyReceive(&sFrame, E_MMAC_RX_START_NOW | E_MMAC_RX_ALLOW_MALFORMED | E_MMAC_RX_ALLOW_FCS_ERROR);
   	}
    else if(0 == strncmp((char*)token, "CUSTOM RPING", 12))
    {
       vCommand_Transmit(E_CMD_ID_PING, 0, NULL);
        Start_Timer();
		bShowCursor=FALSE;
		bAckReceived = FALSE;
    }

    else if(0 == strncmp((char*)token, "CUSTOM SETCHANNEL ", 18))
    {
		token+=18;
		u8rChannel = atoi(token);
		u8lChannel = u8rChannel;

		uint32 u32Mask = 1 << u8rChannel;
		vCommand_SendChannel(u32Mask);

		vMMAC_SetChannel(u8lChannel);
    	PRINTF("Channel set to %d\n\r",u8lChannel);

    }
    else if(0 == strncmp((char*)token, "CUSTOM LSETCHANNEL ", 19))
    {
    	token+=19;
    	u8lChannel = atoi(token);
    	vMMAC_SetChannel(u8lChannel);
    	PRINTF("Channel set to %d\n\r",u8lChannel);
    }
    else if(0 == strncmp((char*)token, "CUSTOM RSETCHANNEL ", 19))
    {
		token+=19;
		u8rChannel = atoi(token);

		uint32 u32Mask = 1 << u8rChannel;
		vCommand_SendChannel(u32Mask);
    }
    else if(0 == strncmp((char*)token, "CUSTOM LGETCHANNEL", 18))
    {
    	PRINTF("Channel:%d\n\r",u8lChannel);
    }
    else if(0 == strncmp((char*)token, "CUSTOM LSETPOWER ", 17))
    {
    	token+=17;
    	i8Power = atoi(token);
		PRINTF("Power set to -%d\n\r",i8Power);
    }
    else if(0 == strncmp((char*)token, "CUSTOM RSETPOWER ", 17))
    {
    	token+=17;
    	i8Power = atoi(token);
    	vCommand_Transmit(E_CMD_ID_SET_POWER, 1,  (uint8 *) &i8Power);
    }
    else if(0 == strncmp((char*)token, "CUSTOM LGETPOWER", 16))
    {
    	PRINTF("Power:%d\n\r",i8Power);
    }
    else if(0 == strncmp((char*)token, "CUSTOM RGETPOWER", 16))
    {
       vCommand_Transmit(E_CMD_ID_GET_POWER, 0, NULL);
		bShowCursor=FALSE;
    }
    else if(0 == strncmp((char*)token, "CUSTOM RSTREAM ", 15))
    {
    	token+=15;
        uint16 u16TimeOut = atoi(token);
    	uint8 data[2];
    	data[0] = u16TimeOut>>8;
    	data[1] = u16TimeOut&0xff;
        vCommand_Transmit(E_CMD_ID_STREAM, 2, data);
    }
    else if(0 == strncmp((char*)token, "CUSTOM RSTART ", 13))
    {
       vCommand_Transmit(E_CMD_ID_START_RX_TEST, 0, NULL);
    }
    else if(0 == strncmp((char*)token, "CUSTOM REND", 11))
    {
       vCommand_Transmit(E_CMD_ID_END_RX_TEST, 0, NULL);
		bShowCursor=FALSE;
    }
    else if(0 == strncmp((char*)token, "CUSTOM REBOOT", 13))
    {
       vCommand_Transmit(E_CMD_ID_REBOOT, 0, NULL);
       RESET_SystemReset();
    }
    else if(0 == strncmp((char*)token, "CUSTOM LREBOOT", 14))
    {
    	RESET_SystemReset();
    }
    else if (0 == strncmp((char*)token, "CUSTOM FIND ", 12))
	{
		token+=12;
		u8rChannel = atoi(token);
		u8lChannel = 11;
		//vMMAC_SetChannel(u8rChannel);
		PRINTF("Set Channel to %d\n\r",u8lChannel);
		bSendingChannel = TRUE;
		uint32 u32Mask = 1 << u8lChannel;
		vMMAC_SetChannel(u8lChannel);
		vCommand_SendChannel(u32Mask);
		bFind=TRUE;
		Start_Timer();
		bShowCursor=FALSE;

	}
    else if(0 == strncmp((char*)token, "CUSTOM RHARDWAREVERSION", 23))
    {
       vCommand_Transmit(E_CMD_ID_GET_HW_VERSION, 0, NULL);
		bShowCursor=FALSE;
    }
    else if(0 == strncmp((char*)token, "CUSTOM RSOFTWAREVERSION", 23))
    {
       vCommand_Transmit(E_CMD_ID_GET_SW_VERSION, 0, NULL);
		bShowCursor=FALSE;
    }
    else if(0 == strncmp((char*)token, "HELP", 4))
    {
    	PRINTF("custom rping\ncustom lpingtimeout <timeout(ms)>\ncustom setchannel <channel>");
    	PRINTF("\ncustom lsetchannel <channel>\ncustom rsetchannel <channel>\ncustom lgetchannel");
    	PRINTF("\ncustom lsetpower <power>\ncustom rsetpower <power>\ncustom lgetpower");
    	PRINTF("\ncustom rgetpower\ncustom rstream <duration(ms)>\ncustom rstart");
    	PRINTF("\ncustom rend\ncustom reboot\ncustom lreboot\ncustom find <channel>");
    	PRINTF("\ncustom rhardwareversion\ncustom rsoftwareversion\n");
    }
    else if(strlen(token)==1){
    	// just carriage return. do nothing
    }
    else{
    	PRINTF("INVALID COMMAND:%s\n\r",token);
    }

    memset(sCommand.au8Buffer, 0, COMMAND_BUF_SIZE);
    sCommand.u8Pos = 0;

}

/****************************************************************************/
/***        END OF FILE                                                   ***/
/****************************************************************************/
