/*******************************************************************************
*
* Copyright 2004-2016, Freescale Semiconductor, Inc.
* Copyright 2016-2019 NXP
* All rights reserved.
*
********************************************************************************
*
* File Name:  ewtg.h
*
* Description: Header file for the EVTG driver
*
* EVTG_VERSION_1 ..  56F83xxx devices, see MC56F83xxxRM.pdf

* 
*******************************************************************************/

#ifndef __EVTG_H
#define __EVTG_H

/* qs.h is a master header file, which must be included */
#if !defined(__ARCH_H) || !defined(__PERIPH_H) || !defined(__APPCONFIG_H)
#error Please include qs.h before aoi.h
#endif

#ifdef __cplusplus
extern "C" {
#endif

/******************************************************************************
*
*                      General Interface Description
*
*
* The AND/OR/INVERT module (known simply as the EVTG module) supports the
* generation of a configurable number of EVENT signals. Each output EVENTn is a
* configurable and/or/invert function of four associated EVTG inputs: An, Bn, Cn, and Dn.
* This module is designed to be integrated in conjuction with one or more inter-peripheral
* crossbar switch (XBAR_DSC) modules. A crossbar switch is typically used to select the
* 4*n EVTG inputs from among available peripheral outputs and GPIO signals. The n
* EVENTn outputs from the EVTG module are typically used as additional inputs to a second
* crossbar switch, adding to it the ability to connect to its outputs an arbitrary 4-input
* boolean function of its other inputs.
*
******************************************************************************/

/************************************************************
* EVTG module identifiers
*************************************************************/

#ifdef EVTG_A_BASE
#define EVTG_A (&ArchIO.Evtg_A)
#endif

#ifdef EVTG_B_BASE
#define EVTG_B (&ArchIO.Evtg_B)
#endif

#ifdef EVTG_C_BASE
#define EVTG_C (&ArchIO.Evtg_C)
#endif

#ifdef EVTG_D_BASE
#define EVTG_D (&ArchIO.Evtg_D)
#endif

/*************************************************************
* arch.h should define device-specific information about EVTG
**************************************************************/

#if !defined(EVTG_VERSION)
#error EVTG not properly described in arch.h
#endif

/****************************************************
* Configurable items, i.e. defines for appconfig.h
*****************************************************

#define EVTG_A_AOI0_BFT01_INIT         0x0000U
#define EVTG_A_AOI0_BFT23_INIT         0x0000U
#define EVTG_A_AOI1_BFT01_INIT         0x0000U
#define EVTG_A_AOI1_BFT23_INIT         0x0000U
#define EVTG_A_CTRL_INIT               0x0000U
#define EVTG_A_AOI0_FILT_INIT          0x0000U
#define EVTG_A_AOI1_FILT_INIT          0x0000U
#define EVTG_B_AOI0_BFT01_INIT         0x0000U
#define EVTG_B_AOI0_BFT23_INIT         0x0000U
#define EVTG_B_AOI1_BFT01_INIT         0x0000U
#define EVTG_B_AOI1_BFT23_INIT         0x0000U
#define EVTG_B_CTRL_INIT               0x0000U
#define EVTG_B_AOI0_FILT_INIT          0x0000U
#define EVTG_B_AOI1_FILT_INIT          0x0000U
#define EVTG_C_AOI0_BFT01_INIT         0x0000U
#define EVTG_C_AOI0_BFT23_INIT         0x0000U
#define EVTG_C_AOI1_BFT01_INIT         0x0000U
#define EVTG_C_AOI1_BFT23_INIT         0x0000U
#define EVTG_C_CTRL_INIT               0x0000U
#define EVTG_C_AOI0_FILT_INIT          0x0000U
#define EVTG_C_AOI1_FILT_INIT          0x0000U
#define EVTG_D_AOI0_BFT01_INIT         0x0000U
#define EVTG_D_AOI0_BFT23_INIT         0x0000U
#define EVTG_D_AOI1_BFT01_INIT         0x0000U
#define EVTG_D_AOI1_BFT23_INIT         0x0000U
#define EVTG_D_CTRL_INIT               0x0000U
#define EVTG_D_AOI0_FILT_INIT          0x0000U
#define EVTG_D_AOI1_FILT_INIT          0x0000U

*/

/*****************************************************************
* ioctl() commands
*
*  Word16 ioctl( EVTG_x, Command_name, Parameter );
*
******************************************************************


 COMMAND NAME                      PARAMETERS, COMMENTS
--------------------------------------------------------*/

#define EVTG_INIT                   /* NULL, Initialize EVTG periheral registers using the appconfig.h _INIT values */

/* Write commands for EVTG registers */ 
#define EVTG_WRITE_AOI0_BFT01    /* UWord16, write in to EVTG Boolean Function Term 0 register (AOI0_BFT01->_) */
#define EVTG_WRITE_AOI0_BFT23    /* UWord16, write in to EVTG Boolean Function Term 0 register (AOI0_BFT23->_) */
#define EVTG_WRITE_AOI1_BFT01    /* UWord16, write in to EVTG Boolean Function Term 1 register (AOI1_BFT01->_) */
#define EVTG_WRITE_AOI1_BFT23    /* UWord16, write in to EVTG Boolean Function Term 1 register (AOI1_BFT23->_) */
#define EVTG_WRITE_CTRL          /* UWord16, write in to EVTG Control/Status register (CTRL->_) */
#define EVTG_WRITE_AOI0_FILT     /* UWord16, write in to AOI0 Input Filter register (AOI0_FILT->_) */
#define EVTG_WRITE_AOI1_FILT     /* UWord16, write in to AOI1 Input Filter register (AOI1_FILT->_) */

/* Read commands for EVTG registers */
#define EVTG_READ_AOI0_BFT01     /* NULL, return UWord16 value of EVTG Boolean Function Term 0 register (AOI0_BFT01->_) */
#define EVTG_READ_AOI0_BFT23     /* NULL, return UWord16 value of EVTG Boolean Function Term 0 register (AOI0_BFT23->_) */
#define EVTG_READ_AOI1_BFT01     /* NULL, return UWord16 value of EVTG Boolean Function Term 1 register (AOI1_BFT01->_) */
#define EVTG_READ_AOI1_BFT23     /* NULL, return UWord16 value of EVTG Boolean Function Term 1 register (AOI1_BFT23->_) */
#define EVTG_READ_CTRL           /* NULL, return UWord16 value of EVTG Control/Status register (CTRL->_) */
#define EVTG_READ_AOI0_FILT      /* NULL, return UWord16 value of AOI0 Input Filter register (AOI0_FILT->_) */
#define EVTG_READ_AOI1_FILT      /* NULL, return UWord16 value of AOI1 Input Filter register (AOI1_FILT->_) */

/* Commands for setup all inputs at once of the one product term */
/* for EVENT0 */
#define EVTG_SET_EVENT0_TERM_0   /* combination of EVTG_INPUT_n_xxx  (n = A/B /C/D; xxx = LOG0/INVERT/NOTINVER/LOG1 ), set all inputs at once of the Term 0 for EVENT0 */
#define EVTG_SET_EVENT0_TERM_1   /* combination of EVTG_INPUT_n_xxx  (n = A/B /C/D; xxx = LOG0/INVERT/NOTINVER/LOG1 ), set all inputs at once of the Term 1 for EVENT0 */
#define EVTG_SET_EVENT0_TERM_2   /* combination of EVTG_INPUT_n_xxx  (n = A/B /C/D; xxx = LOG0/INVERT/NOTINVER/LOG1 ), set all inputs at once of the Term 2 for EVENT0 */
#define EVTG_SET_EVENT0_TERM_3   /* combination of EVTG_INPUT_n_xxx  (n = A/B /C/D; xxx = LOG0/INVERT/NOTINVER/LOG1 ), set all inputs at once of the Term 3 for EVENT0 */

/* Commands for setup all inputs at once of the one product term */
/* for EVENT1 */
#define EVTG_SET_EVENT1_TERM_0   /* combination of EVTG_INPUT_n_xxx  (n = A/B /C/D; xxx = LOG0/INVERT/NOTINVER/LOG1 ), set all inputs at once of the Term 0 for EVENT1 */
#define EVTG_SET_EVENT1_TERM_1   /* combination of EVTG_INPUT_n_xxx  (n = A/B /C/D; xxx = LOG0/INVERT/NOTINVER/LOG1 ), set all inputs at once of the Term 1 for EVENT1 */
#define EVTG_SET_EVENT1_TERM_2   /* combination of EVTG_INPUT_n_xxx  (n = A/B /C/D; xxx = LOG0/INVERT/NOTINVER/LOG1 ), set all inputs at once of the Term 2 for EVENT1 */
#define EVTG_SET_EVENT1_TERM_3   /* combination of EVTG_INPUT_n_xxx  (n = A/B /C/D; xxx = LOG0/INVERT/NOTINVER/LOG1 ), set all inputs at once of the Term 3 for EVENT1 */


/* Commands for setup inputs of the each product term  separately */
/* for EVENT0 */
#define EVTG_SET_EVENT0_TERM_0_INPUT_A       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_A of the Term 0 for EVENT0 */
#define EVTG_SET_EVENT0_TERM_0_INPUT_B       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_B of the Term 0 for EVENT0 */
#define EVTG_SET_EVENT0_TERM_0_INPUT_C       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_C of the Term 0 for EVENT0 */
#define EVTG_SET_EVENT0_TERM_0_INPUT_D       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_D of the Term 0 for EVENT0 */
#define EVTG_SET_EVENT0_TERM_1_INPUT_A       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_A of the Term 1 for EVENT0 */
#define EVTG_SET_EVENT0_TERM_1_INPUT_B       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_B of the Term 1 for EVENT0 */
#define EVTG_SET_EVENT0_TERM_1_INPUT_C       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_C of the Term 1 for EVENT0 */
#define EVTG_SET_EVENT0_TERM_1_INPUT_D       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_D of the Term 1 for EVENT0 */
#define EVTG_SET_EVENT0_TERM_2_INPUT_A       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_A of the Term 2 for EVENT0 */
#define EVTG_SET_EVENT0_TERM_2_INPUT_B       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_B of the Term 2 for EVENT0 */
#define EVTG_SET_EVENT0_TERM_2_INPUT_C       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_C of the Term 2 for EVENT0 */
#define EVTG_SET_EVENT0_TERM_2_INPUT_D       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_D of the Term 2 for EVENT0 */
#define EVTG_SET_EVENT0_TERM_3_INPUT_A       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_A of the Term 3 for EVENT0 */
#define EVTG_SET_EVENT0_TERM_3_INPUT_B       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_B of the Term 3 for EVENT0 */
#define EVTG_SET_EVENT0_TERM_3_INPUT_C       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_C of the Term 3 for EVENT0 */
#define EVTG_SET_EVENT0_TERM_3_INPUT_D       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_D of the Term 3 for EVENT0 */

/* Commands for setup inputs of the each product term  */
/* for EVENT1 */
#define EVTG_SET_EVENT1_TERM_0_INPUT_A       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_A of the Term 0 for EVENT1 */
#define EVTG_SET_EVENT1_TERM_0_INPUT_B       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_B of the Term 0 for EVENT1 */
#define EVTG_SET_EVENT1_TERM_0_INPUT_C       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_C of the Term 0 for EVENT1 */
#define EVTG_SET_EVENT1_TERM_0_INPUT_D       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_D of the Term 0 for EVENT1 */
#define EVTG_SET_EVENT1_TERM_1_INPUT_A       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_A of the Term 1 for EVENT1 */
#define EVTG_SET_EVENT1_TERM_1_INPUT_B       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_B of the Term 1 for EVENT1 */
#define EVTG_SET_EVENT1_TERM_1_INPUT_C       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_C of the Term 1 for EVENT1 */
#define EVTG_SET_EVENT1_TERM_1_INPUT_D       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_D of the Term 1 for EVENT1 */
#define EVTG_SET_EVENT1_TERM_2_INPUT_A       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_A of the Term 2 for EVENT1 */
#define EVTG_SET_EVENT1_TERM_2_INPUT_B       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_B of the Term 2 for EVENT1 */
#define EVTG_SET_EVENT1_TERM_2_INPUT_C       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_C of the Term 2 for EVENT1 */
#define EVTG_SET_EVENT1_TERM_2_INPUT_D       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_D of the Term 2 for EVENT1 */
#define EVTG_SET_EVENT1_TERM_3_INPUT_A       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_A of the Term 3 for EVENT1 */
#define EVTG_SET_EVENT1_TERM_3_INPUT_B       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_B of the Term 3 for EVENT1 */
#define EVTG_SET_EVENT1_TERM_3_INPUT_C       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_C of the Term 3 for EVENT1 */
#define EVTG_SET_EVENT1_TERM_3_INPUT_D       /* EVTG_LOG0/EVTG_INVERT/EVTG_NOTINVERT/EVTG_LOG1, set INPUT_D of the Term 3 for EVENT1 */

/* Commands for setup event trigger */
#define EVTG_SET_INPUT_SYNC_MODE             /* combination of EVTG_INPUT_SYNC_n_xxx  (n = A/B /C/D; xxx = SYNC/ASYNC ), set all option at once of the synchronous/asynchronous mode of inputs.*/
#define EVTG_SET_FEEDBACK_OVERRIDE           /* one of EVTG_INPUT_FB_n  (n = A/B /C/D), set the EVTG output feedback override control.*/
#define EVTG_SET_FLIP_FLOP_MODE              /* one of EVTG_FLIP_FLOP_n  (n = BYPASS/RS_TRIG/T_FF/D_FF/JK_FF/LATCH), set Flip-Flop mode configure.*/
#define EVTG_SET_ENABLE_FF_INIT              /* NULL, Enable the flip-flop initialization*/
#define EVTG_SET_FLIP_FLOP_INIT              /* one of EVTG_FLIP_FLOP_INIT_n  (n = 0/1), Configure flip-flop initial value.*/

/* Commands for AOIx filter */
#define EVTG_SET_AOI0_FILT_COUNT             /* Uword 0-7, Input Filter Sample Count.*/
#define EVTG_SET_AOI0_FILT_PERIOD            /* Uword 0-255, Input Filter Sample Period.*/
#define EVTG_SET_AOI1_FILT_COUNT             /* Uword 0-7, Input Filter Sample Count.*/
#define EVTG_SET_AOI1_FILT_PERIOD            /* Uword 0-255, Input Filter Sample Period.*/


/***********************************************************************************************************
**  PARAMETERS DEFINITION                                                                        **
***********************************************************************************************************/

#define EVTG_INPUT_A_LOG0        0x0000
#define EVTG_INPUT_A_INVERT      0x0080
#define EVTG_INPUT_A_NOTINVERT   0x0040
#define EVTG_INPUT_A_LOG1        0x00C0

#define EVTG_INPUT_B_LOG0        0x0000
#define EVTG_INPUT_B_INVERT      0x0020
#define EVTG_INPUT_B_NOTINVERT   0x0010
#define EVTG_INPUT_B_LOG1        0x0030

#define EVTG_INPUT_C_LOG0        0x0000
#define EVTG_INPUT_C_INVERT      0x0008
#define EVTG_INPUT_C_NOTINVERT   0x0004
#define EVTG_INPUT_C_LOG1        0x000C

#define EVTG_INPUT_D_LOG0        0x0000
#define EVTG_INPUT_D_INVERT      0x0002
#define EVTG_INPUT_D_NOTINVERT   0x0001
#define EVTG_INPUT_D_LOG1        0x0003

#define EVTG_LOG0                0x0000
#define EVTG_INVERT              0x0002
#define EVTG_NOTINVERT           0x0001
#define EVTG_LOG1                0x0003

#define EVTG_INPUT_SYNC_X_MASK   0x0f00U
#define EVTG_INPUT_SYNC_A_SYNC   0x0100U
#define EVTG_INPUT_SYNC_A_ASYNC  0x0000U
#define EVTG_INPUT_SYNC_B_SYNC   0x0200U
#define EVTG_INPUT_SYNC_B_ASYNC  0x0000U
#define EVTG_INPUT_SYNC_C_SYNC   0x0400U
#define EVTG_INPUT_SYNC_C_ASYNC  0x0000U
#define EVTG_INPUT_SYNC_D_SYNC   0x0800U
#define EVTG_INPUT_SYNC_D_ASYNC  0x0000U

#define EVTG_INPUT_FB_MASK       0x00C0U
#define EVTG_INPUT_FB_A          0x0000U
#define EVTG_INPUT_FB_B          0x0040U
#define EVTG_INPUT_FB_C          0x0080U
#define EVTG_INPUT_FB_D          0x00C0U

#define EVTG_FLIP_FLOP_MASK      0x001CU
#define EVTG_FLIP_FLOP_BYPASS    0x0000U
#define EVTG_FLIP_FLOP_RS_TRIG   0x0004U
#define EVTG_FLIP_FLOP_T_FF      0x0008U
#define EVTG_FLIP_FLOP_D_FF      0x000CU
#define EVTG_FLIP_FLOP_JK_FF     0x0010U
#define EVTG_FLIP_FLOP_LATCH     0x0014U

#define EVTG_CTRL_INIT_EN_MASK   0x0002U
#define EVTG_FLIP_FLOP_INIT_MASK 0x0001U
#define EVTG_FLIP_FLOP_INIT_0    0x0000U
#define EVTG_FLIP_FLOP_INIT_1    0x0001U

#define EVTG_AOI0_FILT_FILT_CNT  0x0700U
#define EVTG_AOI0_FILT_FILT_PER  0x00FFU
#define EVTG_AOI1_FILT_FILT_CNT  0x0700U
#define EVTG_AOI1_FILT_FILT_PER  0x00FFU

#define EVTG_MSB_MASK            0xFF00
#define EVTG_LSB_MASK            0x00FF

#define EVTG_PT0_AC_MASK         0xC000
#define EVTG_PT0_BC_MASK         0x3000
#define EVTG_PT0_CC_MASK         0x0C00 
#define EVTG_PT0_DC_MASK         0x0300
#define EVTG_PT1_AC_MASK         0x00C0
#define EVTG_PT1_BC_MASK         0x0030
#define EVTG_PT1_CC_MASK         0x000C
#define EVTG_PT1_DC_MASK         0x0003

/****************************************************************************
* register and bit names used in the ioctl functions below
*****************************************************************************/

/* CTRL register */

/****************************************************************************
*  ioctl macro implementation
*****************************************************************************/

/* EVTG init */

void ewtgInit(arch_sEVTG *pEvtgBase);
#define ioctlEVTG_INIT(pEvtgBase, param) ewtgInit(pEvtgBase)


/*  control register */

#define ioctlSET_EVENT0_TERM_0(pEvtgBase, param) \
        periphBitGrpSR(EVTG_MSB_MASK , (param << 8), &((pEvtgBase)->aoi0_bft01))

#define ioctlSET_EVENT0_TERM_1(pEvtgBase, param) \
        periphBitGrpSR(EVTG_LSB_MASK , param , &((pEvtgBase)->aoi0_bft01))

#define ioctlSET_EVENT0_TERM_2(pEvtgBase, param) \
        periphBitGrpSR(EVTG_MSB_MASK , (param << 8) , &((pEvtgBase)->aoi0_bft23))

#define ioctlSET_EVENT0_TERM_3(pEvtgBase, param) \
        periphBitGrpSR(EVTG_LSB_MASK , param , &((pEvtgBase)->aoi0_bft23))

#define ioctlSET_EVENT1_TERM_0(pEvtgBase, param) \
        periphBitGrpSR(EVTG_MSB_MASK , (param << 8), &((pEvtgBase)->aoi1_bft01))

#define ioctlSET_EVENT1_TERM_1(pEvtgBase, param) \
        periphBitGrpSR(EVTG_LSB_MASK , param , &((pEvtgBase)->aoi1_bft01))

#define ioctlSET_EVENT1_TERM_2(pEvtgBase, param) \
        periphBitGrpSR(EVTG_MSB_MASK , (param << 8) , &((pEvtgBase)->aoi1_bft23))

#define ioctlSET_EVENT1_TERM_3(pEvtgBase, param) \
        periphBitGrpSR(EVTG_LSB_MASK , param , &((pEvtgBase)->aoi1_bft23))

/* write data to EVTG registers */
#define ioctlEVTG_WRITE_AOI0_BFT01(pEvtgBase, param) \
  periphMemWrite(param, &((pEvtgBase)->aoi0_bft01))

#define ioctlEVTG_WRITE_AOI0_BFT23(pEvtgBase, param) \
  periphMemWrite(param, &((pEvtgBase)->aoi0_bft23))

#define ioctlEVTG_WRITE_AOI1_BFT01(pEvtgBase, param) \
  periphMemWrite(param, &((pEvtgBase)->aoi1_bft01))

#define ioctlEVTG_WRITE_AOI1_BFT23(pEvtgBase, param) \
  periphMemWrite(param, &((pEvtgBase)->aoi1_bft23))

#define ioctlEVTG_WRITE_CTRL(pEvtgBase, param) \
  periphMemWrite(param, &((pEvtgBase)->ctrl))

#define ioctlEVTG_WRITE_AOI0_FILT(pEvtgBase, param) \
  periphMemWrite(param, &((pEvtgBase)->aoi0_filt))

#define ioctlEVTG_WRITE_AOI1_FILT(pEvtgBase, param) \
  periphMemWrite(param, &((pEvtgBase)->aoi1_filt))


/* return data from EVTG registers */
#define ioctlEVTG_READ_AOI0_BFT01(pEvtgBase, param) \
        periphMemRead(&((pEvtgBase)->aoi0_bft01))

#define ioctlEVTG_READ_AOI0_BFT23(pEvtgBase, param) \
        periphMemRead(&((pEvtgBase)->aoi0_bft23))

#define ioctlEVTG_READ_AOI1_BFT01(pEvtgBase, param) \
        periphMemRead(&((pEvtgBase)->aoi1_bft01))

#define ioctlEVTG_READ_AOI1_BFT23(pEvtgBase, param) \
        periphMemRead(&((pEvtgBase)->aoi1_bft23))

#define ioctlEVTG_READ_CTRL(pEvtgBase, param) \
        periphMemRead(&((pEvtgBase)->ctrl))

#define ioctlEVTG_READ_AOI0_FILT(pEvtgBase, param) \
        periphMemRead(&((pEvtgBase)->aoi0_filt))

#define ioctlEVTG_READ_AOI1_FILT(pEvtgBase, param) \
        periphMemRead(&((pEvtgBase)->aoi1_filt))

#define ioctlEVTG_SET_EVENT0_TERM_0(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT0_AC_MASK|EVTG_PT0_BC_MASK|EVTG_PT0_CC_MASK|EVTG_PT0_DC_MASK , (param)<<8, &((pEvtgBase)->aoi0_bft01))
#define ioctlEVTG_SET_EVENT0_TERM_1(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT1_AC_MASK|EVTG_PT1_BC_MASK|EVTG_PT1_CC_MASK|EVTG_PT1_DC_MASK , param, &((pEvtgBase)->aoi0_bft01))
#define ioctlEVTG_SET_EVENT0_TERM_2(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT0_AC_MASK|EVTG_PT0_BC_MASK|EVTG_PT0_CC_MASK|EVTG_PT0_DC_MASK , (param)<<8, &((pEvtgBase)->aoi0_bft23))
#define ioctlEVTG_SET_EVENT0_TERM_3(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT1_AC_MASK|EVTG_PT1_BC_MASK|EVTG_PT1_CC_MASK|EVTG_PT1_DC_MASK , param, &((pEvtgBase)->aoi0_bft23))

#define ioctlEVTG_SET_EVENT1_TERM_0(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT0_AC_MASK|EVTG_PT0_BC_MASK|EVTG_PT0_CC_MASK|EVTG_PT0_DC_MASK , (param)<<8, &((pEvtgBase)->aoi1_bft01))
#define ioctlEVTG_SET_EVENT1_TERM_1(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT1_AC_MASK|EVTG_PT1_BC_MASK|EVTG_PT1_CC_MASK|EVTG_PT1_DC_MASK , param, &((pEvtgBase)->aoi1_bft01))
#define ioctlEVTG_SET_EVENT1_TERM_2(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT0_AC_MASK|EVTG_PT0_BC_MASK|EVTG_PT0_CC_MASK|EVTG_PT0_DC_MASK , (param)<<8, &((pEvtgBase)->aoi1_bft23))
#define ioctlEVTG_SET_EVENT1_TERM_3(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT1_AC_MASK|EVTG_PT1_BC_MASK|EVTG_PT1_CC_MASK|EVTG_PT1_DC_MASK , param, &((pEvtgBase)->aoi1_bft23))


#define ioctlEVTG_SET_EVENT0_TERM_0_INPUT_A(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT0_AC_MASK , (param << 14) , &((pEvtgBase)->aoi0_bft01))

#define ioctlEVTG_SET_EVENT0_TERM_0_INPUT_B(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT0_BC_MASK , (param << 12) , &((pEvtgBase)->aoi0_bft01))

#define ioctlEVTG_SET_EVENT0_TERM_0_INPUT_C(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT0_CC_MASK , (param << 10) , &((pEvtgBase)->aoi0_bft01))

#define ioctlEVTG_SET_EVENT0_TERM_0_INPUT_D(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT0_DC_MASK , (param << 8) , &((pEvtgBase)->aoi0_bft01))


#define ioctlEVTG_SET_EVENT0_TERM_1_INPUT_A(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT1_AC_MASK , (param << 6) , &((pEvtgBase)->aoi0_bft01))

#define ioctlEVTG_SET_EVENT0_TERM_1_INPUT_B(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT1_BC_MASK , (param << 4) , &((pEvtgBase)->aoi0_bft01))

#define ioctlEVTG_SET_EVENT0_TERM_1_INPUT_C(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT1_CC_MASK , (param << 2) , &((pEvtgBase)->aoi0_bft01))

#define ioctlEVTG_SET_EVENT0_TERM_1_INPUT_D(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT1_DC_MASK , (param) , &((pEvtgBase)->aoi0_bft01))


#define ioctlEVTG_SET_EVENT0_TERM_2_INPUT_A(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT0_AC_MASK , (param << 14) , &((pEvtgBase)->aoi0_bft23))

#define ioctlEVTG_SET_EVENT0_TERM_2_INPUT_B(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT0_BC_MASK , (param << 12) , &((pEvtgBase)->aoi0_bft23))

#define ioctlEVTG_SET_EVENT0_TERM_2_INPUT_C(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT0_CC_MASK , (param << 10) , &((pEvtgBase)->aoi0_bft23))

#define ioctlEVTG_SET_EVENT0_TERM_2_INPUT_D(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT0_DC_MASK , (param << 8) , &((pEvtgBase)->aoi0_bft23))


#define ioctlEVTG_SET_EVENT0_TERM_3_INPUT_A(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT1_AC_MASK , (param << 6) , &((pEvtgBase)->aoi0_bft23))

#define ioctlEVTG_SET_EVENT0_TERM_3_INPUT_B(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT1_BC_MASK , (param << 4) , &((pEvtgBase)->aoi0_bft23))

#define ioctlEVTG_SET_EVENT0_TERM_3_INPUT_C(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT1_CC_MASK , (param << 2) , &((pEvtgBase)->aoi0_bft23))

#define ioctlEVTG_SET_EVENT0_TERM_3_INPUT_D(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT1_DC_MASK , (param) , &((pEvtgBase)->aoi0_bft23))



#define ioctlEVTG_SET_EVENT1_TERM_0_INPUT_A(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT0_AC_MASK , (param << 14) , &((pEvtgBase)->aoi1_bft01))

#define ioctlEVTG_SET_EVENT1_TERM_0_INPUT_B(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT0_BC_MASK , (param << 12) , &((pEvtgBase)->aoi1_bft01))

#define ioctlEVTG_SET_EVENT1_TERM_0_INPUT_C(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT0_CC_MASK , (param << 10) , &((pEvtgBase)->aoi1_bft01))

#define ioctlEVTG_SET_EVENT1_TERM_0_INPUT_D(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT0_DC_MASK , (param << 8) , &((pEvtgBase)->aoi1_bft01))


#define ioctlEVTG_SET_EVENT1_TERM_1_INPUT_A(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT1_AC_MASK , (param << 6) , &((pEvtgBase)->aoi1_bft01))

#define ioctlEVTG_SET_EVENT1_TERM_1_INPUT_B(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT1_BC_MASK , (param << 4) , &((pEvtgBase)->aoi1_bft01))

#define ioctlEVTG_SET_EVENT1_TERM_1_INPUT_C(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT1_CC_MASK , (param << 2) , &((pEvtgBase)->aoi1_bft01))

#define ioctlEVTG_SET_EVENT1_TERM_1_INPUT_D(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT1_DC_MASK , (param) , &((pEvtgBase)->aoi1_bft01))


#define ioctlEVTG_SET_EVENT1_TERM_2_INPUT_A(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT0_AC_MASK , (param << 14) , &((pEvtgBase)->aoi1_bft23))

#define ioctlEVTG_SET_EVENT1_TERM_2_INPUT_B(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT0_BC_MASK , (param << 12) , &((pEvtgBase)->aoi1_bft23))

#define ioctlEVTG_SET_EVENT1_TERM_2_INPUT_C(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT0_CC_MASK , (param << 10) , &((pEvtgBase)->aoi1_bft23))

#define ioctlEVTG_SET_EVENT1_TERM_2_INPUT_D(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT0_DC_MASK , (param << 8) , &((pEvtgBase)->aoi1_bft23))


#define ioctlEVTG_SET_EVENT1_TERM_3_INPUT_A(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT1_AC_MASK , (param << 6) , &((pEvtgBase)->aoi1_bft23))

#define ioctlEVTG_SET_EVENT1_TERM_3_INPUT_B(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT1_BC_MASK , (param << 4) , &((pEvtgBase)->aoi1_bft23))

#define ioctlEVTG_SET_EVENT1_TERM_3_INPUT_C(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT1_CC_MASK , (param << 2) , &((pEvtgBase)->aoi1_bft23))

#define ioctlEVTG_SET_EVENT1_TERM_3_INPUT_D(pEvtgBase, param) \
        periphBitGrpSR(EVTG_PT1_DC_MASK , (param) , &((pEvtgBase)->aoi1_bft23))

#define ioctlEVTG_SET_INPUT_SYNC_MODE(pEvtgBase, param) \
        periphBitGrpSet(EVTG_INPUT_SYNC_X_MASK , (param) , &((pEvtgBase)->ctrl))

#define ioctlEVTG_SET_FEEDBACK_OVERRIDE(pEvtgBase, param) \
        periphBitGrpSet(EVTG_INPUT_FB_MASK , (param) , &((pEvtgBase)->ctrl))

#define ioctlEVTG_SET_FLIP_FLOP_MODE(pEvtgBase, param) \
        periphBitGrpSet(EVTG_FLIP_FLOP_MASK , (param) , &((pEvtgBase)->ctrl))

#define ioctlEVTG_SET_ENABLE_FF_INIT(pEvtgBase, param) \
        periphBitSet(EVTG_CTRL_INIT_EN_MASK, &((pEvtgBase)->ctrl))

#define ioctlEVTG_SET_FLIP_FLOP_INIT(pEvtgBase, param) \
  if(param) periphBitSet(EVTG_FLIP_FLOP_INIT_MASK, &((pEvtgBase)->ctrl)); \
  else periphBitClear(EVTG_FLIP_FLOP_INIT_MASK, &((pEvtgBase)->ctrl))

#define ioctlEVTG_SET_AOI0_FILT_COUNT(pEvtgBase, param) \
        periphBitGrpSet(EVTG_AOI0_FILT_FILT_CNT, (param)<<8 , &((pEvtgBase)->aoi0_filt))

#define ioctlEVTG_SET_AOI0_FILT_PERIOD(pEvtgBase, param) \
        periphBitGrpSet(EVTG_AOI0_FILT_FILT_PER, (param) , &((pEvtgBase)->aoi0_filt))

#define ioctlEVTG_SET_AOI1_FILT_COUNT(pEvtgBase, param) \
        periphBitGrpSet(EVTG_AOI1_FILT_FILT_CNT, (param)<<8 , &((pEvtgBase)->aoi1_filt))

#define ioctlEVTG_SET_AOI1_FILT_PERIOD(pEvtgBase, param) \
        periphBitGrpSet(EVTG_AOI1_FILT_FILT_PER, (param) , &((pEvtgBase)->aoi1_filt))

#ifdef __cplusplus
}
#endif

#endif
