/*
 * @brief LPC5410x Power library (hidden)
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2016
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */

#include "chip.h"
#include <stdlib.h>
#include <string.h>

/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/
typedef struct
{
  volatile unsigned int SYSMEMREMAP                              ; ///< System Remap                (0x000)
  volatile unsigned int RESERVED0[3]                             ;
  volatile unsigned int AHBMATPRIO                               ; ///< AHB Matrix Priority         (0x010)
  volatile unsigned int RESERVED1[11]                            ;
  volatile unsigned int SYSTCKCAL                                ; ///< System Tick Calibration     (0x040)
  volatile unsigned int RESERVED2                                ;
  volatile unsigned int NMISRC                                   ; ///< NMI Source Select           (0x048)
  volatile unsigned int ASYNCVPBCTRL                             ; ///< Asynch VPB Chiplet Control  (0x04C)
  volatile unsigned int RESERVED3[28]                            ;
  volatile unsigned int PIOPORCAP[2]                             ; ///< PIO Power-On Reset Capture  (0x0C0 - 0x0C4)
  volatile unsigned int RESERVED4[2]                             ;
  volatile unsigned int PIORESCAP[2]                             ; ///< PIO Pad Reset Capture       (0x0D0 - 0x0D4)
  volatile unsigned int RESERVED5[10]                            ;
  volatile unsigned int PRESETCTRL[2]                            ; ///< Peripheral Reset Ctrl       (0x100 - 0x104)
  volatile unsigned int RESERVED6[6]                             ;
  volatile unsigned int PRESETCTRLSET[2]                         ; ///< Peripheral Reset Ctrl Set   (0x120 - 0x124)
  volatile unsigned int RESERVED7[6]                             ;
  volatile unsigned int PRESETCTRLCLR[2]                         ; ///< Peripheral Reset Ctrl Clr   (0x140 - 0x144)
  volatile unsigned int RESERVED8[42]                            ;
  volatile unsigned int SYSRESSTAT                               ; ///< System Reset Stat           (0x1F0)
  volatile unsigned int RESERVED9[3]                             ;
  volatile unsigned int SYSAHBCLKCTRL[2]                         ; ///< AHB Peripheral Clk Enable   (0x200 - 0x204)
  volatile unsigned int RESERVED10[6]                            ;
  volatile unsigned int SYSAHBCLKCTRLSET[2]                      ; ///< AHB Peripheral Clk Enable Set(0x220 - 0x224)
  volatile unsigned int RESERVED11[6]                            ;
  volatile unsigned int SYSAHBCLKCTRLCLR[2]                      ; ///< AHB Peripheral Clk Enable Clr(0x240 - 0x244)
  volatile unsigned int RESERVED12[14]                           ;
  volatile unsigned int MAINCLKSELA                              ; ///< Main Clk Sel Source Sel A   (0x280)
  volatile unsigned int MAINCLKSELB                              ; ///< Main Clk Sel Source Sel B   (0x284)
  volatile unsigned int CLKOUTCLKSELA                            ; ///< Clk Out Sel Source A        (0x288)
  volatile unsigned int RESERVED13                               ;
  volatile unsigned int SYSPLLCLKSEL                             ; ///< System PLL Clk Sel          (0x290)
  volatile unsigned int RESERVED14[4]                            ;
  volatile unsigned int ADCASYNCCLKSEL                           ; ///< ADC Async Clk Sel           (0x2A4)
  volatile unsigned int USBCLKSEL                                ; ///< USB0 Clk Sel                (0x2A8)
  volatile unsigned int RESERVED15                               ;
  volatile unsigned int FXCOMCLKSEL[8]                           ; ///< Flexcomm Clk Sel            (0x2B0 - 0x2CC)
  volatile unsigned int RESERVED16[4]                            ;
  volatile unsigned int FXI2S0MCLKCLKSEL                         ; ///< I2S0 MClk Sel               (0x2E0)
  volatile unsigned int RESERVED17                               ;
  volatile unsigned int FRGCLKSEL                                ; ///< frg Clk Sel                 (0x2E8)
  volatile unsigned int DMICCLKSEL                               ; ///< Dmic Clk Sel                (0x2EC)
  volatile unsigned int RESERVED18[4]                            ;
  volatile unsigned int SYSTICKCLKDIV                            ; ///< Systick Clock Divider       (0x300)
  volatile unsigned int TRACECLKDIV                              ; ///< Trace Clk Divider           (0x304)
  volatile unsigned int RESERVED19[30]                           ;
  volatile unsigned int SYSAHBCLKDIV                             ; ///< Main Clk Divider            (0x380)
  volatile unsigned int CLKOUTDIV                                ; ///< Clk Out Divider             (0x384)
  volatile unsigned int RESERVED20[3]                            ;
  volatile unsigned int ADCASYNCCLKDIV                           ; ///< ADC Async Clk Divider       (0x394)
  volatile unsigned int USBCLKDIV                                ; ///< USB0 Clk Divider            (0x398)
  volatile unsigned int RESERVED21                               ;
  volatile unsigned int FRGCTRL                                  ; ///< Fraction Rate Generator Ctrl(0x3A0)
  volatile unsigned int RESERVED22                               ;
  volatile unsigned int DMICCLKDIV                               ; ///< Dmic Clk Div                (0x3A8)
  volatile unsigned int I2SMCLKDIV                               ; ///< I2S0 MClk Div               (0x3AC)
  volatile unsigned int RESERVED23[20]                           ;
  volatile unsigned int FLASHCFG                                 ; ///< Flash Config                (0x400)
  volatile unsigned int RESERVED24[2]                            ;
  volatile unsigned int USBCLKCTRL                               ; ///< USB Clock Control           (0x40C)
  volatile unsigned int USBCLKSTAT                               ; ///< Usb Clock Stat              (0x410)
  volatile unsigned int RESERVED25                               ;
  volatile unsigned int FREQMECTRL                               ; ///< Frequency Measure Control   (0x418)
  volatile unsigned int RESERVED26                               ;
  volatile unsigned int MCLKIO                                   ; ///< MClk I/O                    (0x420)
  volatile unsigned int RESERVED27[55]                           ;
  volatile unsigned int FROCTRL                                  ; ///< FRO Oscillator Control      (0x500)
  volatile unsigned int RESERVED28                               ;
  volatile unsigned int WDTOSCCTRL                               ; ///< Watchdog Oscillator Control (0x508)
  volatile unsigned int RTCOSCCTRL                               ; ///< RTC Oscillator Control      (0x50C)
  volatile unsigned int RESERVED29[28]                           ;
  volatile unsigned int SYSPLLCTRL                               ; ///< SYS PLL CTRL                (0x580)
  volatile unsigned int SYSPLLSTAT                               ; ///< SYS PLL STAT                (0x584)
  volatile unsigned int SYSPLLNDEC                               ; ///< SYS PLL NDEC                (0x588)
  volatile unsigned int SYSPLLPDEC                               ; ///< SYS Pll PDEC                (0x58C)
  volatile unsigned int SYSPLLSSCGCTRL[2]                        ; ///< Spread Spectrum Control     (0x590 - 0x594)
  volatile unsigned int RESERVED30[26]                           ;
  volatile unsigned int PDSLEEPCFG[2]                            ; ///< Power Down Sleep Config     (0x600 - 0x604)
  volatile unsigned int RESERVED31[2]                            ;
  volatile unsigned int PDRUNCFG[2]                              ; ///< Power Down Run Config       (0x610 - 0x614)
  volatile unsigned int RESERVED32[2]                            ;
  volatile unsigned int PDRUNCFGSET[2]                           ; ///< Power Down Run Config Set   (0x620 - 0x624)
  volatile unsigned int RESERVED33[2]                            ;
  volatile unsigned int PDRUNCFGCLR[2]                           ; ///< Power Down Run Config Clr   (0x630 - 0x634)
  volatile unsigned int RESERVED34[18]                           ;
  volatile unsigned int STARTER[2]                               ; ///< Start Signal Enable Register(0x680 - 0x684)
  volatile unsigned int RESERVED35[6]                            ;
  volatile unsigned int STARTERSET[2]                            ; ///< Start Signal Enable Set Register (0x6A0 - 0x6A4)
  volatile unsigned int RESERVED36[6]                            ;
  volatile unsigned int STARTERCLR[2]                            ; ///< Start Signal Enable Clr Register (0x6C0 - 0x6C4)
  volatile unsigned int RESERVED37[14]                           ;
  volatile unsigned int IRCPDSTRETCH                             ; ///< Stretch Entry into PD using IRC(0x700)
  volatile unsigned int IRCSAFETY                                ; ///< IRC Safety Wakeup Counter Ctrl (0x704)
  volatile unsigned int RESERVED38[30]                           ;
  volatile unsigned int HWWAKE                                   ; ///< HW Wake                     (0x780)
  volatile unsigned int RESERVED39[31]                           ;
  volatile unsigned int CPUCTRL                                  ; ///< CPU CTRL                    (0x800)
  volatile unsigned int CPBOOT                                   ; ///< COPROCESSOR BOOT            (0x804)
  volatile unsigned int CPSTACK                                  ; ///< COPROCESSOR STACK           (0x808)
  volatile unsigned int CPUSTAT                                  ; ///< CPU STAT                    (0x80C)
  volatile unsigned int RESERVED40[60]                           ;
  volatile unsigned int RESERVED41[64]                           ;
	volatile unsigned int RESERVED42[64]                           ;
	volatile unsigned int RESERVED43[64]                           ;
	volatile unsigned int RESERVED44[64]                           ;
	volatile unsigned int RESERVED45[65]                           ;
  volatile unsigned int AUTOCGOR                                 ; ///< Auto Clockgating Override   (0xE04)
	volatile unsigned int RESERVED46[123]                          ;
  volatile unsigned int JTAG_IDCODE                              ; ///< JTAG ID Code Register       (0xFF4)
  volatile unsigned int DEVICE_ID0                               ; ///< Part ID Register            (0xFF8)
  volatile unsigned int DEVICE_ID1                               ; ///< Boot ROM Die Revision Register(0xFFC)
} ROM_SYSCON_T;

#define ROM_SYSCON         ((ROM_SYSCON_T          *) LPC_SYSCON_BASE)

/** @brief The real 'PMU/POWER', private
 */
typedef volatile struct {
  volatile uint32_t VDCTRL[4]                                    ; ///< VD1, VD2, VD3, VD8 Domain Voltage Control (0x00 - 0x0C)
  volatile uint32_t RESERVED0[12]                                ;
  volatile uint32_t LPCTRL                                       ; ///< LP VD Control               (0x40)
} ROM_POWER_T;

#define ROM_POWER          ((ROM_POWER_T            *) LPC_PMU_BASE)

#define MAX_CLOCK_KHZ  (100000)

/* Power domains */
typedef enum  {
	POWER_VD1 = 0x0,
	POWER_VD2 = 0x1,
	POWER_VD3 = 0x2,
	POWER_VD8 = 0x3
} POWER_DOMAIN_T;

/* Power coarse voltage levels */
typedef enum {
	POWER_V0650 = 0x0,
	POWER_V0700 = 0x1,
	POWER_V0750 = 0x2,
	POWER_V0800 = 0x3,
	POWER_V0850 = 0x4,
	POWER_V0900 = 0x5,
	POWER_V0950 = 0x6,
	POWER_V1000 = 0x7,
	POWER_V1050 = 0x8,
	POWER_V1100 = 0x9,
	POWER_V1150 = 0xA,
	POWER_V1200 = 0xB,
	POWER_V1250 = 0xC,
	POWER_V1300 = 0xD,
	POWER_V1350 = 0xE,
	POWER_V1400 = 0xF
} POWER_VOLTAGELEVEL_T;

/* Low power voltage selection */
typedef enum {
	POWER_LP_V0700 = 0x0,
	POWER_LP_V1200 = 0x1
} POWER_LPVOLTAGELEVEL_T;

/* Low power fine voltage selection */
typedef enum {
	POWER_FINE_LP_V_M050  = 0x0,
	POWER_FINE_LP_V_NONE  = 0x1,
	POWER_FINE_LP_V_P050  = 0x2,
	POWER_FINE_LP_V_P100  = 0x3
} POWER_LPFINEVOLTAGELEVEL_T;

/* PDRUNCFG bit definitions */
#define PDRUNCFG_MAINCLK_SHUTOFF  (1 << 0)
#define PDRUNCFG_DEEP_PD          (1 << 1)
#define PDRUNCFG_LP_VD1           (1 << 2)
#define PDRUNCFG_PD_FRO_OSC_EN    (1 << 3)	/* Not used per feedback from H/W team. */
#define PDRUNCFG_PD_FRO_EN        (1 << 4)
#define PDRUNCFG_PD_FLASH         (1 << 5)
#define PDRUNCFG_PD_TS            (1 << 6)
#define PDRUNCFG_PD_BOD_RESET     (1 << 7)
#define PDRUNCFG_PD_BOD_INTR      (1 << 8)
#define PDRUNCFG_PD_VD2_ANA       (1 << 9)
#define PDRUNCFG_PD_ADC0          (1 << 10)
#define PDRUNCFG_PD_VDDFLASH_ENA  (1 << 11)
#define PDRUNCFG_LP_VDDFLASH      (1 << 12)
#define PDRUNCFG_PD_RAM0          (1 << 13)
#define PDRUNCFG_PD_RAM1          (1 << 14)
#define PDRUNCFG_PD_RAM2          (1 << 15)
#define PDRUNCFG_PD_RAMX          (1 << 16)
#define PDRUNCFG_PD_ROM           (1 << 17)
#define PDRUNCFG_PD_VDDHV_ENA     (1 << 18)
#define PDRUNCFG_PD_VD7_ENA       (1 << 19)
#define PDRUNCFG_PD_WDT_OSC       (1 << 20)
#define PDRUNCFG_PD_USB0_PHY      (1 << 21)
#define PDRUNCFG_PD_SYS_PLL0      (1 << 22)
#define PDRUNCFG_PD_VREFP_SW      (1 << 23)
#define PDRUNCFG_PD_32K_OSC       (1 << 24)
#define PDRUNCFG_PD_FLASH_BG      (1 << 25)
#define PDRUNCFG_PD_VD3           (1 << 26)
#define PDRUNCFG_LP_VD2           (1 << 27)
#define PDRUNCFG_LP_VD3           (1 << 28)
#define PDRUNCFG_LP_VD8           (1UL << 29)
#define PDRUNCFG_REQ_DELAY        (1UL << 30)
#define PDRUNCFG_FORCE_RBB        (1UL << 31)

#define PDRUNCFG1_PD_ALT_FLASH_IBG  (1UL << 28)
#define PDRUNCFG1_SEL_ALT_FLASH_IBG (1UL << 29)

#define PCFG_DEEP_SLEEP  (   PDRUNCFG_MAINCLK_SHUTOFF   \
                            | PDRUNCFG_LP_VD1            \
                            | PDRUNCFG_PD_FRO_EN         \
                            | PDRUNCFG_PD_FLASH         \
                            | PDRUNCFG_PD_TS             \
                            | PDRUNCFG_PD_BOD_RESET      \
                            | PDRUNCFG_PD_BOD_INTR       \
                            | PDRUNCFG_PD_VD2_ANA        \
                            | PDRUNCFG_PD_ADC0           \
                            | PDRUNCFG_PD_VDDFLASH_ENA   \
                            | PDRUNCFG_LP_VDDFLASH       \
                            | PDRUNCFG_PD_RAM1           \
                            | PDRUNCFG_PD_RAM2           \
                            | PDRUNCFG_PD_RAMX           \
                            | PDRUNCFG_PD_ROM            \
                            | PDRUNCFG_PD_VDDHV_ENA      \
                            | PDRUNCFG_PD_VD7_ENA        \
                            | PDRUNCFG_PD_WDT_OSC        \
                            | PDRUNCFG_PD_USB0_PHY       \
                            | PDRUNCFG_PD_SYS_PLL0       \
                            | PDRUNCFG_PD_VREFP_SW       \
                            | PDRUNCFG_PD_FLASH_BG       \
                            | PDRUNCFG_LP_VD2            \
                            | PDRUNCFG_LP_VD3            \
                            | PDRUNCFG_LP_VD8            \
                            | PDRUNCFG_REQ_DELAY         \
                            | PDRUNCFG_FORCE_RBB         \
                            | PDRUNCFG_PD_32K_OSC        \
                          )
                      
#define PERIPH_CTRL_MASK_DEEP_SLEEP (PDRUNCFG_PD_FRO_EN | PDRUNCFG_PD_BOD_RESET | PDRUNCFG_PD_BOD_INTR | PDRUNCFG_PD_WDT_OSC | \
                                     PDRUNCFG_PD_RAM1 | PDRUNCFG_PD_RAM2 | PDRUNCFG_PD_RAMX | PDRUNCFG_PD_USB0_PHY | PDRUNCFG_PD_32K_OSC)

#define PCFG_DEEP_POWERDOWN 0xFFFFFFFF

#define POWER_NORMAL_VLEVEL(X)    ((X & 0xf)<<0)

#define EN0_FLASH                 (1 << 7)

typedef struct {
	uint32_t data[9];
} pwr_vd_table_t;

static const pwr_vd_table_t vd_table[1] = {
	/* LOW POWER mode */
	{
			/* VD1				VD2						VD3					VD8					WS */
		/* 0~12 Mhz */
		{
			POWER_V0950 | (POWER_V1200 << 6) | (POWER_V1200 << 12) | (POWER_V0950 << 18) | (0 << 24),

			/* 12~24 Mhz */
			POWER_V1000 | (POWER_V1200 << 6) | (POWER_V1200 << 12) | (POWER_V1000 << 18) | (1 << 24),

			/* 24~36 Mhz */
			POWER_V1050 | (POWER_V1200 << 6) | (POWER_V1200 << 12) | (POWER_V1050 << 18) | (2 << 24),

			/* 36~48 Mhz */
			POWER_V1100 | (POWER_V1200 << 6) | (POWER_V1200 << 12) | (POWER_V1100 << 18) | (2 << 24),

			/* 48~60 Mhz */
			POWER_V1150 | (POWER_V1200 << 6) | (POWER_V1200 << 12) | (POWER_V1150 << 18) | (3 << 24),

			/* 60~72 Mhz */
			POWER_V1200 | (POWER_V1200 << 6) | (POWER_V1200 << 12) | (POWER_V1200 << 18) | (3 << 24),

			/* 72~84 Mhz */
			POWER_V1200 | (POWER_V1200 << 6) | (POWER_V1200 << 12) | (POWER_V1200 << 18) | (4 << 24),

			/* 84~96 Mhz */
			POWER_V1250 | (POWER_V1200 << 6) | (POWER_V1250 << 12) | (POWER_V1250 << 18) | (5 << 24),

			/* 96~100 Mhz */
			POWER_V1300 | (POWER_V1200 << 6) | (POWER_V1300 << 12) | (POWER_V1300 << 18) | (5 << 24),

		}

	}
};
/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/

/*****************************************************************************
 * Private functions
 ****************************************************************************/

/* Return the boot ROM version */
static uint32_t getBROMVersion(void)
{
	uint32_t command[5], result[4];

	command[0] = IAP_READ_BOOT_CODE_CMD;
	iap_entry(command, result);

	return result[1];
}

void set_vd_level (uint32_t domain, uint32_t level) 
{
  ROM_POWER->VDCTRL[domain] = POWER_NORMAL_VLEVEL(level);
}

static void __set_pmu_sleep(void)
{
	SCB->SCR &= ~SCB_SCR_SLEEPDEEP_Msk;
	__WFI();
}

static void __set_pmu_deep_sleep(uint32_t peripheral_ctrl)
{
	uint32_t pdruncfg0_val, sysahbclkctrl_val, pmsk;
	uint32_t i;
	volatile uint32_t j;
	volatile uint32_t temp = temp;

	pmsk = __get_PRIMASK();
	__disable_irq();

	SCB->SCR |= SCB_SCR_SLEEPDEEP_Msk;

	pdruncfg0_val = ROM_SYSCON->PDRUNCFG[0];

	ROM_SYSCON->PDSLEEPCFG[0] = (PCFG_DEEP_SLEEP | pdruncfg0_val) & ~(peripheral_ctrl & PERIPH_CTRL_MASK_DEEP_SLEEP) & ~PDRUNCFG_PD_FRO_OSC_EN;

	sysahbclkctrl_val = ROM_SYSCON->SYSAHBCLKCTRL[0];
	/* TURN ON clock for flash controller */
	ROM_SYSCON->SYSAHBCLKCTRLSET[0] = EN0_FLASH;
	/* Turn VDDHV off and assert FLASH PD */
	ROM_SYSCON->PDRUNCFGSET[0]      = PDRUNCFG_PD_VDDHV_ENA | PDRUNCFG_PD_FLASH;
	/* Assert flash controller reset */
	ROM_SYSCON->PRESETCTRLSET[0]    = EN0_FLASH;

	/* Enter powerdown mode */
	__WFI();

	if (!(pdruncfg0_val & PDRUNCFG_PD_FLASH)) {
		/* Turn VDDHV on and deassert FLASH PD */
		ROM_SYSCON->PDRUNCFGCLR[0] = PDRUNCFG_PD_FLASH | PDRUNCFG_PD_VDDHV_ENA;
		/* 150usec start up time for flash bandgap */
#if   defined ( __CC_ARM ) /*------------------RealView Compiler -----------------*/
		/* Optimization -O3 */
		for (i = 0; i <= 226; i++) {
#elif defined ( __ICCARM__ ) /*------------------ ICC Compiler -------------------*/
		/* Optimization High Balanced */
		for (i = 0; i <= 227; i++) {
#elif defined ( __GNUC__ ) /*------------------ GNU Compiler ---------------------*/
		/* Optimization -Os */
		for (i = 0; i <= 224; i++) {
#endif
			j = i;
			__NOP();
		}
	}

	/* Deassert flash controller reset */
	ROM_SYSCON->PRESETCTRLCLR[0] = EN0_FLASH;

	/* Restore original pdruncfg */
	ROM_SYSCON->PDRUNCFG[0] = pdruncfg0_val & ~PDRUNCFG_PD_FRO_OSC_EN;

	/* Dummy read back flash to know that initialisation is completed. Any location above vector table */
	temp = *(volatile uint32_t *) 0x1000;
	if (!(sysahbclkctrl_val & EN0_FLASH)) {
		/* TURN OFF clock for flash controller */
		ROM_SYSCON->SYSAHBCLKCTRLCLR[0] = EN0_FLASH;
	}

	SCB->SCR &= ~SCB_SCR_SLEEPDEEP_Msk;

	__set_PRIMASK(pmsk);
}

static void __set_pmu_deep_powerdown(uint32_t peripheral_ctrl)
{
	/* System can only exit deep powerdown from wakeup reset */
	__disable_irq();
	SCB->SCR |= SCB_SCR_SLEEPDEEP_Msk;

	/* The system will enter powerdown on the following call */
	/* Turn off everything except the blocks excluded */
	ROM_SYSCON->PDRUNCFG[0] = PCFG_DEEP_POWERDOWN & ~(peripheral_ctrl & PDRUNCFG_PD_32K_OSC);

	/* It shouldn't make it here */
	__WFI();
}

/*****************************************************************************
 * Public functions
 ****************************************************************************/

#define TRIM_ADDR_48MHZ     (0x01000444)
#define TRIM_ADDR_96MHZ     (0x01000430)

/* Sets the High Frequency FRO rate to (48MHz or 96MHz) */
void Chip_POWER_SetFROHFRate(uint32_t freq)
{
	if(freq == SYSCON_FRO48MHZ_FREQ) {
		ROM_SYSCON->FROCTRL = ((*(uint32_t *) TRIM_ADDR_48MHZ) & ~(SYSCON_FROCTRL_MASK | SYSCON_FROCTRL_SEL96MHZ)) | SYSCON_FROCTRL_WRTRIM | SYSCON_FROCTRL_HSPDCLK;
	}
	else if(freq == SYSCON_FRO96MHZ_FREQ) {
		ROM_SYSCON->FROCTRL = ((*(uint32_t *) TRIM_ADDR_96MHZ) & ~SYSCON_FROCTRL_MASK) | SYSCON_FROCTRL_SEL96MHZ | SYSCON_FROCTRL_WRTRIM | SYSCON_FROCTRL_HSPDCLK;
	}
}

/* Sets up the System PLL given the PLL input frequency and feedback multiplier */
uint32_t Chip_POWER_SetPLL(uint32_t multiply_by, uint32_t input_freq)
{
	if ( (multiply_by == 0) || (((multiply_by*input_freq)/1000) > MAX_CLOCK_KHZ) ) {
		return ( ERR_CLK_INVALID_PARAM );
	}
	/* First parameter is the multiplier, the second parameter is
	   the input frequency in MHz */
	Chip_Clock_SetupSystemPLL(multiply_by, input_freq);

	/* Turn on the PLL by clearing the power down bit */
	Chip_SYSCON_PowerUp(SYSCON_PDRUNCFG_PD_SYS_PLL);

	/* Wait for PLL to lock */
	while (!Chip_Clock_IsSystemPLLLocked()) {}

	return LPC_OK;
}

/* Set optimal system voltage based on passed system frequency */
uint32_t Chip_POWER_SetVoltage(uint32_t desired_freq)
{
	uint32_t flashcfg_def;
	uint32_t volt_configure;

	/* Check the requested frequency to be sure it is in range */
	if ( (desired_freq/1000) > MAX_CLOCK_KHZ ) {
		return ( ERR_CLK_INVALID_PARAM );
	}

	ROM_SYSCON->PDRUNCFGCLR[0] = PDRUNCFG_LP_VD1 | PDRUNCFG_LP_VD2 | PDRUNCFG_LP_VD3 | PDRUNCFG_LP_VD8;
	flashcfg_def = ROM_SYSCON->FLASHCFG & ~(0xF<<12);

	if ( desired_freq <= 12000000 ) {
		volt_configure = vd_table[0].data[0];
	} else if ( desired_freq <= 24000000 ) {
		volt_configure = vd_table[0].data[1];
	} else if ( desired_freq <= 36000000 ) {
		volt_configure = vd_table[0].data[2];
	} else if ( desired_freq <= 48000000 ) {
		volt_configure = vd_table[0].data[3];
	} else if ( desired_freq <= 60000000 ) {
		volt_configure = vd_table[0].data[4];
	} else if ( desired_freq <= 72000000 ) {
		volt_configure = vd_table[0].data[5];
	} else if ( desired_freq <= 84000000 ) {
		volt_configure = vd_table[0].data[6];
	} else if ( desired_freq <= 96000000 ) {
		volt_configure = vd_table[0].data[7];
	} else {
		volt_configure = vd_table[0].data[8];
	}

	/* This check handles the case that the vd_table data is not valid.
	 * shouldn't happen in this library, but leaving for completeness. */
	if ( (volt_configure == 0x0) || ( volt_configure == 0xFFFFFFFF) ) {
		/* If the table (used to be the index sector) is not configured correctly, set the highest voltage and longest wait state. */
		set_vd_level(POWER_VD1, POWER_V1150);
		set_vd_level(POWER_VD2, POWER_V1200);
		set_vd_level(POWER_VD3, POWER_V1200);
		set_vd_level(POWER_VD8, POWER_V1100);

		/* Set wait states to 6 */
		ROM_SYSCON->FLASHCFG = flashcfg_def | (0x6<<12);
	}
	else {
		set_vd_level(POWER_VD1, (volt_configure & 0xF));
		set_vd_level(POWER_VD2, ((volt_configure>>6) & 0xF));
		set_vd_level(POWER_VD3, ((volt_configure>>12) & 0xF));
		set_vd_level(POWER_VD8, ((volt_configure>>18) & 0xF));

		/* Set wait states */
		ROM_SYSCON->FLASHCFG = flashcfg_def | (((volt_configure>>24)&0xF)<<12);
	}
	return ( LPC_OK );
}

/* Enter sleep/powerdown mode and prepare TBD */
void Chip_POWER_EnterPowerMode(POWER_MODE_T mode, uint32_t peripheral_ctrl)
{
	static uint32_t relocMem[512];
	void (*fptr)(uint32_t);
    
	switch (mode) {
	case POWER_SLEEP:
		__set_pmu_sleep();
		break;

	case POWER_DEEP_SLEEP:
		/* relocate to RAM so that Flash can be put to standy mode */
		memcpy(relocMem, (void *)((uint32_t)__set_pmu_deep_sleep & ~3), sizeof(relocMem));
		fptr = (void (*)(uint32_t))((uint32_t)relocMem + ((uint32_t)__set_pmu_deep_sleep & 3U));
		fptr(peripheral_ctrl);
		break;

	case POWER_DEEP_POWER_DOWN:
		__set_pmu_deep_powerdown(peripheral_ctrl);
		break;
	}
}

/* Set low-power voltage levels for LP mode */
void Chip_POWER_SetLowPowerVoltage(uint32_t freq)
{
    switch(freq) {
        case 12000000:
            ROM_POWER->LPCTRL = ((POWER_LP_V0700) | (POWER_LP_V1200 << 2) | (POWER_LP_V1200 << 4) | 
                                 (POWER_LP_V1200 << 6) | (POWER_LP_V1200 << 8) | (POWER_FINE_LP_V_P050 << 10));
        
            ROM_SYSCON->PDRUNCFGSET[0] = PDRUNCFG_LP_VD1 | PDRUNCFG_LP_VD2 | PDRUNCFG_LP_VD3 | PDRUNCFG_LP_VD8 | PDRUNCFG_LP_VDDFLASH;
            ROM_SYSCON->FLASHCFG = (ROM_SYSCON->FLASHCFG & ~(0xF << 12));
            break;
        case 48000000:
            ROM_POWER->LPCTRL = ((POWER_LP_V0700) | (POWER_LP_V1200 << 2) | (POWER_LP_V1200 << 4) | 
                                 (POWER_LP_V1200 << 6) | (POWER_LP_V1200 << 8) | (POWER_FINE_LP_V_P100 << 10));
            ROM_SYSCON->PDRUNCFGSET[0] = PDRUNCFG_LP_VD1 | PDRUNCFG_LP_VD2 | PDRUNCFG_LP_VD3 | PDRUNCFG_LP_VD8 | PDRUNCFG_LP_VDDFLASH;
            ROM_SYSCON->FLASHCFG = (ROM_SYSCON->FLASHCFG & ~(0xF << 12)) | (2 << 12);
            break;
        default:
            /* LP mode is not supported for other frequencies*/
            ROM_SYSCON->PDRUNCFGCLR[0] = PDRUNCFG_LP_VD1 | PDRUNCFG_LP_VD2 | PDRUNCFG_LP_VD3 | PDRUNCFG_LP_VD8 | PDRUNCFG_LP_VDDFLASH;
            break;
    }
}

/* Return ROM version */
uint32_t Chip_POWER_GetROMVersion(void)
{
	return getBROMVersion();
}
