/*
 * The Clear BSD License
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer below) provided
 * that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of the copyright holder nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * NO EXPRESS OR IMPLIED LICENSES TO ANY PARTY'S PATENT RIGHTS ARE GRANTED BY THIS LICENSE.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "board.h"
#include "fsl_debug_console.h"
#include "fsl_dma.h"

#include "pin_mux.h"
#include <stdbool.h>
/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define BUFF_LENGTH 3000
#define BYTE_WIDTH 4U
#define EXAMPLE_DMA_CHANNEL 0U
#define TRANSFER_BLOCK_SZ   4096U /* Maximum Transfer count (1024) X Byte width (4) = 4096 bytes*/
#define MAX_NUM_DESC 10
/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*******************************************************************************
 * Variables
 ******************************************************************************/
dma_handle_t g_DMA_Handle;
volatile bool g_Transfer_Done = false;
dma_transfer_config_t transferConfig;

/*! @brief Static table of descriptors */
#if defined(__ICCARM__)
#pragma data_alignment = 512
dma_descriptor_t g_linked_desc[MAX_NUM_DESC] = {0};
#elif defined(__CC_ARM)
__attribute__((aligned(512))) dma_descriptor_t g_linked_desc[MAX_NUM_DESC] = {0};
#elif defined(__GNUC__)
__attribute__((aligned(512))) dma_descriptor_t g_linked_desc[MAX_NUM_DESC] = {0};
#endif

/*******************************************************************************
 * Code
 ******************************************************************************/

/* User callback function for DMA transfer. */
void DMA_Callback(dma_handle_t *handle, void *param, bool transferDone, uint32_t tcds)
{
    if (transferDone)
    {
        g_Transfer_Done = true;
    }
}

void APP_DMALinkedDescriptors(void *src, void *dest, unsigned long size)
{
    int i,num_desc = 0;
    int len;
    void *srcaddr = src;
    void *destaddr = dest;
    
    /* Calculate number of linked descriptors required */
    num_desc = size / TRANSFER_BLOCK_SZ;
    if(size % TRANSFER_BLOCK_SZ != 0)
    {
            num_desc += 1;
    }
    
    /* Setup the linked descriptors */
    for(i = 0; i < num_desc; i++)
    {
            len = (size < TRANSFER_BLOCK_SZ) ? size : TRANSFER_BLOCK_SZ;
            
            if(i == 0)
            {
                    /* If this is only descriptor in the chain */
                    if(i == num_desc - 1)
                    {
                            DMA_PrepareTransfer(&transferConfig, srcaddr, destaddr, BYTE_WIDTH, len, kDMA_MemoryToMemory,
                    NULL);
                    }
                    else /* Prepare and submit first descriptor */
                    {
                                    DMA_PrepareTransfer(&transferConfig, srcaddr, destaddr, BYTE_WIDTH, len, kDMA_MemoryToMemory,
                    &g_linked_desc[1]);
                    }

                    DMA_SubmitTransfer(&g_DMA_Handle, &transferConfig);
            }
            else if(i == num_desc -1)
            {
                      /* Create last descriptor in chain */
                            DMA_PrepareTransfer(&transferConfig, srcaddr, destaddr, BYTE_WIDTH, len, kDMA_MemoryToMemory,
                    NULL);
                                            DMA_CreateDescriptor(&g_linked_desc[i], &transferConfig.xfercfg, srcaddr, destaddr,
                    NULL);
            }
            else
            {
                    /* Create the descriptors in chain */
                    DMA_PrepareTransfer(&transferConfig, srcaddr, destaddr, BYTE_WIDTH, len, kDMA_MemoryToMemory,
                    &g_linked_desc[i+1]);
                    DMA_CreateDescriptor(&g_linked_desc[i], &transferConfig.xfercfg, srcaddr, destaddr,
                    &g_linked_desc[i+1]);
            }
            
            /* Update source address, destination address and size */
            srcaddr = (void*)((uint32_t)srcaddr + len);
            destaddr = (void*)((uint32_t)destaddr + len);
            transferConfig.srcAddr = (uint8_t*)(srcaddr);
            transferConfig.dstAddr = (uint8_t*)(destaddr);
            size = size - len;
            
    }
	
}
/*!
 * @brief Main function
 */
int main(void)
{
    uint32_t i = 0;
    uint8_t errorFlag = 0;
    uint32_t srcAddr[BUFF_LENGTH] = {0};
    uint32_t destAddr[BUFF_LENGTH] = {0};
    
    /* attach 12 MHz clock to FLEXCOMM0 (debug console) */
    CLOCK_AttachClk(kFRO12M_to_FLEXCOMM0);
    BOARD_InitPins();
    BOARD_BootClockFROHF96M();
    BOARD_InitDebugConsole();
    /* Print source buffer */
    PRINTF("\n\rDMA Linked transfer from memory to memory begin.\r\n");
    PRINTF("Initialize Source and Destination Buffers\r\n");
    for (i = 0; i < BUFF_LENGTH; i++)
    {
            destAddr[i]  = 0;
            srcAddr[i] = i;
    }
                    
    /* Configure DMA */
    /* Initialize DMA peripheral */
    DMA_Init(DMA0);
    /* Enable DMA channel */
    DMA_EnableChannel(DMA0, EXAMPLE_DMA_CHANNEL);
    /* Create DMA handle */
    DMA_CreateHandle(&g_DMA_Handle, DMA0, EXAMPLE_DMA_CHANNEL);
    /* Install a callback function for the DMA transfer */
    DMA_SetCallback(&g_DMA_Handle, DMA_Callback, NULL);

    /* Setup the linked descriptors for the transfer */	
    APP_DMALinkedDescriptors(srcAddr,destAddr,sizeof(srcAddr));

    /* Start DMA transfer */
    DMA_StartTransfer(&g_DMA_Handle);

    /* Wait for DMA transfer finish */
    while (g_Transfer_Done != true)
    {
    }

    /* Verify the destination buffer contents */
    for(i = 0; i< BUFF_LENGTH; i++)
    {
            if(destAddr[i] != srcAddr[i])
            {
                    errorFlag = 1;
                    break;
            }
    }

    /* Check for error */
    if(errorFlag)
    {
            PRINTF("\n\rDMA memory to memory transfer failed !");
    }
    else
    {
            PRINTF("\n\rDMA memory to memory transfer Success !");
    }

    while (1)
    {
            __WFI();
    }
}
