/*
 * Copyright (c) 2001-2003 Swedish Institute of Computer Science.
 * All rights reserved. 
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission. 
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
 * SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
 * OF SUCH DAMAGE.
 *
 * This file is part of the lwIP TCP/IP stack.
 * 
 * Author: Adam Dunkels <adam@sics.se>
 *
 */
#include "tcpSocket.h"

#include "lwip/opt.h"

#if LWIP_NETCONN

#include "lwip/memp.h"
#include "lwip/sys.h"
#include "lwip/api.h"
#include "app_gui.h"
#include "fsl_gpio.h"
#include "fsl_adc.h"
/*-----------------------------------------------------------------------------------*/

enum tcpSocket_raw_states
{
    S_NONE = 0,
    S_CONNECTED,
    S_CLOSING
};

struct tcpSocket_connection *g_ConnectListHead = NULL, *g_ConnectListTail = NULL;
static uint32_t active_conns = 0;

/*freeRTOS Handles*/
TimerHandle_t poll_timer;
SemaphoreHandle_t poll_sem;
QueueSetHandle_t queue_set_handle;
#if(NODE==CLIENT1)||(NODE==CLIENT2)
/* ReConnect flag */
static uint8_t reConnect = 0;
#endif

static void convertHexToAscii(char *dest, uint16_t data)
{
    uint8_t i;
    for(i = 0; i < 4; i++) {
        if((data & 0x0f) < 0x0a) {
            dest[3-i] = '0' + (data & 0x0f);
        }
        else {
            dest[3-i] = 'a' + ((data & 0x0f) - 0x0a);
        }
        data >>= 4;
    }
}

static void convertAsciiToHex(uint16_t *dest, char *data)
{
    uint8_t i;
    *dest = 0;
    for(i = 0; i < 4; i++) {
        if((data[i] >= '0') && (data[i] <= '9')) {
            *dest = (*dest << 4) | (data[i] - '0');
        }
        else {
            *dest = (*dest << 4) | ((data[i] - 'a') + 0xa);
        }
    }
}

/* Sends a message to the GUI thread to clear LCD */
static void clearLCD(uint16_t active_conns)
{
    guiQueueElement_t gui_update;
    
    gui_update.type = E_GUI_CLEAR;
    gui_update.index = 0;
    gui_update.numConns = active_conns;
    while(xQueueSendToBack(guiRdQueue, &gui_update, portMAX_DELAY) != pdTRUE);
}

/* Sends a message to the GUI thread to update the GUI with the provided information */
static void updateLCD(uint16_t active_conns)
{
    guiQueueElement_t gui_update;
    struct tcpSocket_connection *temp = g_ConnectListHead;
    uint8_t index = 0;
    /* If no active connections then send a NULL message */
    if(active_conns == 0) {
        gui_update.type = E_GUI_UPDATE;
        gui_update.index = 0;
        gui_update.numConns = 0;
        gui_update.adcData = 0;
        gui_update.ledStatus = 0;
        gui_update.ledSwitch = 0;
        while(xQueueSendToBack(guiRdQueue, &gui_update, portMAX_DELAY) != pdTRUE);
    }
    else {
        /* Send an update message for each node in the connection list */
        while(temp) {
            gui_update.type = E_GUI_UPDATE;
            gui_update.index = index++;
            gui_update.numConns = active_conns;
            gui_update.adcData = temp->adcData;
            gui_update.ledStatus = temp->ledStatus;
            gui_update.ledSwitch = temp->ledSwitch;
            while(xQueueSendToBack(guiRdQueue, &gui_update, portMAX_DELAY) != pdTRUE);
            temp = temp->next;
        }
    }
}

/* Callback function when the poll timer expires, this is called every 500ms */
static void tcpSocket_poll_timer( TimerHandle_t timer )
{
    xSemaphoreGive(poll_sem);
}

/* Function to process the received data for a connected node */
static void process_rx_data(struct tcpSocket_connection *curr, uint8_t index, void *rx_data, uint16_t rx_len)
{
    guiQueueElement_t gui_update;
    
    /* For each pBuf parse all the string data that was received */
    while(rx_len > 0) {
        if((strncmp(rx_data, "ADC Data ", strlen("ADC Data ")) == 0)) {
            convertAsciiToHex(&(curr->adcData), &(((char *)rx_data)[9])); 
            /* Update ADC value after receiving ADC data from node */
            gui_update.type = E_GUI_ADC;
            gui_update.index = index;
            gui_update.adcData = curr->adcData;
            while(xQueueSendToBack(guiRdQueue, &gui_update, portMAX_DELAY) != pdTRUE);
        }
        /* Check if LED off message was received */
        else if(strcmp(rx_data, "LED Off") == 0) {
            switch(index) {
                case 0:
                    GPIO_SetPinsOutput(GPIO, 2, 1u << 2);
                    break;
                case 1:
                    GPIO_SetPinsOutput(GPIO, 3, 1u << 3);
                    break;
                default:
                    break;
            }
            curr->ledStatus = 0;
            /* Update LED Data */
            gui_update.type = E_GUI_LED;
            gui_update.index = index;
            gui_update.ledStatus = curr->ledStatus;
            while(xQueueSendToBack(guiRdQueue, &gui_update, portMAX_DELAY) != pdTRUE);
        }
        /* Check if LED On message was received */
        else if(strcmp(rx_data, "LED On") == 0) {
            switch(index) {
                case 0:
                    GPIO_ClearPinsOutput(GPIO, 2, 1u << 2);
                    break;
                case 1:
                    GPIO_ClearPinsOutput(GPIO, 3, 1u << 3);
                    break;
                default:
                    break;
            }
            curr->ledStatus = 1;
            /* Update LED Data */
            gui_update.type = E_GUI_LED;
            gui_update.index = index;
            gui_update.ledStatus = curr->ledStatus;
            while(xQueueSendToBack(guiRdQueue, &gui_update, portMAX_DELAY) != pdTRUE);
        }
        /* Check if heart beat message was received */
        else if(strcmp(rx_data, "Alive") == 0) {
            curr->hbFlag = 1;
            curr->hbMissCount = 0;
        }
        rx_len -= (strlen(rx_data) + 1);
        rx_data = (void *)((uint32_t)rx_data + strlen(rx_data) + 1);                        
    }
}

/* Function to add a newly node to the connection list */
static void tcpSocket_add_node(struct netconn *new_conn)
{
    struct tcpSocket_connection *curr;
    
    if(g_ConnectListTail) {
        /* If there is atleast one node in the list then add the node to the end */
        curr = g_ConnectListTail->next = (struct tcpSocket_connection *)mem_malloc(sizeof(struct tcpSocket_connection));
        if(g_ConnectListTail->next != NULL) {
            g_ConnectListTail = g_ConnectListTail->next;
        }
        
    }
    else {
        /* If list is empty then add the first node */
        curr = g_ConnectListHead = g_ConnectListTail = (struct tcpSocket_connection *)mem_malloc(sizeof(struct tcpSocket_connection));
    }
    if (curr != NULL) {
        memset(curr, 0, sizeof(struct tcpSocket_connection));
        /* Add Receive mailbox to queue set */
        xQueueAddToSet(new_conn->recvmbox, queue_set_handle);
        netconn_set_recvtimeout(new_conn, 1);
        /*Clear LCD */
        clearLCD(active_conns);
        active_conns++;
        curr->state = S_CONNECTED;
        curr->conn = new_conn;
        /* Refresh LCD */
        updateLCD(active_conns);
        if(active_conns == 1) {
            while(xTimerStart(poll_timer, portMAX_DELAY) != pdPASS);
        }
    }
}

/* Function to free a node from the connection list */
static void tcpSocket_free(struct tcpSocket_connection *node)
{
    struct tcpSocket_connection *curr = g_ConnectListHead, *prev = NULL;
    /* Find the matching node from the list */
    while(curr && curr != node) {
        prev = curr;
        curr = curr->next;
    }
    /* Check if valid node was found */
    if((curr!= NULL) || (prev!= NULL)) {
        if(prev== NULL) {
            /* Head node is being freed */
            g_ConnectListHead = curr->next;
            if(g_ConnectListHead == NULL) {
                g_ConnectListTail = NULL;
            }
        }
        else if(curr != NULL) {
            /*Any other node other than head */
            prev->next = curr->next;
            /* If tail is being freed then update global tail pointer */
            if(curr->next == NULL) {
                g_ConnectListTail = prev;
            }
        }
    }
    /* Free node if a matching node was found */
    if(curr) {
        mem_free(curr);
    }
}

/* Function to close a TCP connection */
static void tcpSocket_close(struct netconn *conn, struct tcpSocket_connection *curr)
{   
    /* Remove Receive mailbox from queue set */
    xQueueRemoveFromSet(conn->recvmbox, queue_set_handle);
    
    netconn_close(conn);
    netconn_delete(conn);
    
    tcpSocket_free(curr);
    
    /*Clear LCD */
    clearLCD(active_conns);
    if(active_conns) {
        active_conns--;
        if(active_conns == 0) {
            /* Stop timer */
            while(xTimerStop(poll_timer, portMAX_DELAY) != pdPASS);
        }
    }
    /* Refresh LCD */
    updateLCD(active_conns);
#if(NODE==CLIENT1)||(NODE==CLIENT2)
    if(active_conns == 0) {
        /* If client node got disconnected, then retry connection */
        reConnect = 1;
    }
#endif
}

#if(NODE==CLIENT1)||(NODE==CLIENT2)
/* Function to connect a TCP client node */
static void tcpSocket_client_connect(struct netconn *conn)
{
    ip4_addr_t server_ipaddr;
    IP4_ADDR(&server_ipaddr, serverIP_ADDR0, serverIP_ADDR1, serverIP_ADDR2, serverIP_ADDR3);
    /* Try to connect until it is successful */
    while(netconn_connect(conn, &server_ipaddr, 54321) != ERR_OK) {
        netconn_delete(conn);
        conn = netconn_new(NETCONN_TCP);
    }
    /* Add connected server node to the connection list */
    tcpSocket_add_node(conn);
}
#endif

/* TCP socket thread */
static void tcpSocket_thread(void *arg)
{
    struct netconn *conn;
#if(NODE==SERVER)    
    struct netconn *newconn;
#endif    
    QueueSetMemberHandle_t event_handle;
    struct tcpSocket_connection *curr, *temp;
    uint8_t index;
    guiQueueElement_t gui_update;
    char strOn[] = "LED On";
    char strOff[] = "LED Off";
    char hb_msg[6] = "Alive";
    static char adc_msg[] = "ADC Data 0000";
    adc_result_info_t adcResultInfoStruct;
    struct netbuf *net_buffer;
    void *rx_data;
    uint16_t rx_len;
    
    LWIP_UNUSED_ARG(arg);
    /* Create Semaphores, Message queue set and timers */
    poll_sem = xSemaphoreCreateBinary();
    queue_set_handle = xQueueCreateSet(60);
    poll_timer = xTimerCreate("poll_timer", 500/portTICK_PERIOD_MS, pdTRUE, NULL, tcpSocket_poll_timer);

    /* Create a new connection identifier. */
    /* Bind connection to well known port number 7. */
    conn = netconn_new(NETCONN_TCP);
    if(conn!= NULL) {
#if(NODE==SERVER)          
        netconn_bind(conn, IP_ADDR_ANY, 54321);
        /* Tell connection to go into listening mode. */
        netconn_listen(conn);
        /* Add Accept mailbox to queue set */
        xQueueAddToSet(conn->acceptmbox, queue_set_handle);
        /* Set receive timeout to minimum */
        netconn_set_recvtimeout(conn, 1);
#elif(NODE==CLIENT1)||(NODE==CLIENT2)
        /* Blocking connect */
        tcpSocket_client_connect(conn);
#endif      
    }
    else {
        LWIP_ERROR("tcpecho: invalid conn", (conn != NULL), return;);
    }
    
    /* Add Semaphore for poll timer and GUI Write Mailbox to queue set */
    xQueueAddToSet(poll_sem, queue_set_handle);
    xQueueAddToSet(guiWrQueue, queue_set_handle);

    while (1) {

        /* Block the task until the queue set has valid event */
        while((event_handle = xQueueSelectFromSet(queue_set_handle, portMAX_DELAY)) == NULL);
        if(event_handle == guiWrQueue) {
            /* New checkbox event */
            if(xQueueReceive(guiWrQueue, &gui_update, 0) == pdPASS) {
                /* No need to check type of queue element as there is only one now */
                curr = g_ConnectListHead;
                index = 0;
                /* Find the node based on index */
                while(curr && (index < gui_update.index)) {
                    curr = curr->next;
                    index++;
                }
                if(curr) {
                    /* Send LED message */
                    curr->ledSwitch = gui_update.ledSwitch;
                    if(curr->ledSwitch) {
                        netconn_write(curr->conn, strOn, 7, NETCONN_COPY);
                    }
                    else {
                        netconn_write(curr->conn, strOff, 8, NETCONN_COPY);
                    }
                }
            }
        }
#if(NODE==SERVER)        
        else if(event_handle == conn->acceptmbox) {
            /* Grab new connection. */
            if(netconn_accept(conn, &newconn) == ERR_OK) {
                /* Add newly connected client node into the connection list */
                tcpSocket_add_node(newconn);
            }
        }
#endif        
        else if(event_handle == poll_sem) {
            /* Poll Timer Expired */
            if((xSemaphoreTake(poll_sem, 0) == pdPASS) && g_ConnectListHead) {
                /*Do polling stuff */
                /* Read ADC value and start conversion only for the first node */
                if(g_ConnectListHead->adcCount == 0) {
                    ADC_GetChannelConversionResult(ADC0, 0, &adcResultInfoStruct);
                    convertHexToAscii(&adc_msg[9], adcResultInfoStruct.result);
                    ADC_DoSoftwareTriggerConvSeqA(ADC0);
                }
                /* Loop through all nodes in the connection list*/
                curr = g_ConnectListHead;
                while(curr) {
                    /* If the connected node sent heartbeat message then clear flags and counts */
                    if(curr->hbFlag) {
                        curr->hbFlag = 0;
                        curr->hbMissCount = 0;
                    }
                    else {
                        curr->hbMissCount++;
                        if(curr->hbMissCount >= 6) {
                            /* Close connection after missing heart beat for 3s */
                            curr->state = S_CLOSING;
                            temp = curr->next;
                            tcpSocket_close(curr->conn, curr);
                            curr = temp;
                            continue;
                        }
                    }
                    /* Send heart beat message at a different slot than ADC data */
                    if(curr->hbCount == 1) {
                        netconn_write(curr->conn, hb_msg, 6, NETCONN_COPY);
                    }
                    /* Send ADC data */
                    if(curr->adcCount == 0) {
                        netconn_write(curr->conn, adc_msg, 14, NETCONN_COPY);
                    }
                    /* Send ADC data every 2s */
                    if(++curr->adcCount >= 4) {
                        curr->adcCount = 0;
                    }
                    /* Send heart beat message every 1s */
                    if(++curr->hbCount >= 2) {
                        curr->hbCount = 0;
                    }
                    curr = curr->next;
                }
            }
        }
        else {
            /* Check through receive mailbox of all the active connections and find the matching node */
            curr = g_ConnectListHead;
            index = 0;
            while(curr && (event_handle != curr->conn->recvmbox)) {
                index++;
                curr = curr->next;
            }
            if(curr) {
                if(netconn_recv(curr->conn, &net_buffer) == ERR_OK) {
                    /* process all the received pBuf's for this node */
                    do {
                        netbuf_data(net_buffer, &rx_data, &rx_len);
                        /* Process the received data for this pBuf*/
                        process_rx_data(curr, index, rx_data, rx_len);
                    } while (netbuf_next(net_buffer) >= 0);
                    netbuf_delete(net_buffer);
                }
                else {
                    tcpSocket_close(curr->conn, curr);
                }
            }
        }
#if(NODE==CLIENT1)||(NODE==CLIENT2)        
        if(reConnect) {
            /* If the client got disconnected then retry connection */
            reConnect = 0;
            conn = netconn_new(NETCONN_TCP);
            if(conn!= NULL) {
                /* Blocking connect */
                tcpSocket_client_connect(conn);
            }
        }
#endif        
    }
}
/*-----------------------------------------------------------------------------------*/
void tcpSocket_init(void)
{
    /* Create TCP socket thread */
    sys_thread_new("tcpSocket_thread", tcpSocket_thread, NULL, DEFAULT_THREAD_STACKSIZE, DEFAULT_THREAD_PRIO);
}
/*-----------------------------------------------------------------------------------*/

#endif /* LWIP_NETCONN */
