/*
* Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
* All rights reserved.
*
* Redistribution and use in source and binary forms, with or without modification,
* are permitted provided that the following conditions are met:
*
* o Redistributions of source code must retain the above copyright notice, this list
*   of conditions and the following disclaimer.
*
* o Redistributions in binary form must reproduce the above copyright notice, this
*   list of conditions and the following disclaimer in the documentation and/or
*   other materials provided with the distribution.
*
* o Neither the name of the copyright holder nor the names of its
*   contributors may be used to endorse or promote products derived from this
*   software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
* DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
* ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
* LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
* ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
* SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*******************************************************************************
 * Includes
 ******************************************************************************/

#include "lwip/tcpip.h"
#include "netif/ethernet.h"
#include "ethernetif.h"

#include "board.h"
#include "app.h"
#include "fsl_inputmux.h"
#include "fsl_pint.h"
#include "fsl_lcdc.h"
#include "fsl_i2c.h"
#include "fsl_i2c_freertos.h"
#include "fsl_ft5406.h"
#include "fsl_adc.h"
#include "app_gui.h"


/*******************************************************************************
 * Definitions
 ******************************************************************************/
 
#define TOUCH_POINT_GET_EVENT(T) ((touch_event_t)((T).XH >> 6))
#define TOUCH_POINT_GET_ID(T) ((T).YH >> 4)
#define TOUCH_POINT_GET_X(T) ((((T).XH & 0x0f) << 8) | (T).XL)
#define TOUCH_POINT_GET_Y(T) ((((T).YH & 0x0f) << 8) | (T).YL)

/*******************************************************************************
* Prototypes
******************************************************************************/

typedef struct
{
    uint8_t XH;
    uint8_t XL;
    uint8_t YH;
    uint8_t YL;
    uint8_t RESERVED[2];
} coordinates_t;

typedef struct _touch_data
{
    uint8_t GEST_ID;
    uint8_t TD_STATUS;
    coordinates_t TOUCH[FT5406_MAX_TOUCHES];
} touch_data_t;

/*******************************************************************************
* Variables
******************************************************************************/

/* I2C handle for touch controller*/
i2c_rtos_handle_t i2cHandle;

/* Frame end flag. */
static int32_t s_LCDpendingBuffer;

/*******************************************************************************
 * Code
 ******************************************************************************/

void APP_LCD_IRQHandler(void)
{
    uint32_t addr;
    uint32_t intStatus = LCDC_GetEnabledInterruptsPendingStatus(APP_LCD);

    LCDC_ClearInterruptsStatus(APP_LCD, intStatus);

    if (intStatus & kLCDC_VerticalCompareInterrupt)
    {
        if (s_LCDpendingBuffer >= 0)
        {
            /* Calculate address of the given buffer */
            addr = VRAM_ADDR + VRAM_SIZE * s_LCDpendingBuffer;
            /* Make the given buffer visible */
            LCDC_SetPanelAddr(APP_LCD, kLCDC_UpperPanel, addr);
            /* Send a confirmation that the given buffer is visible */
            GUI_MULTIBUF_Confirm(s_LCDpendingBuffer);
            s_LCDpendingBuffer = -1;
        }
    }
    __DSB();
    /* Add for ARM errata 838869, affects Cortex-M4, Cortex-M4F Store immediate overlapping
      exception return operation might vector to incorrect interrupt */
#if defined __CORTEX_M && (__CORTEX_M == 4U)
    __DSB();
#endif
}

static status_t APP_I2C_Init(void)
{
    i2c_master_config_t masterConfig;

    I2C_MasterGetDefaultConfig(&masterConfig);

    /* Change the default baudrate configuration */
    masterConfig.baudRate_Bps = I2C_BAUDRATE;

    /* Initialize the I2C master peripheral */
    I2C_RTOS_Init(&i2cHandle, EXAMPLE_I2C_MASTER, &masterConfig, I2C_MASTER_CLOCK_FREQUENCY);
    NVIC_SetPriority(FLEXCOMM2_IRQn, 4);

    return kStatus_Success;
}

static status_t APP_LCDC_Init(void)
{
    /* Initialize the display. */
    lcdc_config_t lcdConfig;

    LCDC_GetDefaultConfig(&lcdConfig);

    lcdConfig.panelClock_Hz = LCD_PANEL_CLK;
    lcdConfig.ppl = LCD_PPL;
    lcdConfig.hsw = LCD_HSW;
    lcdConfig.hfp = LCD_HFP;
    lcdConfig.hbp = LCD_HBP;
    lcdConfig.lpp = LCD_LPP;
    lcdConfig.vsw = LCD_VSW;
    lcdConfig.vfp = LCD_VFP;
    lcdConfig.vbp = LCD_VBP;
    lcdConfig.polarityFlags = LCD_POL_FLAGS;
    lcdConfig.upperPanelAddr = VRAM_ADDR;
    lcdConfig.bpp = kLCDC_16BPP565;
    lcdConfig.display = kLCDC_DisplayTFT;
    lcdConfig.swapRedBlue = false;

    LCDC_Init(APP_LCD, &lcdConfig, LCD_INPUT_CLK_FREQ);

    /* Trigger interrupt at start of every vertical back porch. */
    LCDC_SetVerticalInterruptMode(APP_LCD, kLCDC_StartOfBackPorch);
    LCDC_EnableInterrupts(APP_LCD, kLCDC_VerticalCompareInterrupt);
    NVIC_SetPriority(APP_LCD_IRQn, 3);
    NVIC_EnableIRQ(APP_LCD_IRQn);

    LCDC_Start(APP_LCD);
    LCDC_PowerUp(APP_LCD);

    return kStatus_Success;
}

/*******************************************************************************
 * Application implemented functions required by emWin library
 ******************************************************************************/
void LCD_X_Config(void)
{
    GUI_MULTIBUF_Config(GUI_BUFFERS);
    GUI_DEVICE_CreateAndLink(GUIDRV_LIN_16, GUICC_565, 0, 0);

    LCD_SetSizeEx(0, LCD_WIDTH, LCD_HEIGHT);
    LCD_SetVSizeEx(0, LCD_WIDTH, LCD_HEIGHT);

    LCD_SetVRAMAddrEx(0, (void *)VRAM_ADDR);
}

int LCD_X_DisplayDriver(unsigned LayerIndex, unsigned Cmd, void *p)
{
    int r;
    LCD_X_SHOWBUFFER_INFO *pData;
    switch (Cmd)
    {
        case LCD_X_SHOWBUFFER:
        {
            pData = (LCD_X_SHOWBUFFER_INFO *)p;
            //
            // Remember buffer index to be used by ISR
            //
            s_LCDpendingBuffer = pData->Index;
            return 0;
        }
        default:
            r = -1;
    }
    return r;
}

void GUI_X_Config(void)
{
    /* Assign work memory area to emWin */
    GUI_ALLOC_AssignMemory((void *)GUI_MEMORY_ADDR, GUI_NUMBYTES);

    /* Select default font */
    GUI_SetDefaultFont(GUI_FONT_6X8);
}

void GUI_X_Init(void)
{
}

/* Dummy RTOS stub required by emWin */
void GUI_X_InitOS(void)
{
}

/* Dummy RTOS stub required by emWin */
void GUI_X_Lock(void)
{
}

/* Dummy RTOS stub required by emWin */
void GUI_X_Unlock(void)
{
}

/* Dummy RTOS stub required by emWin */
U32 GUI_X_GetTaskId(void)
{
    return 0;
}

void GUI_X_ExecIdle(void)
{
}

GUI_TIMER_TIME GUI_X_GetTime(void)
{
    return 0;
}

void GUI_X_Delay(int Period)
{
}

void GUI_X_ErrorOut(char const *s)
{
}

void *emWin_memcpy(void *pDst, const void *pSrc, long size)
{
    return memcpy(pDst, pSrc, size);
}

static void cbBackgroundWin(WM_MESSAGE *pMsg)
{
    switch (pMsg->MsgId)
    {
        case WM_NOTIFY_PARENT:
            /* Call application GUI callback */
            app_gui_msg_cb(pMsg);
            break;
        default:
            WM_DefaultProc(pMsg);
    }
}

/* Low Priority GUI thread */
static void emwin_gui_thread(void *arg)
{
    status_t status;
    ft5406_handle_t touch_handle;
    touch_event_t touch_event;
    uint8_t mode;
    GUI_PID_STATE pid_state;
    touch_data_t *touch_data;
    
    /* Initialize touch panel controller */
    touch_handle.base = EXAMPLE_I2C_MASTER;

    /* clear transfer structure and buffer */
    memset(&touch_handle.xfer, 0, sizeof(touch_handle.xfer));
    memset(touch_handle.touch_buf, 0, FT5406_TOUCH_DATA_LEN);

    /* set device mode to normal operation */
    mode = 0;
    touch_handle.xfer.slaveAddress = 0x38;
    touch_handle.xfer.direction = kI2C_Write;
    touch_handle.xfer.subaddress = 0;
    touch_handle.xfer.subaddressSize = 1;
    touch_handle.xfer.data = &mode;
    touch_handle.xfer.dataSize = 1;
    touch_handle.xfer.flags = kI2C_TransferDefaultFlag;

    status = I2C_RTOS_Transfer(&i2cHandle, &touch_handle.xfer);

    /* prepare transfer structure for reading touch data */
    touch_handle.xfer.slaveAddress = 0x38;
    touch_handle.xfer.direction = kI2C_Read;
    touch_handle.xfer.subaddress = 1;
    touch_handle.xfer.subaddressSize = 1;
    touch_handle.xfer.data = touch_handle.touch_buf;
    touch_handle.xfer.dataSize = FT5406_TOUCH_DATA_LEN;
    touch_handle.xfer.flags = kI2C_TransferDefaultFlag;
    
    if (status != kStatus_Success)
    {
        PRINTF("Touch panel init failed\n");
    }
    assert(status == kStatus_Success);
    
    while (1)
    {
        /* Capture any touch input */
        if (I2C_RTOS_Transfer(&i2cHandle, &touch_handle.xfer) == kStatus_Success)
        {
            touch_data = (touch_data_t *)(touch_handle.touch_buf);

            touch_event = TOUCH_POINT_GET_EVENT(touch_data->TOUCH[0]);

            /* Update coordinates only if there is touch detected */
            if (touch_event != kTouch_Reserved)
            {
                if((touch_event == kTouch_Down) || (touch_event == kTouch_Contact)) {
                    pid_state.x = TOUCH_POINT_GET_Y(touch_data->TOUCH[0]);
                    pid_state.y = TOUCH_POINT_GET_X(touch_data->TOUCH[0]);
                }
                pid_state.Pressed = ((touch_event == kTouch_Down) || (touch_event == kTouch_Contact));
                pid_state.Layer = 0;
                /* provide touch state to emWin library */
                GUI_TOUCH_StoreStateEx(&pid_state);
            }
        }
        /* Call App layer GUI thread function */
        app_gui_thread();
        
        /* emWin Execution */
        GUI_MULTIBUF_Begin();
        GUI_Exec();
        GUI_MULTIBUF_End();
        
        /* Block for 10ms */
        vTaskDelay(GUI_THREAD_RATE_MS/portTICK_PERIOD_MS);
    }
}

/*!
 * @brief Initializes emWin GUI thread.
 */
void emwin_gui_init(void)
{
    status_t status;
    
    /* Define the init structure for the output LED pin*/
    
    gpio_pin_config_t pin_config = {
        kGPIO_DigitalOutput, 1,
    };
    
    /* Initialize I2C for touch panel controller */
    status = APP_I2C_Init();
    if (status != kStatus_Success)
    {
        PRINTF("I2C init failed\n");
    }
    assert(status == kStatus_Success);

    /* Enable touch panel controller */
    GPIO_PinInit(GPIO, 2, 27, &pin_config);
    
    /* Initialize LCD controller */
    status = APP_LCDC_Init();
    if (status != kStatus_Success)
    {
        PRINTF("LCD init failed\n");
    }
    assert(status == kStatus_Success);

    /* emWin start */
    GUI_Init();

    /* Set size and default color for the background window */
    WM_SetSize(WM_HBKWIN, LCD_WIDTH, LCD_HEIGHT);
    WM_SetDesktopColor(GUI_WHITE);
    /* Set callback for the backgroung window */
    WM_SetCallback(WM_HBKWIN, cbBackgroundWin);

    /* Solid color display */
    GUI_SetBkColor(GUI_WHITE);
    GUI_Clear();
    
    app_gui_init();

    sys_thread_new("gui_thread", emwin_gui_thread, NULL, DEFAULT_THREAD_STACKSIZE, DEFAULT_THREAD_PRIO - 1);
}
