/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of the copyright holder nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "fsl_device_registers.h"
#include "fsl_debug_console.h"
#include "board.h"
#include "app.h"
#include "fsl_adc.h"
#include "fsl_clock.h"
#include "fsl_power.h"
#include "fsl_dma.h"
#include "fsl_usart.h"
#include "fsl_usart_dma.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/
/* DMA channel Connected to Sample 0 ADC Channel */
#define DEMO_DMA_ADC_SAMPLE0_CHANNEL 1U
/*ADC Channel 0 Data register address*/
#define DEMO_ADC_DATA0_REG_ADDR (uint32_t)(&(DEMO_ADC_BASE->DAT[DEMO_ADC_SAMPLE0_CHANNEL_NUMBER]))

/*******************************************************************************
 * Prototypes
 ******************************************************************************/
static void ADC_Configuration(void);
static void DMA_Configuration(void);
static void USART_Configuration(void);

dma_handle_t g_uartTxDmaHandle;
usart_dma_handle_t g_uartDmaHandle;
volatile bool gAdcConvSeqAIntFlag; /* Flag of Sequence A ADC conversion. */

uint32_t adc_clock;

/*******************************************************************************
 * Code
 ******************************************************************************/
/*!
 * @brief ADC SequenceA IrqHandler
 */
void DEMO_ADC_IRQ_SEQA_HANDLER_FUNC(void)
{
    if (kADC_ConvSeqAInterruptFlag == (kADC_ConvSeqAInterruptFlag & ADC_GetStatusFlags(DEMO_ADC_BASE)))
    {
        ADC_ClearStatusFlags(DEMO_ADC_BASE, kADC_ConvSeqAInterruptFlag);
        gAdcConvSeqAIntFlag = true;
    }
}

/*!
 * @brief Main function
 */
int main(void)
{
    /* Initialize board hardware. */
    BOARD_InitHardware();

    /* Configure USART */
    USART_Configuration();
    /* Configure ADC */
    ADC_Configuration();
    /* Configure DMA */
    DMA_Configuration();

    while (1)
    {
        /* Get the input from terminal and trigger the converter by software. */
        GETCHAR();

        gAdcConvSeqAIntFlag = false;
        ADC_DoSoftwareTriggerConvSeqA(DEMO_ADC_BASE); /* Trigger the ADC and start the conversion. */

        /* Wait for the converter & transfer to be done. */
        while (!gAdcConvSeqAIntFlag)
        {
        }

        /*Start the DMA transfer*/
        DMA_StartTransfer(&g_uartTxDmaHandle);
    }
}

/*!
 * @brief USART Configuration function
 */
static void USART_Configuration(void)
{
    usart_config_t config;

    /* Initialize the UART. */
    /* Default configuration
   *
     * config.baudRate_Bps = 115200U;
     * config.parityMode = kUART_ParityDisabled;
     * config.stopBitCount = kUART_OneStopBit;
     * config.txFifoWatermark = 0;
     * config.rxFifoWatermark = 1;
     * config.enableTx = false;
     * config.enableRx = false;
     */
    USART_GetDefaultConfig(&config);
    config.baudRate_Bps = BOARD_DEBUG_UART_BAUDRATE;
    config.enableTx = true;

    USART_Init(DEMO_USART, &config, DEMO_USART_CLK_FREQ);
}

/*!
 * @brief ADC Configuration function
 */
static void ADC_Configuration(void)
{
    adc_config_t adcConfigStruct;
    adc_conv_seq_config_t adcConvSeqAConfigStruct;

    /* Configure the converter. */
    adcConfigStruct.clockMode = kADC_ClockSynchronousMode; /* Using sync clock source. */
    adcConfigStruct.clockDividerNumber = 1;                /* The divider for sync clock is 2. */
    adcConfigStruct.resolution = kADC_Resolution12bit;     /* 12-bit resolution */
    adcConfigStruct.enableBypassCalibration = false;       /* Calibration not Bypassed*/
    adcConfigStruct.sampleTimeNumber = 0U;                 /* Default Sample Time */
                                                           /* Initialize ADC */
    ADC_Init(DEMO_ADC_BASE, &adcConfigStruct);

    /* Calibration after power up. */
    if (!ADC_DoSelfCalibration(DEMO_ADC_BASE))
    {
        while (1)
            ;
    }

#if !(defined(FSL_FEATURE_ADC_HAS_NO_INSEL) && FSL_FEATURE_ADC_HAS_NO_INSEL)
    /* Use the temperature sensor input to channel 0. */
    ADC_EnableTemperatureSensor(DEMO_ADC_BASE, true);
#endif /* FSL_FEATURE_ADC_HAS_NO_INSEL. */

    /* Enable channel DEMO_ADC_SAMPLE0_CHANNEL_NUMBER's conversion in Sequence A. */
    adcConvSeqAConfigStruct.channelMask =
        (1U << DEMO_ADC_SAMPLE0_CHANNEL_NUMBER); /* Includes channel DEMO_ADC_SAMPLE0_CHANNEL_NUMBER. */
    adcConvSeqAConfigStruct.triggerMask = 0U;
    adcConvSeqAConfigStruct.triggerPolarity = kADC_TriggerPolarityPositiveEdge;
    adcConvSeqAConfigStruct.enableSingleStep = false;
    adcConvSeqAConfigStruct.enableSyncBypass = false;
    adcConvSeqAConfigStruct.interruptMode = kADC_InterruptForEachSequence; /* Enable the interrupt/DMA trigger. */

    ADC_SetConvSeqAConfig(DEMO_ADC_BASE, &adcConvSeqAConfigStruct);
    /* Enable conversion sequence A */
    ADC_EnableConvSeqA(DEMO_ADC_BASE, true);

    /* Enable interrupts for conversion sequence A*/
    ADC_EnableInterrupts(DEMO_ADC_BASE, kADC_ConvSeqAInterruptEnable);
    NVIC_EnableIRQ(DEMO_ADC_IRQ_SEQA_ID);
}

/*!
 * @brief DMA Configuration function
 */
static void DMA_Configuration(void)
{
    dma_transfer_config_t dmaTransferConfigStruct;

    /* Initialize DMA */
    DMA_Init(DMA0);

    /* Enable DMA Channel for USART TX*/
    DMA_EnableChannel(DMA0, USART_TX_DMA_CHANNEL);

    /*Create the DMA handle*/
    DMA_CreateHandle(&g_uartTxDmaHandle, DMA0, USART_TX_DMA_CHANNEL);

    /* Create USART DMA handle. */
    USART_TransferCreateHandleDMA(DEMO_USART, &g_uartDmaHandle, NULL, NULL, &g_uartTxDmaHandle, NULL);

    /* Enable DMA request from txFIFO */
    USART_EnableTxDMA(DEMO_USART, true);

    /* Prepare transfer. */
    DMA_PrepareTransfer(&dmaTransferConfigStruct, (uint8_t *)DEMO_ADC_DATA0_REG_ADDR, (void *)&DEMO_USART->FIFOWR,
                        sizeof(uint8_t), sizeof(uint8_t) * 4, kDMA_StaticToStatic, NULL);

    dmaTransferConfigStruct.xfercfg.srcInc = 1;

    /* Submit transfer. */
    DMA_SubmitTransfer(&g_uartTxDmaHandle, &dmaTransferConfigStruct);
}
