/*
 * @brief USART Timeout Workaround example 
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2014
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */

/**
 * Routine to demonstrate using Timer 0 as a FIFO receive timeout device.  If 
 * the number of characters (FIFO_TRIG_LVL characters) are not received in the
 * expected time, Timer 0 will cause a timeout.  Uses loopback mode to send 
 * characters to itself.  It is neither efficient nor elegant nor does it cover 
 * important corner cases but it can provide a template for the user to implement 
 * the functionality as needed for their application.
 */
#include "chip.h"
#include "board.h"
#include <stdio.h>

#define APB_BUS_CLK_HZ        48000000
#define BAUD_RATE             9600
#define NUM_DATA_BITS         8
#define NUM_START_BITS				1
#define NUM_STOP_BITS         1
#define FIFO_TRIG_LVL         8

#define LPC_MINUART0          ((LPC_USART_T *)        LPC_FLEXCOMM0_BASE)  
#define MINUART0_IRQ					FLEXCOMM0_IRQn
#define CTIMER0_IRQ						CT32B0_IRQn

#define USART0_FLEXCOMM 			0
#define UART_RB_SIZE 					512
#define LPC_USART       			LPC_USART0
#define LPC_IRQNUM      			USART0_IRQn
#define LPC_UARTHNDLR   			USART0_IRQHandler



volatile uint32_t uart_err_cnt = 0, fifo_err_cnt = 0;
volatile uint32_t timeout_in_progress = false;

/**
 * Start the timeout timer
 * 
 * Enables the timeout timer.
 */
void start_timeout_timer(void) 
{
  timeout_in_progress = true;
  LPC_TIMER0->TC = 0;
  LPC_TIMER0->PC = 0;
  Chip_TIMER_Enable(LPC_TIMER0);
}

/**
 * Stop the timeout timer
 * 
 * Disables the timeout timer.  
 */
void stop_timeout_timer(void) 
{
  timeout_in_progress = false;
	Chip_TIMER_Disable(LPC_TIMER0);
}

volatile uint32_t uart_isr_cnt = 0;


void set_rx_fifo_trig_level(uint32_t level) 
{
	
  uint32_t fifotrig;
 
	fifotrig = (LPC_USART->FIFOTRIG & ~(UART_FIFOTRIG_RXLVL(0xf))) | UART_FIFOTRIG_RXLVL(level);
  
	LPC_USART->FIFOTRIG = fifotrig;
}

void empty_fifo(void) 
{
  Chip_UART_FlushFIFOs(LPC_USART);
}




/**
 * UART ISR 
 * 
 */
void FLEXCOMM0_IRQHandler(void) {
  uint32_t uartintstat, fifostat, rxlvl, fiford;
  int i;

  uartintstat = LPC_USART->INTSTAT;
  fifostat =  LPC_USART->FIFOSTAT;

  /* If any UART errors, clear them */
  if (uartintstat & (UART_STAT_FRM_ERRINT | UART_STAT_PAR_ERRINT |
          UART_STAT_RXNOISEINT | UART_STAT_ABERR)) {
    Chip_UART_ClearStatus(LPC_USART,UART_STAT_FRM_ERRINT | UART_STAT_PAR_ERRINT |
          UART_STAT_RXNOISEINT | UART_STAT_ABERR);
    uart_err_cnt++;
  }

  /* If any FIFO errors clear them */
  if (fifostat & (UART_FIFOINT_TXERR | UART_FIFOINT_RXERR)) {
    Chip_UART_ClearFIFOStatus(LPC_USART,UART_FIFOSTAT_TXERR| UART_FIFOSTAT_RXERR);
    fifo_err_cnt++;
  }

  if (fifostat & UART_FIFOSTAT_RXNOTEMPTY) {
    rxlvl = (LPC_USART->FIFOSTAT >> 16) & 0x1f;
    if (rxlvl < FIFO_TRIG_LVL) {
      if (!timeout_in_progress) {
        set_rx_fifo_trig_level(FIFO_TRIG_LVL - 1);
        start_timeout_timer();
      }
    } else {
      stop_timeout_timer();
      for (i = 0; i < rxlvl; i++) {
        fiford = LPC_USART->FIFORD; // For this example we just compare and throw away
        if (fiford != i) {
          while(1);
        }
      }
      set_rx_fifo_trig_level(0);
    }
  }
  uart_isr_cnt++;
}


volatile uint32_t timer_isr_cnt = 0;


/**
 * Timer timeout ISR 
 * 
 * Invoked if the timer times out. Depending on implementation it will do 
 * different things.  For this example, it empties any characters in the UART
 * FIFO.
 */
void CT32B0_IRQHandler(void) {

  /* Clear timer interrupt */
  Chip_TIMER_ClearMatch(LPC_TIMER0, 0);
  stop_timeout_timer();
  /* For this example we flush out whatever characters have been received*/
	empty_fifo();  
  set_rx_fifo_trig_level(0);  
	
  timer_isr_cnt++;
}

/**
 * Initialize the UART
 *
 * Initialize the UART @ BAUD_RATE,n,8,1.  
 * Setup so initially that any number of received chars in FIFO will cause a FIFO level 
 * trigger interrupt, i.e., interrupt when not empty. 
 *
 */
void init_uart() {
 int ret;
	
	ret = Chip_UART_Init(LPC_USART);
	LPC_ASSERT(ret == 0, __FILE__, __LINE__);

	/* Enable UART and set at 8 bits, no parity, 1 stop in loop mode */
	Chip_UART_ConfigData(LPC_USART, UART_CFG_DATALEN_8 | UART_CFG_PARITY_NONE | UART_CFG_STOPLEN_1 | UART_CFG_LOOP);
	Chip_UART_Enable(LPC_USART);


  /* Enable error interrupts */
	Chip_UART_IntEnable(LPC_USART, (UART_INT_FRAMERR|UART_INT_PARITYERR|UART_INT_RXNOISE));
	Chip_UART_SetBaud(LPC_USART, BAUD_RATE);


  /* Setup FIFO RX triggers.  At initialization we setup to trigger on *any* 
   * number of characters in the FIFO 
   */
	Chip_UART_SetFIFOTrigLevel(LPC_USART,0,0);

  /* Setup FIFO interrupts */
	Chip_UART_EnableFIFOInts(LPC_USART, (UART_FIFOINT_TXERR|UART_FIFOINT_RXERR|UART_FIFOINT_RXLVL));
	NVIC_EnableIRQ(LPC_IRQNUM);

}

/**
 * Initialize timer 0 as the UART RX timeout timer 
 *
 * Initialize timer in Timer mode so that when the counter is enabled, TC will increment on 
 * every APB bus clock cycle.  Setup so that interrupt occurs when TC == value set in 
 * Match Register 0.  At exit everything is setup and ready except for enabling the counter.
 */
void init_timer(float timeout_secs) {
  float apb_bus_clk_secs = 1.0 / APB_BUS_CLK_HZ;

  uint32_t match_value = (uint32_t) (timeout_secs / apb_bus_clk_secs + 0.5);
	
  /* Reset and enable clock to Timer 0 */
	Chip_TIMER_Init(LPC_TIMER0);
	
  /* Set Timer to Timer Mode and set PR to 0 to increment TC on every APB bus clock */
  LPC_TIMER0->CTCR = (0x0<<0);
  Chip_TIMER_PrescaleSet(LPC_TIMER0, 0);
	Chip_TIMER_Reset(LPC_TIMER0);

  /* Set so interrupt occurs when TC reaches value in MR0 */
	Chip_TIMER_SetMatch(LPC_TIMER0,0,match_value);
  Chip_TIMER_MatchEnableInt(LPC_TIMER0,0);

	NVIC_EnableIRQ(CT32B0_IRQn);
}


int main() {
	int i;
  float timeout_secs, char_time_secs;
  volatile uint32_t delay;

	SystemCoreClockUpdate();
	Board_Init();

  /* Calculate time of a character */
  char_time_secs = (1.0 / BAUD_RATE) * (NUM_DATA_BITS + NUM_STOP_BITS + NUM_START_BITS);

  /* Calculate timeout. FIFO_TRIG_LVL - 1 is used because the timeout counter is started after the first character is received. */
  timeout_secs = (FIFO_TRIG_LVL - 1) * char_time_secs;

  init_timer(timeout_secs);

  init_uart();

  /* This first test should not cause a timeout because the characters are sent back to back.
   * The UART ISR  is entered twice and the Timer ISR is never entered  */
  uart_isr_cnt = 0;
  timer_isr_cnt = 0;
  delay = 1000000;
  for (i = 0; i < FIFO_TRIG_LVL; i++) {
		LPC_USART->FIFOWR = i ;
  }
  while (uart_isr_cnt != 2);
  while (delay--);
  if (timer_isr_cnt != 0) {
    while (1);
  }

  /* This next test should cause a timeout because the last character that is expected is not sent.  
	 * The UART ISR is entered once and the Timer ISR is entered once */
  uart_isr_cnt = 0;
  timer_isr_cnt = 0;
  for (i = 0; i < FIFO_TRIG_LVL - 1; i++) {
    LPC_USART->FIFOWR = i ;
  }
  while (uart_isr_cnt != 1);
  while (timer_isr_cnt != 1);

  /* This test ensures that system recovers after a timeout.  It should not cause a timeout 
   * because the characters are sent back to back.  The UART ISR is entered twice and the Timer ISR is never entered */
  uart_isr_cnt = 0;
  timer_isr_cnt = 0;
  delay = 1000000;
  for (i = 0; i < FIFO_TRIG_LVL; i++) {
    LPC_USART->FIFOWR = i ;
  }
  while (uart_isr_cnt != 2);
  while (delay--);
  if (timer_isr_cnt != 0) {
    while (1);
  }
    
  while (1);

}
