/****************************************************************************
 *
 * MODULE:             Demo coordinator code
 *
 * COMPONENT:          $RCSfile: SensorDemoCoord.c,v $
 *
 * VERSION:            $Name:  $
 *
 * REVISION:           $Revision: 1.2 $
 *
 * DATED:              $Date: 2008/05/08 14:53:30 $
 *
 * STATUS:             $State: Exp $
 *
 * AUTHOR:             CJG
 *
 * DESCRIPTION:
 * Coordinator for demonstrator. Acts as time coordinator for up to four
 * sensor endpoints and manages LCD panel and keys on central coordinator.
 *
 * LAST MODIFIED BY:   $Author: cjg $
 *                     $Modtime: $
 *
 ****************************************************************************
 *
 *  (c) Copyright 2007 JENNIC Ltd
 *
 ****************************************************************************/

/****************************************************************************/
/***        Include files                                                 ***/
/****************************************************************************/
#include "jendefs.h"
#include "AppHardwareApi.h"
#include "PeripheralRegs.h"
#include "Printf.h"

extern void vLoadBootImage(uint32 u32AppId);

/****************************************************************************/
/***        Macro Definitions                                             ***/
/****************************************************************************/

/****************************************************************************/
/***        Type Definitions                                              ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Function Prototypes                                     ***/
/****************************************************************************/
PRIVATE void vDumpFlash(uint32 u32Sector, uint32 Remap);
PRIVATE char acGetC(void );
PRIVATE void vInvalidateCurrentImage(void );
PRIVATE void vValidateImage(uint32 u32Address);

/****************************************************************************/
/***        Exported Variables                                            ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Variables                                               ***/
/****************************************************************************/
PRIVATE uint8 NextValidSector = 8;
PRIVATE uint32 u32MagicAddress = 0;

/****************************************************************************/
/***        Exported Functions                                            ***/
/****************************************************************************/
/****************************************************************************
 *
 * NAME: AppColdStart
 *
 * DESCRIPTION:
 * Entry point for application from boot loader. Initialises system and runs
 * main loop.
 *
 * RETURNS:
 * Never returns.
 *
 ****************************************************************************/
PUBLIC void AppColdStart(void)
{
	uint16 SectorCount = 0;
	volatile uint32 u32ctr = 0;

	uint32 u32Remap = u32REG_SysRead(REG_SYS_FLASH_REMAP);

	vAHI_WatchdogStop();

	// Wait for clock to switch to crystal so UART is at correct rate
	while (bAHI_GetClkSource() == TRUE);
	vAHI_OptimiseWaitStates();

    vUART_printInit();
	u32AHI_Init();

	vPrintf("Flash Remap %8x.\n\r",u32Remap);

#if defined JN5169
	for(SectorCount=0;SectorCount<16;SectorCount++){
#elif defined JN5168
	for(SectorCount=0;SectorCount<8;SectorCount++){
#elif defined JN5164
	for(SectorCount=0;SectorCount<5;SectorCount++){
#elif defined JN5161
	for(SectorCount=0;SectorCount<2;SectorCount++){
#else
#pragma message  "* Device is not defined *"
		{  // this is done to avoid confusing the syntax highlighter in Beyond Studio
#endif
		vDumpFlash(SectorCount,(u32Remap>>(SectorCount*4))  & 0xf);
	}

	if((u32Remap & 0xf) == 0){
		vPrintf("Complete.\n\r");
	}
	else {
		vPrintf("NextValidSector:%x\n",NextValidSector);

		bAHI_FlashInit(E_FL_CHIP_INTERNAL, NULL);

		vPrintf("Press a key to invalidate this image.\n\r");
		acGetC();

		vValidateImage(u32MagicAddress);

		vInvalidateCurrentImage();

		// Add a delay
		for (u32ctr=0;u32ctr<500000; u32ctr++);

		vAHI_SwReset();
	}

	// should never get here
	while(1);
}


/****************************************************************************
 *
 * NAME: AppWarmStart
 *
 * DESCRIPTION:
 * Entry point for application from boot loader. Simply jumps to AppColdStart
 * as, in this instance, application will never warm start.
 *
 * RETURNS:
 * Never returns.
 *
 ****************************************************************************/
PUBLIC void AppWarmStart(void)
{
    vUART_printInit();
    vPrintf("Warm Started. We should not get here.\n\r");
    while (1);
}

/****************************************************************************/
/****************************************************************************/
/***        Local Functions                                               ***/
/****************************************************************************/
/****************************************************************************/
#define FLASH_BASE_ADDR             (0x80000)

uint8 au8MagicBytes[16] = {
        0x12, 0x34, 0x56, 0x78,
        0x11, 0x22, 0x33, 0x44,
        0x55, 0x66, 0x77, 0x88,
        0x08, 0x01, 0x00, 0x00};

uint8 au8ZeroBytes[16] = {
        0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00};

PRIVATE void vDumpFlash(uint32 u32Sector, uint32 u32Remap)
{
    bool_t bPassed=TRUE;
    bool_t bAllOnes=TRUE;
    bool_t bAllZeros=TRUE;
    int n;

    uint32 u32Address = u32Remap * (32 * 1024UL);

    //Check Magic Bytes
    uint8* pu8FlashPtr = (uint8 *)(u32Address + FLASH_BASE_ADDR);

    for(n = 0; n < 12; n++){
    	if(au8MagicBytes[n] != pu8FlashPtr[n]){
    		 bPassed=FALSE;
    	}
    	if(pu8FlashPtr[n] != 0){
    		bAllZeros=FALSE;
    	}
    	if(pu8FlashPtr[n] != 0xff){
    		bAllOnes=FALSE;
    	}
    }

    if(bPassed){
    	vPrintf("Image in Sector %d\r\n",u32Sector);
    	vPrintf("* Configuration Byte A : %2x\r\n",pu8FlashPtr[12]);
    	vPrintf("* Configuration Byte B : %2x\r\n",pu8FlashPtr[13]);
    	uint16 u16appid = pu8FlashPtr[14];
    	u16appid = (u16appid*256) +pu8FlashPtr[15];
    	vPrintf("* Application ID : %4x\r\n",u16appid);
    	u16appid = pu8FlashPtr[30];
    	u16appid = (u16appid*256) +pu8FlashPtr[31];
    	vPrintf("* Software Config : %4x\r\n",u16appid);

    	uint32 ptr = 32;

    	uint32 u32value = pu8FlashPtr[ptr++];
    	u32value = (u32value*256) + pu8FlashPtr[ptr++];
    	u32value = (u32value*256) + pu8FlashPtr[ptr++];
    	u32value = (u32value*256) + pu8FlashPtr[ptr++];
    	vPrintf("* Image Length : %8x\r\n",u32value);

    	u32value = pu8FlashPtr[ptr++];
    	u32value = (u32value*256) + pu8FlashPtr[ptr++];
    	u32value = (u32value*256) + pu8FlashPtr[ptr++];
    	u32value = (u32value*256) + pu8FlashPtr[ptr++];
    	vPrintf("* Flash Start : %8x\r\n",u32value);

    	uint16 u16value = pu8FlashPtr[ptr++];
    	u16value = (u16value*256) + pu8FlashPtr[ptr++];
    	vPrintf("* Ram Start : %8x\r\n",u16value);

    	u16value = pu8FlashPtr[ptr++];
    	u16value = (u16value*256) + pu8FlashPtr[ptr++];
    	vPrintf("* Ram Length : %8x\r\n",u16value);

    	u16value = pu8FlashPtr[ptr++];
		u16value = (u16value*256) + pu8FlashPtr[ptr++];
		vPrintf("* BSS Start : %8x\r\n",u16value);

		u16value = pu8FlashPtr[ptr++];
		u16value = (u16value*256) + pu8FlashPtr[ptr++];
		vPrintf("* BSS Length : %8x\r\n",u16value);

    	u32value = pu8FlashPtr[ptr++];
    	u32value = (u32value*256) + pu8FlashPtr[ptr++];
    	u32value = (u32value*256) + pu8FlashPtr[ptr++];
    	u32value = (u32value*256) + pu8FlashPtr[ptr++];
    	vPrintf("* AppColdStart Entry Point : %8x\r\n",u32value);

    	u32value = pu8FlashPtr[ptr++];
    	u32value = (u32value*256) + pu8FlashPtr[ptr++];
    	u32value = (u32value*256) + pu8FlashPtr[ptr++];
    	u32value = (u32value*256) + pu8FlashPtr[ptr++];
    	vPrintf("* AppWarmStart Entry Point : %8x\r\n\r\n",u32value);
    }
    else{
    	if (bAllOnes){
    		NextValidSector = u32Sector;
    	    u32MagicAddress = u32Address;

    		vPrintf("No Image in Sector %d. Magic Bytes are all ones\r\n",u32Sector);
    	}
    	else if(bAllZeros){
    		vPrintf("No Image in Sector %d. Magic Bytes are all zeros\r\n",u32Sector);
    	}
    	else {
        	vPrintf("No Image in Sector %d\r\n",u32Sector);
    	}
    	return;
    }
}

/****************************************************************************
 *
 * NAME:       acGetC
 *
 * DESCRIPTION:
 * Reads a character from the uart connected to the pc. If no character is
 * waiting in the rx buffer, it will wait until there is.
 *
 *
 * RETURNS:
 * char, Character read from the UART
 *
 ****************************************************************************/
PRIVATE char acGetC(void)
{
    //wait for rx fifo empty (bit 5 set in LSR when empty)
    while ((u8AHI_UartReadLineStatus(E_AHI_UART_0) & E_AHI_UART_LS_DR ) != E_AHI_UART_LS_DR);
    return  u8AHI_UartReadData(E_AHI_UART_0);
}

PRIVATE void vValidateImage(uint32 u32Address)
{
	vPrintf("Enabling image at address %8x\r\n",u32Address);
	bAHI_FullFlashProgram(u32Address,16,au8MagicBytes);
}

PRIVATE void vInvalidateCurrentImage(void)
{
	bAHI_FullFlashProgram(0,16,au8ZeroBytes);
}

/****************************************************************************/
/***        END OF FILE                                                   ***/
/****************************************************************************/
