/*****************************************************************************
 *
 * MODULE:             JN-AN-xxxx
 *
 * COMPONENT:          nsc.c
 *
 * DESCRIPTION:        NFC Secure Commissioning
 *
 ****************************************************************************
 *
 * This software is owned by NXP B.V. and/or its supplier and is protected
 * under applicable copyright laws. All rights are reserved. We grant You,
 * and any third parties, a license to use this software solely and
 * exclusively on NXP products [NXP Microcontrollers such as JN5168, JN5164,
 * JN5161, JN5148, JN5142, JN5139].
 * You, and any third parties must reproduce the copyright and warranty notice
 * and any other legend of ownership on each copy or partial copy of the
 * software.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * Copyright NXP B.V. 2015. All rights reserved
 *
 ***************************************************************************/

/* Doxygen info, do not remove! */
/**
 * @file  nsc.c
 * @brief This file implements the NFC Secure Commissioning module.
 * \image html SW-LAYERS.JPG
 */

/****************************************************************************/
/***        Include files                                                 ***/
/****************************************************************************/

#include "app_config.h"

#ifdef NFC_COMMISSIONING

#include <string.h>
#include <jendefs.h>
#include "aessw_ccm.h"
#include "zps_apl_af.h"
#include "zps_apl_zdo.h"
#include "zps_nwk_sec.h"
#include "dbg.h"
#include "AppHardwareApi.h"
#include "ndef.h"
//#include "dump.h"
#include "nsc.h"
//#include "app_util.h"

#if defined(Router) || defined(HeatingPump) || defined(SmartPlug)
#include "zha_router_node.h"
#include "Controls.h"
#endif

#if defined(ZoneThermostatV2)
#include "zha_tstat_node.h"
#include "Controls.h"
#endif

#if defined(Light_ExtendedColorLight) || defined(Light_ColorTemperatureLight)
#include "zpr_light_node.h"
#endif

#if defined(DimmableLight) || defined(ColorDimmableLight) || defined(ColorTempTunableWhiteLight) || defined (OTAColorDimmableLightWithOccupancy)
#include "zha_light_node.h"
#include "Controls.h"
#endif

#if defined(DimmerSwitch)
#include "zha_switch_node.h"
#include "Controls.h"
#endif

#if defined(OTAServer)
#include "zha_upgrade_server_node.h"
#include "Controls.h"
#endif

/****************************************************************************/
/***        Type Definitions                                              ***/
/****************************************************************************/

/****************************************************************************/
//#define IOT_GATEWAY_DEVICE          0x02
//#define CLIMATE_MANAGER             0x03
#define ZIGBEE_ROUTER_DEVICE        0x04
#define CLIMATE_UI_SENSOR_DEVICE    0x20
//#define CLIMATE_SENSOR_DEVICE       0x22
#define CLIMATE_ACTUATOR_VALVE      0x23
#define ZIGBEE_SMART_PLUG           0x30
#define LAMP_DIMMABLE               0x41
#define LAMP_TUNABLE_WHITE          0x42
#define LAMP_RGB                    0x43
#define DIMMER_SWITCH               0x50
#define ZIGBEE_OTA_SERVER           0x04    // Must be changed into 0x60?

#if defined(Router)
    #define CURRENT_DEVICE          ZIGBEE_ROUTER_DEVICE
#elif defined(HeatingPump)
    #define CURRENT_DEVICE          CLIMATE_ACTUATOR_VALVE
#elif defined(ZoneThermostatV2)
    #define CURRENT_DEVICE          CLIMATE_UI_SENSOR_DEVICE
#elif defined(SmartPlug)
    #define CURRENT_DEVICE          ZIGBEE_SMART_PLUG
#elif defined(DimmableLight)
    #define CURRENT_DEVICE          LAMP_DIMMABLE
#elif defined(ColorDimmableLight)
    #define CURRENT_DEVICE          LAMP_RGB
#elif defined(OTAColorDimmableLightWithOccupancy)
    #define CURRENT_DEVICE          LAMP_RGB
#elif defined(Light_ExtendedColorLight)
    #define CURRENT_DEVICE          LAMP_RGB
#elif defined(ColorTempTunableWhiteLight)
    #define CURRENT_DEVICE          LAMP_TUNABLE_WHITE
#elif defined(DimmerSwitch)
    #define CURRENT_DEVICE          DIMMER_SWITCH
#elif defined(OTAServer)
    #define CURRENT_DEVICE          ZIGBEE_OTA_SERVER
#else
    #define CURRENT_DEVICE          0xFF    // Unknown
#endif
/****************************************************************************/

#define NSC_NDEF_DATA_VERSION       11

typedef struct nsc_ndef_out_struct
{
    /* Keep version as the first byte in the struct */
    uint8  Version;                             //!< Version number, identifies the total NSC NDEF message layout
    /* From this device to NFC reader device */
    uint8  From;                                //!< ID of this device
    uint8  SeqNr;                               //!< Sequence number of output NDEF message
    uint8  Fill[69];                            //!< Fill blank
    /* The next data is also known as the link_info */
    uint8  NodeMacAddress[8];                   //!< MAC address of this device
    uint8  NodeLinkKey[ZPS_SEC_KEY_LENGTH];     //!< Link key of this device
}  nsc_ndef_out_t;

typedef struct nsc_ndef_in_struct
{
    /* From NFC reader device to this device */
    uint8  From;                                //!< ID of NFC reader (phone or IoT gateway)
    uint8  SeqNr;                               //!< Sequence number of input NDEF message
    uint8  Command;                             //!< Command: None=0, FactoryNew=1, Join=2, SecureJoin=3, Leave=4
    uint8  Fill[66];                            //!< Fill blank
    uint8  Channel;                             //!< Channel number of active network
    uint8  KeySeq;                              //!< Active key sequence
    uint8  PanIdMSB;                            //!< Most significant byte of PAN ID
    uint8  PanIdLSB;                            //!< Least significant byte of PAN ID
    uint8  EncMic[ZPS_SEC_KEY_LENGTH];          //!< Encrypted MIC (bytes [12..15] also used for non encrypted MIC)
    uint8  EncKey[ZPS_SEC_KEY_LENGTH];          //!< Encrypted ZigBee network key
    uint8  ExtPanId[8];                         //!< Extended PAN ID of ZigBee network
    uint8  TrustCenterAddress[8];               //!< MAC address of ZigBee trust center
} nsc_ndef_in_t;

typedef struct nsc_ndef_struct
{
    nsc_ndef_out_t  DataOut;                    //!< Structure containing join output data from device
    nsc_ndef_in_t   DataIn;                     //!< Structure containing join input data to device
}  nsc_ndef_t;

// -----------------------
// COMMAND DEFINITIONS
// -----------------------

#define CMD_NONE                    0x00   // empty NDEF in_struct
#define CMD_FACTORY_NEW             0x01
#define CMD_JOIN_NETWORK            0x02   // not secure
#define CMD_SECURE_JOIN_NETWORK     0x03   // secure, using ECC
#define CMD_LEAVE_NETWORK           0x04

/****************************************************************************/
/***        Imported Variables                                            ***/
/****************************************************************************/


/****************************************************************************/
/***        Local Variables                                               ***/
/****************************************************************************/

#ifdef DEBUG_NSC
static bool_t bTraceNsc = TRUE;
#endif
const  uint8  au8NscNdefType[NSC_NDEF_TYPE_LENGTH] = {'c','o','m','.','n','x','p',':','w','i','s','e','0','1'};
static nsc_ndef_t  NscNdef;

/****************************************************************************/
/***        Local Functions                                               ***/
/****************************************************************************/

static void vNscConvert_u64_to_au8(uint64 u64dWord, uint8 *pu8Array)
{
    pu8Array[0] = u64dWord >>  0;
    pu8Array[1] = u64dWord >>  8;
    pu8Array[2] = u64dWord >> 16;
    pu8Array[3] = u64dWord >> 24;
    pu8Array[4] = u64dWord >> 32;
    pu8Array[5] = u64dWord >> 40;
    pu8Array[6] = u64dWord >> 48;
    pu8Array[7] = u64dWord >> 56;
}

static void vNscDecryptKey(uint8* pu8InKey, uint8* pu8LinkKey, uint8* au8Mic, uint8* pu8OutKey)
{
    uint64        u64MacAddr = ZPS_u64AplZdoGetIeeeAddr();
    AESSW_Block_u uNonce;
    AESSW_Block_u uEncKey;
    AESSW_Block_u uLnkKey;
    bool_t        bFlag = FALSE;

    memset((uint8 *)&uNonce, 0, sizeof(uNonce));
    memcpy((uint8 *)&uEncKey, pu8InKey,   sizeof(uEncKey));
    memcpy((uint8 *)&uLnkKey, pu8LinkKey, sizeof(uLnkKey));
    vNscConvert_u64_to_au8(u64MacAddr, uNonce.au8);
    AESSW_bCCMstar(&uLnkKey, TRUE, XCV_REG_AES_SET_MODE_CCM_D, 4, 0, 16, &uNonce,
                   (uint8*)&uEncKey, (uint8*)&uEncKey, (uint8*)&uEncKey, au8Mic, &bFlag);
    memcpy(pu8OutKey, (uint8 *)&uEncKey, sizeof(uEncKey));
}

static void vNscGenerateNdefMessage(bool bClearInStruct)
{
    NscNdef.DataOut.SeqNr += 1;
    teNDEF_Status status = eNdefLockNtag(au8NscNdefType);
    if (E_NDEF_SUCCESS == status)
    {
        uint32 u32NdefOffset = 0;
        if (bClearInStruct)
        {
#ifdef DEBUG_NSC
            DBG_vPrintf(bTraceNsc, "NSC: WRITE -> ECC pub key + link_info + clear in_struct\n");
#endif
            // Clear the in_struct
            memset(&NscNdef.DataIn, 0, sizeof(nsc_ndef_in_t));
            // Write the out_struct plus the cleared in_struct to the commissioning NDEF message
            status = eNdefWriteMsg(au8NscNdefType, u32NdefOffset, (uint8 *)&NscNdef, sizeof(nsc_ndef_t));
        }
        else
        {
#ifdef DEBUG_NSC
            DBG_vPrintf(bTraceNsc, "NSC: WRITE -> ECC pub key + link_info\n");
#endif
            // Only write the out_struct to the commissioning NDEF message. Don't touch the in_struct!
            status = eNdefWriteMsg(au8NscNdefType, u32NdefOffset, (uint8 *)&NscNdef.DataOut, sizeof(nsc_ndef_out_t));
        }
        eNdefUnlockNtag(au8NscNdefType);
    }
    if (E_NDEF_SUCCESS != status)
    {
#ifdef DEBUG_NSC
        DBG_vPrintf(TRUE, "NSC: Error writing NDEF payload\n");
#endif
    }
}

static void vNscProcessNdefMessage(ndef_rec_info_t *pNdefRecInfo)
{
    if (E_NDEF_SUCCESS == eNdefLockNtag(au8NscNdefType))
    {
        uint32 u32NdefOffset = sizeof(nsc_ndef_out_t);
        if (E_NDEF_SUCCESS == eNdefReadMsg(au8NscNdefType, u32NdefOffset, (uint8 *)&NscNdef.DataIn, sizeof(nsc_ndef_in_t)))
        {
            switch (NscNdef.DataIn.Command)
            {
            case CMD_NONE:
                break;

#ifdef NFC_ALLOW_FACTORY_NEW_CMD
            case CMD_FACTORY_NEW:
  #ifdef DEBUG_NSC
                DBG_vPrintf(bTraceNsc, "NSC: READ  -> factory new\n");
  #endif
                break;
#endif  // NFC_ALLOW_FACTORY_NEW_CMD

#ifdef NFC_ALLOW_UNSECURE_JOIN_NWK
            case CMD_JOIN_NETWORK:
  #ifdef DEBUG_NSC
                DBG_vPrintf(bTraceNsc, "NSC: READ  -> join network\n" );
  #endif
                break;
#endif  // NFC_ALLOW_UNSECURE_JOIN_NWK

#ifdef NFC_ALLOW_LEAVE_NWK
            case CMD_LEAVE_NETWORK:
  #ifdef DEBUG_NSC
                DBG_vPrintf(bTraceNsc, "NSC: READ  -> leave network\n");
  #endif
                break;
#endif  // NFC_ALLOW_LEAVE_NWK

            default:
#ifdef DEBUG_NSC
                DBG_vPrintf(TRUE, "NSC: READ  -> UNKNOWN CMD %d !!!!!!\n", NscNdef.DataIn.Command);
#endif
                NscNdef.DataIn.Command = CMD_NONE;
                break;
            }
        }
        eNdefUnlockNtag(au8NscNdefType);
    }
}

/****************************************************************************/
/***        Exported Functions                                            ***/
/****************************************************************************/

/* Doxygen info, do not remove! */
/**
 * @brief   This function initializes the NSC module (this source file), and
 *          registers a NDEF payload type with the NDEF layer driver.
 * @return  None.
 * @warning THIS FUNCTION MUST NOT BE CALLED BY THE APPLICATION BECAUSE
 *          IT IS CALLED BY FUNCTION vNfcInit() ALREADY AT STARTUP !!!
 */

void vNscInit(void)
{
    memset(&NscNdef, 0, sizeof(nsc_ndef_t));
    eNdefRegisterMsg(au8NscNdefType,
                     sizeof(au8NscNdefType),
                     NULL,   // No NDEF payload identifier
                     0,      // Length of NDEF payload identifier is zero
                     sizeof(nsc_ndef_out_t) + sizeof(nsc_ndef_in_t),
                     vNscProcessNdefMessage);
}

/* Doxygen info, do not remove! */
/**
 * @brief   This function takes care for the actual joining or leaving just
 *          after startup, and initiates the writing of ECC public key and
 *          link_info into the NTAG.
 * @return  None.
 * @note    This function may only be called after JenOS is started, because
 *          of API calls to JenOS.
 * @warning THIS FUNCTION MUST NOT BE CALLED BY THE APPLICATION BECAUSE
 *          IT IS CALLED BY FUNCTION vNfcInitPostProcessing() ALREADY AT STARTUP !!!
 */

void vNscInitPostProcessing(void)
{
    uint8   Command = NscNdef.DataIn.Command;       // First remember the incoming command !!!
    int     i;
    bool    bEncMicInvalid = FALSE;
    uint64  mac_address = ZPS_u64AplZdoGetIeeeAddr();
    uint8  *pmac = (uint8*)&mac_address;

    NscNdef.DataIn.Command  = CMD_NONE;
    NscNdef.DataOut.Version = NSC_NDEF_DATA_VERSION;
    NscNdef.DataOut.From    = CURRENT_DEVICE;       // It would have been better to use the ZigBee device_ID here
    for (i=0; i<8; i++)
    {
        NscNdef.DataOut.NodeMacAddress[i] = *pmac++;
    }
    // Read the link_key from the EEPROM of JN5168
    if (getLinkKey(NscNdef.DataOut.NodeLinkKey))
    {
        // link_key is zero, generate one now
        i = 0;
        while( i < ZPS_SEC_KEY_LENGTH )
        {
            vAHI_StartRandomNumberGenerator(E_AHI_RND_SINGLE_SHOT, E_AHI_INTS_DISABLED);
            while(!bAHI_RndNumPoll());
            uint16 randomNumber = u16AHI_ReadRandomNumber();
            NscNdef.DataOut.NodeLinkKey[i++] = (uint8)(randomNumber >> 0) & 0xFF;
            NscNdef.DataOut.NodeLinkKey[i++] = (uint8)(randomNumber >> 8) & 0xFF;
        }
        // Store this link_key in EEPROM of JN5168
        setLinkKey(NscNdef.DataOut.NodeLinkKey);
    }
    // At this stage, the first 12 bytes of the global variable EncMic should be zero
    for (i=0; i<12; i++)
    {
        if (0 != NscNdef.DataIn.EncMic[i])
        {
            bEncMicInvalid = TRUE;
        }
    }

    switch (Command)
    {
    case CMD_FACTORY_NEW:
        vNscGenerateNdefMessage(TRUE); // clear the in_struct which holds the FACTORY NEW command
        APP_vResetPDM();
        vAHI_SwReset();
        break;

    case CMD_JOIN_NETWORK:
    case CMD_SECURE_JOIN_NETWORK:
        if (!APP_bDeviceJoined())
        {
            uint8  SecureKey[ZPS_SEC_KEY_LENGTH];
            uint16 u16PanID = (NscNdef.DataIn.PanIdMSB << 8) | NscNdef.DataIn.PanIdLSB;
            uint64 u64ExtPanid  = 0;
            uint64 u64TcAddress = 0;
            vNscGenerateNdefMessage(FALSE); // Repair out_struct but do not clear the in_struct till we finally joined the network
            for (i=0; i<8; i++)
            {
                u64ExtPanid  = u64ExtPanid << 8;
                u64ExtPanid  = u64ExtPanid + NscNdef.DataIn.ExtPanId[i];
                u64TcAddress = u64TcAddress << 8;
                u64TcAddress = u64TcAddress + NscNdef.DataIn.TrustCenterAddress[i];
            }
#ifdef DEBUG_NSC
            if (bTraceNsc)
            {
                DBG_vPrintf(TRUE, "NSC: Channel        = %d\n",             NscNdef.DataIn.Channel);
                DBG_vPrintf(TRUE, "NSC: Active key seq = 0x%02x\n",         NscNdef.DataIn.KeySeq);
                DBG_vPrintf(TRUE, "NSC: PAN ID         = 0x%04x\n",         u16PanID);
                DBG_vPrintf(TRUE, "NSC: Ext PAN ID     = 0x%016llx\n",      u64ExtPanid);
                DBG_vPrintf(TRUE, "NSC: TC address     = 0x%016llx\n",      u64TcAddress);
                //dump_bytes(       "NSC: MIC            = ", (char*)&NscNdef.DataIn.EncMic[12], 4);
                //dump_bytes(       "NSC: Link key       = ", (char*)NscNdef.DataOut.NodeLinkKey, ZPS_SEC_KEY_LENGTH);
               // dump_bytes(       "NSC: Enc NWK key    = ", (char*)NscNdef.DataIn.EncKey, ZPS_SEC_KEY_LENGTH);
            }
#endif
            // Now do the AES decryption of the encrypted ZigBee network key, this results in the SecureKey (aka decrypted network key)
            vNscDecryptKey(NscNdef.DataIn.EncKey, NscNdef.DataOut.NodeLinkKey, &NscNdef.DataIn.EncMic[12], SecureKey);
#ifdef DEBUG_NSC
            if (bTraceNsc)
            {
                  // WARNING: The (decrypted) network key must be kept secret at all time,
                  //          otherwise your network is vulnerable for violent attacks!!!
//                dump_bytes(       "NSC: Dec NWK key    = ", (char*)SecureKey, ZPS_SEC_KEY_LENGTH);
            }
#endif
            APP_vRequestSecureJoin(SecureKey,
                                   u16PanID,
                                   u64ExtPanid,
                                   NscNdef.DataIn.Channel,
                                   NscNdef.DataIn.KeySeq,
                                   u64TcAddress);
            // Now it is up to the ZigBee stack to re-join this device in the network
        }
        else
        {
            uint16 u16PanID = (NscNdef.DataIn.PanIdMSB << 8) | NscNdef.DataIn.PanIdLSB;
#ifdef DEBUG_NSC
            DBG_vPrintf(bTraceNsc, "NSC: (2) ALREADY JOINED IN A NETWORK !!!\n");
#endif
#ifdef NFC_ALLOW_JOIN_OTHER_NWK
            if (ZPS_u16AplZdoGetNetworkPanId() == u16PanID)
            {
  #ifdef DEBUG_NSC
                DBG_vPrintf(bTraceNsc, "NSC: Same network, so do nothing\n");
  #endif
                APP_vAnnounce(); // Let everybody know that we are still here
                vNscGenerateNdefMessage(TRUE); // Repair out_struct and clear the in_struct which holds the JOIN command
            }
            else if (bEncMicInvalid)
            {
  #ifdef DEBUG_NSC
                DBG_vPrintf(bTraceNsc, "NSC: Invalid EncMic, so stay in current network\n");
  #endif
                vNscGenerateNdefMessage(TRUE); // Repair out_struct and clear the in_struct which holds the JOIN command
            }
            else
            {
                vNscGenerateNdefMessage(FALSE); // Repair out_struct but do not clear the in_struct
                // Delete the current network information from the PDM (abruptly leave/forget the network)
                // but keep any application data that is in the PDM
                APP_vResetPDM();
                // Write link_key back in PDM, because we need this for the AES decryption of the network key
                setLinkKey(NscNdef.DataOut.NodeLinkKey);
                // Reboot the system so that the secure_join_struct data can be
                // decrypted while JenOS isn't running. This to have enough RAM
                // available to do the decryption.
                vAHI_SwReset();
            }
#else   // NFC_ALLOW_JOIN_OTHER_NWK
            // Already in a network and not allowed to join an other network
            APP_vAnnounce(); // Let everybody know that we are still here
            vNscGenerateNdefMessage(TRUE); // Repair out_struct and clear the in_struct which holds the JOIN command
#endif  // NFC_ALLOW_JOIN_OTHER_NWK
        }
        break;

    case CMD_LEAVE_NETWORK:
        if (APP_bDeviceJoined())
        {
            vNscGenerateNdefMessage(FALSE); // Repair out_struct but do not clear the in_struct till we finally left the network
            APP_vRequestDecommission();     // try to do an orderly leave of the network
            // Now it is up to the ZigBee stack to let this device orderly leave the network
        }
        else
        {
            // Not in a network, so do nothing
            vNscGenerateNdefMessage(TRUE); // Repair out_struct and clear the in_struct which holds the LEAVE command
        }
        break;

    default:
        vNscGenerateNdefMessage(TRUE); // Repair out_struct and clear the in_struct anyway
        break;
    }
}

/* Doxygen info, do not remove! */
/**
 * @brief   This function initiates the NFC Secure Commissioning when a join
 *          or leave command have just been written in the NTAG via the RF side
 *          of the NTAG IC.
 * @return  None.
 * @note    This function may only be called after JenOS is started, because
 *          of API calls to JenOS.
 * @warning THIS FUNCTION MUST NOT BE CALLED BY THE APPLICATION BECAUSE
 *          IT IS CALLED BY FUNCTION vNfcLoop() (see nfc.c) WHEN THE NFC RF
 *          FIELD GOES AWAY FROM THE NTAG ANTENNA !!!
 */

void vNscActOnCommand(void)
{
    uint8 Command = NscNdef.DataIn.Command;  // First remember the incoming command !!!
    NscNdef.DataIn.Command = CMD_NONE;
    switch (Command)
    {
    case CMD_FACTORY_NEW:
        vNscGenerateNdefMessage(TRUE); // clear the in_struct which holds the FACTORY NEW command
        APP_vResetPDM();
        vAHI_SwReset();
        break;

    case CMD_JOIN_NETWORK:
    case CMD_SECURE_JOIN_NETWORK:
        if (!APP_bDeviceJoined())
        {
            // Reboot the system so that the secure_join_struct data can be
            // decrypted while JenOS isn't running. This to have enough RAM
            // available to do the decryption. So don't touch the NDEF message.
            vAHI_SwReset();
        }
        else
        {
#ifdef DEBUG_NSC
            DBG_vPrintf(bTraceNsc, "NSC: (1) ALREADY JOINED IN A NETWORK !!!\n");
#endif
#ifdef NFC_ALLOW_JOIN_OTHER_NWK
            // Reboot the system so that the secure_join_struct data can be
            // decrypted while JenOS isn't running. This to have enough RAM
            // available to do the decryption. So don't touch the NDEF message.
            vAHI_SwReset();
#else   // NFC_ALLOW_JOIN_OTHER_NWK
            // Already in a network and not allowed to join an other network
            APP_vAnnounce(); // Let everybody know that we are still here
            vNscGenerateNdefMessage(TRUE); // Repair out_struct and clear the in_struct which holds the JOIN command
#endif  // NFC_ALLOW_JOIN_OTHER_NWK
        }
        break;

    case CMD_LEAVE_NETWORK:
        if (APP_bDeviceJoined())
        {
            APP_vRequestDecommission();     // try to do an orderly leave of the network
            // Now it is up to the ZigBee stack to let this device orderly leave the network
        }
        else
        {
            // Not in a network, so do nothing
            vNscGenerateNdefMessage(TRUE); // Repair out_struct and clear the in_struct which holds the LEAVE command
        }
        break;

    default:
        vNscGenerateNdefMessage(TRUE); // Repair out_struct and clear the in_struct anyway
        break;
    }
}

/* Doxygen info, do not remove! */
/**
 * @brief   This function initiates the writing of NFC secure join/leave
 *          info into the NTAG, and clears the join command from the NTAG.
 * @return  None.
 * @warning THIS FUNCTION MUST NOT BE CALLED BY THE APPLICATION BECAUSE
 *          IT IS CALLED AS A RESULT OF A JOIN EVENT FROM THE ZIGBEE STACK.
 */

void vNscJoinNetworkFinished(void)
{
#ifdef DEBUG_NSC
    DBG_vPrintf(bTraceNsc, "NSC: JOINED THE NETWORK !!!\n");
#endif
    vNscGenerateNdefMessage(TRUE); // Repair out_struct and clear the in_struct which holds the JOIN command
}

/* Doxygen info, do not remove! */
/**
 * @brief   This function initiates the writing of NFC secure join/leave
 *          info into the NTAG, clears the leave command from the NTAG,
 *          and clears the ECC persistent data.
 * @return  None.
 * @warning THIS FUNCTION MUST NOT BE CALLED BY THE APPLICATION BECAUSE
 *          IT IS CALLED AS A RESULT OF A LEAVE EVENT FROM THE ZIGBEE STACK.
 */

void vNscLeaveNetworkFinished(void)
{
#ifdef DEBUG_NSC
    DBG_vPrintf(bTraceNsc, "NSC: LEFT THE NETWORK !!!\n");
#endif
    vNscGenerateNdefMessage(TRUE); // clear the in_struct which holds the LEAVE command
}


void vNscToggleDBG(void)
{
#ifdef DEBUG_NSC
    bTraceNsc = !bTraceNsc;
#endif
}

#endif  // NFC_COMMISSIONING

/****************************************************************************/
/***        END OF FILE                                                   ***/
/****************************************************************************/
