/*****************************************************************************
 *
 * MODULE:             JN-AN-xxxx
 *
 * COMPONENT:          nfc.c
 *
 * DESCRIPTION:        NFC interface implementation
 *
 ****************************************************************************
 *
 * This software is owned by NXP B.V. and/or its supplier and is protected
 * under applicable copyright laws. All rights are reserved. We grant You,
 * and any third parties, a license to use this software solely and
 * exclusively on NXP products [NXP Microcontrollers such as JN5168, JN5164,
 * JN5161, JN5148, JN5142, JN5139].
 * You, and any third parties must reproduce the copyright and warranty notice
 * and any other legend of ownership on each copy or partial copy of the
 * software.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * Copyright NXP B.V. 2015. All rights reserved
 *
 ***************************************************************************/

/* Doxygen info, do not remove! */
/**
 * @file  nfc.c
 * @brief This file implements the NFC loop task.
 * \image html SW-LAYERS.JPG
 */

/****************************************************************************/
/***        Include files                                                 ***/
/****************************************************************************/

#include "app_config.h"
#include <string.h>
#include <jendefs.h>
#include "os.h"
#include "os_gen.h"

//#ifdef NFC_SUPPORT

#include "aessw_ccm.h"
#include "dbg.h"
#include "AppHardwareApi.h"
#include "ntag.h"
#include "ndef.h"
#include "nfc.h"
#include "I2C_Driver.h"
#ifdef NFC_COMMISSIONING
#include "nsc.h"
#endif  // NFC_COMMISSIONING

/****************************************************************************/
/***        Type Definitions                                              ***/
/****************************************************************************/
#define APP_TIME_MS(t)  (16000UL * (t))
#define MSECS_NTAG_FD_DELAY      APP_TIME_MS(500)    // in milli seconds
#define MSECS_NFC_LOOP_SLEEP     APP_TIME_MS(100)    // in milli seconds
#define CYCLES_BEFORE_STOP_NFC   5

#define END_BLOCK_NR   NTAG_LAST_USER_MEM_BLOCK

/****************************************************************************/
/***        Local Function prototypes                                     ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Variables                                               ***/
/****************************************************************************/

#ifdef DEBUG_NFC
static bool_t bTraceNfc = TRUE;
#endif
static bool_t bUseFDpin;
static bool_t bNfcLoopRunning;
static uint32 rf_not_present_cnt;
//static uint32 NfcLoopCnt;


/****************************************************************************/
/***        Local Functions                                               ***/
/****************************************************************************/

static void vNfcInitGlobals(void)
{
    bUseFDpin       = FALSE;
    bNfcLoopRunning = FALSE;
    rf_not_present_cnt = 0;
//    NfcLoopCnt         = 1;
}

// --------------------------------------------------------------------------
// POWER CYCLE THE NTAG I2C IC AND PROGRAM VOLATILE SESSION REGISTERS
// --------------------------------------------------------------------------

static bool bNfcResetNtag(ntag_power_cmd_t mode)
{
    bool bRetval = FALSE;
    i2c_LockBus();
    if (bNtagPower(mode))
    {
        // set watchdog timer in NTAG I2C IC as long as possible (>600ms)
        if (bNtagSetup(0xFFFF, END_BLOCK_NR))
        {
            bRetval = TRUE;
        }
        else
        {
#ifdef DEBUG_NFC
            DBG_vPrintf(TRUE, "NFC: bNtagSetup FAILED\n");
#endif
        }
    }
    else
    {
#ifdef DEBUG_NFC
        DBG_vPrintf(TRUE, "NFC: bNtagPower FAILED\n");
#endif
    }
    i2c_UnlockBus();
    return bRetval;
}

// --------------------------------------------------------------------------
// Read and handle incoming NDEF messages
// --------------------------------------------------------------------------

static void vNfcLoop(bool bJenOSactive)
{
    bool  bExitNfcLoop = !bJenOSactive;  // No looping when JenOS is not running
    uint8 regdata = 0xFF;
    // Read the NTAG status register NS_REG
    i2c_LockBus();
    if (!bNtagGetNsReg(&regdata))
    {
#ifdef DEBUG_NFC
        DBG_vPrintf(TRUE, "NFC: Could not read NS_REG\n");
#endif
        regdata = 0;          // don't act on this status
    }
    i2c_UnlockBus();
 
#ifdef DEBUG_NFC
//    DBG_vPrintf(bTraceNfc, "NFC: >>>>> NFC loop (%d), NS_REG=0x%02x\n", NfcLoopCnt++, regdata);
#endif

    //---------------------------------
    // Check for abnormal NS_REG values
    //---------------------------------
    if (regdata == 0xFF)
    {
#ifdef DEBUG_NFC
        DBG_vPrintf(TRUE, "NFC: I2C data line stays high (NS_REG=0x%02x)\n", regdata);
#endif
        regdata = 0;          // don't act on this status any longer
    }
    if (regdata == 0xAB)
    {
#ifdef DEBUG_NFC
        DBG_vPrintf(TRUE, "NFC: NTAG reports strange status (NS_REG=0x%02x)\n", regdata);
#endif
        regdata = 0;          // don't act on this status any longer
        bExitNfcLoop = TRUE;
    }
    if (regdata & NS_REG_I2C_LOCKED)
    {
#ifdef DEBUG_NFC
        DBG_vPrintf(TRUE, "NFC: I2C_LOCKED!!!\n"); // should never see this message
#endif
    }

    //--------------------------------
    // Check if RF field still present
    //--------------------------------
    if (regdata & NS_REG_RF_FIELD_PRESENT)
    {
        rf_not_present_cnt = 0;
    }
    else
    {
        // No field
        rf_not_present_cnt++;
        if (rf_not_present_cnt >= CYCLES_BEFORE_STOP_NFC)
        {
            bExitNfcLoop = TRUE;
        }
    }

    if (bExitNfcLoop)
    {
    	vNdefParser();
#ifdef DEBUG_NFC
        DBG_vPrintf(bTraceNfc, "NFC: EXIT  NFC loop <<<<<<<\n");
#endif
        rf_not_present_cnt = 0;
        bNfcLoopRunning = FALSE;
        if (bJenOSactive)
        {
#ifdef ZoneThermostatV2
            bNfcResetNtag(NTAG_POWER_OFF_ON);
#endif
#ifdef NFC_COMMISSIONING
            vNscActOnCommand();
#endif  // NFC_COMMISSIONING
        }
        if (bUseFDpin)
        {
#ifdef NTAG_FD_PIN
            vAHI_DioInterruptEnable((1 << NTAG_FD_PIN), 0); // Enable NTAG FD IRQ
#endif  // NTAG_FD_PIN
        }
    }
    else
    {
        // Restart NFC loop after MSECS_NFC_LOOP_SLEEP
        OS_eStartSWTimer(APP_NFCLoopTimer, MSECS_NFC_LOOP_SLEEP, NULL);
    }
}

OS_TASK(APP_NFCLoopTask)
{
	OS_eStopSWTimer(APP_NFCLoopTimer);
    vNfcLoop(TRUE);  // JenOS running
}

/****************************************************************************/
/***        Exported Functions                                            ***/
/****************************************************************************/

/* Doxygen info, do not remove! */
/**
 * @brief   This function initializes the NFC module (this source file),
 *          initializes the ntag layer (source file ntag.c), and takes care
 *          of initializing the NFC Secure Commissioning module (source file
 *          nsc.c) when NFC_COMMISSIONING is enabled. After the NTAG-I2C IC
 *          has been successfully initialized, the EEPROM of the NTAG is
 *          read out to check whether there is a Secure Join instruction
 *          in the NTAG.
 * @return  None.
 * @note    This function must be called before JenOS is started, because
 *          the Elliptic Curve decryption must be done before JenOS is
 *          started because of the amount of RAM (heap and stack) that is
 *          needed for this decryption.
 * @warning THIS FUNCTION MUST NOT BE CALLED BY THE APPLICATION BECAUSE
 *          IT IS CALLED BY FUNCTION vAppMain() ALREADY AT STARTUP !!!
 */

void vNfcInit(void)
{
    bool_t bSucces = FALSE;
    vNfcInitGlobals();
#ifdef ZoneThermostatV2
    bSucces = bNfcResetNtag(NTAG_POWER_OFF_ON);
#else
    bSucces = bNfcResetNtag(NTAG_POWER_ON);
#endif
#ifdef NFC_COMMISSIONING
    vNscInit();
#endif  // NFC_COMMISSIONING
    if (bSucces)
    {
        // Start the NFC loop to read data from NTAG
#ifdef DEBUG_NFC
        DBG_vPrintf(bTraceNfc, "NFC: ENTER NFC loop >>>>>>>\n");
#endif
        bNfcLoopRunning = TRUE;
        vNfcLoop(FALSE);  // JenOS not running
    }
}

/* Doxygen info, do not remove! */
/**
 * @brief   This function does the NFC Secure Commissioning post processing
 *          (when NFC_COMMISSIONING is enabled) after reset of the CPU, as
 *          post processing of function vNfcInit(). Optionally it enables the
 *          IRQ detection for the field detect (FD) pin of the NTAG
 * @param   bUseNTAGirqs - When the FD pin of the NTAG is connected to a DIO
 *          pin of the JN516x, then this boolean can be set to TRUE to enable
 *          IRQ detection.
 * @return  None.
 * @note    This function may only be called after JenOS is started, because
 *          of API calls to JenOS.
 * @warning THIS FUNCTION MUST NOT BE CALLED BY THE APPLICATION BECAUSE
 *          IT IS CALLED BY FUNCTION APP_vInitialiseNode() ALREADY AT STARTUP !!!
 */

void vNfcInitPostProcessing(bool bUseNTAGirqs)
{
#ifdef NFC_COMMISSIONING
    vNscInitPostProcessing();
#endif  // NFC_COMMISSIONING
    if (bUseNTAGirqs)
    {
        bUseFDpin = TRUE;
#ifdef NTAG_FD_PIN
        vAHI_DioSetDirection((1 << NTAG_FD_PIN), 0);    // Set NTAG FD INT as input
        vAHI_DioSetPullup((1 << NTAG_FD_PIN), 0);
        vAHI_DioInterruptEdge(0, (1 << NTAG_FD_PIN));   // Set the edge detection for falling edge
        vAHI_DioInterruptEnable((1 << NTAG_FD_PIN), 0); // Enable NTAG FD IRQ
#endif  // NTAG_FD_PIN
    }
}

/* Doxygen info, do not remove! */
/**
 * @brief   This function is called when the Field Detect (FD) pin of the
 *          NTAG-I2C IC goes low, because of the presence of a NFC field.
 * @return  None.
 * @note    This function may only be called after JenOS is started, because
 *          of API calls to JenOS.
 * @warning THIS FUNCTION MUST NOT BE CALLED BY THE APPLICATION BECAUSE
 *          IT IS CALLED BY OS_ISR(vISR_SystemController) ALREADY !!!
 */

void vNfcTagFdPinIRQ(void)
{
    if (!bNfcLoopRunning)
    {
        bNfcLoopRunning = TRUE;
        if (bUseFDpin)
        {
#ifdef NTAG_FD_PIN
            vAHI_DioInterruptEnable(0, (1 << NTAG_FD_PIN)); // Disable NTAG FD IRQ
#endif  // NTAG_FD_PIN
        }
#ifdef DEBUG_NFC
        DBG_vPrintf(bTraceNfc, "NFC: ENTER NFC loop (IRQ) >>>\n");
#endif
        // Don't start the NFC loop immediately. Give the FD line time to settle,
        // and give the NFC reader (smart phone) time to read the NDEF message first.
        OS_eStartSWTimer(APP_NFCLoopTimer, MSECS_NTAG_FD_DELAY, NULL);
    }
}

void vNfcToggleDBG(void)
{
#ifdef DEBUG_NFC
    bTraceNfc = !bTraceNfc;
#endif
}

//#endif  // NFC_SUPPORT

/****************************************************************************/
/***        END OF FILE                                                   ***/
/****************************************************************************/
