/*****************************************************************************
 *
 * MODULE:             JN-AN-xxxx
 *
 * COMPONENT:          I2C_Driver.c
 *
 * DESCRIPTION:        I2C driver
 *
 *****************************************************************************
 *
 * This software is owned by NXP B.V. and/or its supplier and is protected
 * under applicable copyright laws. All rights are reserved. We grant You,
 * and any third parties, a license to use this software solely and
 * exclusively on NXP products [NXP Microcontrollers such as JN5168, JN5164,
 * JN5161, JN5148, JN5142, JN5139].
 * You, and any third parties must reproduce the copyright and warranty notice
 * and any other legend of ownership on each copy or partial copy of the
 * software.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * Copyright NXP B.V. 2015. All rights reserved
 *
 ****************************************************************************/

/* Doxygen info, do not remove! */
/**
 * @file  I2C_Driver.c
 * @brief This file implements the I2C driver.
 */

/****************************************************************************/
/***        Include files                                                 ***/
/****************************************************************************/

#include "app_config.h"

#ifdef I2C_SUPPORT

#include "I2C_Driver.h"
#include <AppHardwareApi.h>
#include "dbg.h"
#include "os.h"
#include "os_gen.h"

/****************************************************************************/
/***        Type Definitions                                              ***/
/****************************************************************************/

/****************************************************************************/
/***        Global variables                                              ***/
/****************************************************************************/

#ifdef DEBUG_I2C_DRIVER
static bool_t bTraceI2c = FALSE;
#endif
static bool_t bBusLockingEnabled;

/****************************************************************************/
/***        Exported Functions                                            ***/
/****************************************************************************/

/* Doxygen info, do not remove! */
/**
 * @brief   This function initializes the I2C HW block of the JN516x.
 *          When multiple I2C slaves are connected to the I2C bus, concurrent
 *          access to these slaves via this driver must be prevented by a semaphore.
 *          However, since these semaphores are only available when JenOS is started,
 *          the boolean bBusLockingEnabled is introduced to make access to these
 *          slaves possible, even before JenOS is started. After this init the
 *          I2C bus locking is enabled. Use the function i2c_EnableBusLocking() to
 *          disable/enable I2C bus locking when needed before JenOS is started.
 * @param   bUseDio16Dio17 - Boolean to select I2C SDA and SCL lines (TRUE= DIO16-17, FALSE= DIO14-15)
 * @param   bI2Cclk400khz  - Boolean to select between two I2C bus clock speeds (TRUE=400KHz, FALSE=100KHz)
 * @return  None.
 */

void i2c_init(bool bUseDio16Dio17, bool bI2Cclk400khz)
{
#ifdef DEBUG_I2C_DRIVER
    DBG_vPrintf(bTraceI2c, "i2c_init: Enter\n");
#endif
    uint8 u8PreScaler = 31;  // default 100KHz

    bBusLockingEnabled = TRUE;

    // i2c bus clk = 16MHz / ((Prescaler + 1) * 5)
    //   or
    // Prescaler = ((16MHz / i2c bus clk) / 5) - 1
    // Prescaler = ((16000kHz / 400kHz) / 5) - 1 = ( 40 / 5) - 1 =  8 - 1 = 7
    // Prescaler = ((16000kHz / 100kHz) / 5) - 1 = (160 / 5) - 1 = 32 - 1 = 31
    // Prescaler = ((16000kHz /  80kHz) / 5) - 1 = (200 / 5) - 1 = 40 - 1 = 39
    // Prescaler = ((16000kHz /  40kHz) / 5) - 1 = (400 / 5) - 1 = 80 - 1 = 79
    if (bI2Cclk400khz)
    {
        u8PreScaler = 7;    // run I2C bus at 400KHz
    }
    else
    {
#if defined(ZoneThermostatV2)
//        u8PreScaler = 79;   // run I2C bus at 40kHz
        u8PreScaler = 39;   // run I2C bus at 80kHz
#else
        u8PreScaler = 31;   // run I2C bus at 100KHz
#endif
    }
    vAHI_SiMasterConfigure(/* bPulseSuppressionEnable */ TRUE,
                           /* bInterruptEnable */ FALSE,
                           /* Prescaler */ u8PreScaler);
    vAHI_SiSetLocation(bUseDio16Dio17);   // TRUE=use DIO16-17, FALSE=use DIO14-15 (default)

#ifdef DEBUG_I2C_DRIVER
    DBG_vPrintf(bTraceI2c, "i2c_init: Leave\n");
#endif
}

/* Doxygen info, do not remove! */
/**
 * @brief   This function sets the internal boolean bBusLockingEnabled.
 * @param   bEnable - This enables or disables the use of the I2C semaphore to do bus locking
 * @return  None.
 */

void i2c_EnableBusLocking(bool_t bEnable)
{
    bBusLockingEnabled = bEnable;
}

/* Doxygen info, do not remove! */
/**
 * @brief   This function locks the I2C bus depending on the internal boolean bBusLockingEnabled.
 * @return  None.
 */

void i2c_LockBus(void)
{
#ifdef DEBUG_I2C_DRIVER
    DBG_vPrintf(bTraceI2c, "i2c_LockBus: Enter\n");
#endif
    if (bBusLockingEnabled)
    {
#ifdef I2C_USE_MUTEX
        OS_eEnterCriticalSection(mutexI2Cbus);
#endif
    }
#ifdef DEBUG_I2C_DRIVER
    DBG_vPrintf(bTraceI2c, "i2c_LockBus: Leave\n");
#endif
}

/* Doxygen info, do not remove! */
/**
 * @brief   This function unlocks the I2C bus depending on the internal boolean bBusLockingEnabled.
 * @return  None.
 */

void i2c_UnlockBus(void)
{
#ifdef DEBUG_I2C_DRIVER
    DBG_vPrintf(bTraceI2c, "i2c_UnlockBus: Enter\n");
#endif
    if (bBusLockingEnabled)
    {
#ifdef I2C_USE_MUTEX
        OS_eExitCriticalSection(mutexI2Cbus);
#endif
    }
#ifdef DEBUG_I2C_DRIVER
    DBG_vPrintf(bTraceI2c, "i2c_UnlockBus: Leave\n");
#endif
}

/* Doxygen info, do not remove! */
/**
 * @brief   This function writes data to a I2C slave.
 * @param   u8Address - 7 bit I2C slave address
 * @param   u8Command - I2C slave command (this can indicate a register)
 * @param   u8Length - Length of the data
 * @param   pu8Data - Pointer to the data
 * @return  TRUE when data has been written successfully.
 */

bool i2c_BusWriteReg(uint8 u8Address, uint8 u8Command, uint8 u8Length, uint8* pu8Data)
{
#ifdef DEBUG_I2C_DRIVER
    DBG_vPrintf(bTraceI2c, "i2c_BusWriteReg: Enter\n");
#endif

    // First write I2C address + WRITE BIT on I2C bus
    vAHI_SiMasterWriteSlaveAddr(u8Address, /* bReadStatus */ FALSE);
    if (!bAHI_SiMasterSetCmdReg(/* bSetSTA */  E_AHI_SI_START_BIT,
                                /* bSetSTO */  E_AHI_SI_NO_STOP_BIT,
                                /* bSetRD */   FALSE,
                                /* bSetWR */   TRUE,
                                /* bSetAckCtrl */ FALSE,
                                /* bSetIACK */ FALSE))
    {
#ifdef DEBUG_I2C_DRIVER
        DBG_vPrintf(TRUE, "i2c_BusWriteReg: Exit 1\n");
#endif
        return FALSE;
    }
    while(bAHI_SiMasterPollTransferInProgress());

    // Now write Command to I2C device
    vAHI_SiMasterWriteData8(u8Command);

    // Now write Data to I2C device
    while (u8Length)
    {
        if (!bAHI_SiMasterSetCmdReg(/* bSetSTA */  E_AHI_SI_NO_START_BIT,
                                    /* bSetSTO */  E_AHI_SI_NO_STOP_BIT,
                                    /* bSetRD */   FALSE,
                                    /* bSetWR */   TRUE,
                                    /* bSetAckCtrl */ FALSE,
                                    /* bSetIACK */ FALSE))
        {
#ifdef DEBUG_I2C_DRIVER
            DBG_vPrintf(TRUE, "i2c_BusWriteReg: Exit 2\n");
#endif
            return FALSE;
        }
        while(bAHI_SiMasterPollTransferInProgress());

        vAHI_SiMasterWriteData8(*pu8Data);
        u8Length -= 1;
        pu8Data++;
    }
    if (!bAHI_SiMasterSetCmdReg(/* bSetSTA */  E_AHI_SI_NO_START_BIT,
                                /* bSetSTO */  E_AHI_SI_STOP_BIT,
                                /* bSetRD */   FALSE,
                                /* bSetWR */   TRUE,
                                /* bSetAckCtrl */ FALSE,
                                /* bSetIACK */ FALSE))
    {
#ifdef DEBUG_I2C_DRIVER
        DBG_vPrintf(TRUE, "i2c_BusWriteReg: Exit 3\n");
#endif
        return FALSE;
    }
    while(bAHI_SiMasterPollTransferInProgress());

#ifdef DEBUG_I2C_DRIVER
    DBG_vPrintf(bTraceI2c, "i2c_BusWriteReg: Leave\n");
#endif
    return TRUE;
}

/* Doxygen info, do not remove! */
/**
 * @brief   This function reads data from a I2C slave.
 * @param   u8Address - 7 bit I2C slave address
 * @param   u8Command - I2C slave command (this can indicate a register)
 * @param   u8Length - Length of the data
 * @param   pu8Data - Pointer to uint8 array to receive the read data
 * @return  TRUE when data has been read successfully.
 */

bool i2c_BusReadData(uint8 u8Address, uint8 u8Command, uint8 u8Length, uint8* pu8Data)
{
#ifdef DEBUG_I2C_DRIVER
    DBG_vPrintf(bTraceI2c, "i2c_BusReadData: Enter\n");
#endif

    // First write I2C address + WRITE BIT on I2C bus
    vAHI_SiMasterWriteSlaveAddr(u8Address, /* bReadStatus */ FALSE);
    if (!bAHI_SiMasterSetCmdReg(/* bSetSTA */  E_AHI_SI_START_BIT,
                                /* bSetSTO */  E_AHI_SI_NO_STOP_BIT,
                                /* bSetRD */   FALSE,
                                /* bSetWR */   TRUE,
                                /* bSetAckCtrl */ FALSE,
                                /* bSetIACK */ FALSE))
    {
#ifdef DEBUG_I2C_DRIVER
        DBG_vPrintf(TRUE, "i2c_BusReadData: Exit 1\n");
#endif
        return FALSE;
    }
    while(bAHI_SiMasterPollTransferInProgress());

    // Now write Command to I2C device
    vAHI_SiMasterWriteData8(u8Command);
    if (!bAHI_SiMasterSetCmdReg(/* bSetSTA */  E_AHI_SI_NO_START_BIT,
                                /* bSetSTO */  E_AHI_SI_STOP_BIT,
                                /* bSetRD */   FALSE,
                                /* bSetWR */   TRUE,
                                /* bSetAckCtrl */ FALSE,
                                /* bSetIACK */ FALSE))
    {
#ifdef DEBUG_I2C_DRIVER
        DBG_vPrintf(TRUE, "i2c_BusReadData: Exit 2\n");
#endif
        return FALSE;
    }
    while(bAHI_SiMasterPollTransferInProgress());

    // Again write I2C address + READ BIT on I2C bus
    vAHI_SiMasterWriteSlaveAddr(u8Address, /* bReadStatus */ TRUE);
    if (!bAHI_SiMasterSetCmdReg(/* bSetSTA */  E_AHI_SI_START_BIT,
                                /* bSetSTO */  E_AHI_SI_NO_STOP_BIT,
                                /* bSetRD */   FALSE,
                                /* bSetWR */   TRUE,
                                /* bSetAckCtrl */ FALSE,
                                /* bSetIACK */ FALSE))
    {
#ifdef DEBUG_I2C_DRIVER
        DBG_vPrintf(TRUE, "i2c_BusReadData: Exit 3\n");
#endif
        return FALSE;
    }
    while(bAHI_SiMasterPollTransferInProgress());

    // Now read data from I2C device
    while (u8Length > 1)
    {
        if (!bAHI_SiMasterSetCmdReg(/* bSetSTA */  E_AHI_SI_NO_START_BIT,
                                    /* bSetSTO */  E_AHI_SI_NO_STOP_BIT,
                                    /* bSetRD */   TRUE,
                                    /* bSetWR */   FALSE,
                                    /* bSetAckCtrl */ E_AHI_SI_SEND_ACK,
                                    /* bSetIACK */ FALSE))
        {
#ifdef DEBUG_I2C_DRIVER
            DBG_vPrintf(TRUE, "i2c_BusReadData: Exit 4\n");
#endif
            return FALSE;
        }
        while(bAHI_SiMasterPollTransferInProgress());
        *pu8Data++ = u8AHI_SiMasterReadData8();
        u8Length -= 1;
    }
    if (!bAHI_SiMasterSetCmdReg(/* bSetSTA */  E_AHI_SI_NO_START_BIT,
                                /* bSetSTO */  E_AHI_SI_STOP_BIT,
                                /* bSetRD */   TRUE,
                                /* bSetWR */   FALSE,
                                /* bSetAckCtrl */ E_AHI_SI_SEND_NACK,
                                /* bSetIACK */ FALSE))
    {
#ifdef DEBUG_I2C_DRIVER
        DBG_vPrintf(TRUE, "i2c_BusReadData: Exit 5\n");
#endif
        return FALSE;
    }
    while(bAHI_SiMasterPollTransferInProgress());
    *pu8Data = u8AHI_SiMasterReadData8();

#ifdef DEBUG_I2C_DRIVER
    DBG_vPrintf(bTraceI2c, "i2c_BusReadData: Leave\n");
#endif
    return TRUE;
}

/* Doxygen info, do not remove! */
/**
 * @brief   This function reads out a specified NTAG register.
 * @param   u8Address - 7 bit I2C slave address
 * @param   u8Command - I2C slave command
 * @param   u8Reg - NTAG register address
 * @param   pu8Data - Pointer to uint8 which will be assigned the read register value
 * @return  TRUE when data has been read successfully.
 */

bool i2c_ReadNtagReg(uint8 u8Address, uint8 u8Command, uint8 u8Reg, uint8* pu8Data)
{
#ifdef DEBUG_I2C_DRIVER
    DBG_vPrintf(bTraceI2c, "i2c_ReadNtagReg: Enter\n");
#endif

    // First write I2C address + WRITE BIT on I2C bus
    vAHI_SiMasterWriteSlaveAddr(u8Address, /* bReadStatus */ FALSE);
    if (!bAHI_SiMasterSetCmdReg(/* bSetSTA */  E_AHI_SI_START_BIT,
                                /* bSetSTO */  E_AHI_SI_NO_STOP_BIT,
                                /* bSetRD */   FALSE,
                                /* bSetWR */   TRUE,
                                /* bSetAckCtrl */ FALSE,
                                /* bSetIACK */ FALSE))
    {
#ifdef DEBUG_I2C_DRIVER
        DBG_vPrintf(TRUE, "i2c_ReadNtagReg: Exit 1\n");
#endif
        return FALSE;
    }
    while(bAHI_SiMasterPollTransferInProgress());

    // Now write Command to NTAG I2C device
    vAHI_SiMasterWriteData8(u8Command);
    if (!bAHI_SiMasterSetCmdReg(/* bSetSTA */  E_AHI_SI_NO_START_BIT,
                                /* bSetSTO */  E_AHI_SI_NO_STOP_BIT,
                                /* bSetRD */   FALSE,
                                /* bSetWR */   TRUE,
                                /* bSetAckCtrl */ FALSE,
                                /* bSetIACK */ FALSE))
    {
#ifdef DEBUG_I2C_DRIVER
        DBG_vPrintf(TRUE, "i2c_ReadNtagReg: Exit 2\n");
#endif
        return FALSE;
    }
    while(bAHI_SiMasterPollTransferInProgress());

    // Now write Register to NTAG I2C device
    vAHI_SiMasterWriteData8(u8Reg);
    if (!bAHI_SiMasterSetCmdReg(/* bSetSTA */  E_AHI_SI_NO_START_BIT,
                                /* bSetSTO */  E_AHI_SI_STOP_BIT,
                                /* bSetRD */   FALSE,
                                /* bSetWR */   TRUE,
                                /* bSetAckCtrl */ FALSE,
                                /* bSetIACK */ FALSE))
    {
#ifdef DEBUG_I2C_DRIVER
        DBG_vPrintf(TRUE, "i2c_ReadNtagReg: Exit 3\n");
#endif
        return FALSE;
    }
    while(bAHI_SiMasterPollTransferInProgress());

    // Again write I2C address + READ BIT on I2C bus
    vAHI_SiMasterWriteSlaveAddr(u8Address, /* bReadStatus */ TRUE);
    if (!bAHI_SiMasterSetCmdReg(/* bSetSTA */  E_AHI_SI_START_BIT,
                                /* bSetSTO */  E_AHI_SI_NO_STOP_BIT,
                                /* bSetRD */   FALSE,
                                /* bSetWR */   TRUE,
                                /* bSetAckCtrl */ FALSE,
                                /* bSetIACK */ FALSE))
    {
#ifdef DEBUG_I2C_DRIVER
        DBG_vPrintf(TRUE, "i2c_ReadNtagReg: Exit 4\n");
#endif
        return FALSE;
    }
    while(bAHI_SiMasterPollTransferInProgress());

    // Now read Register data from NTAG I2C device
    if (!bAHI_SiMasterSetCmdReg(/* bSetSTA */  E_AHI_SI_NO_START_BIT,
                                /* bSetSTO */  E_AHI_SI_STOP_BIT,
                                /* bSetRD */   TRUE,
                                /* bSetWR */   FALSE,
                                /* bSetAckCtrl */ E_AHI_SI_SEND_NACK,
                                /* bSetIACK */ FALSE))
    {
#ifdef DEBUG_I2C_DRIVER
        DBG_vPrintf(TRUE, "i2c_ReadNtagReg: Exit 5\n");
#endif
        return FALSE;
    }
    while(bAHI_SiMasterPollTransferInProgress());
    *pu8Data = u8AHI_SiMasterReadData8();

#ifdef DEBUG_I2C_DRIVER
    DBG_vPrintf(bTraceI2c, "i2c_ReadNtagReg: Leave\n");
#endif
    return TRUE;
}

/* Doxygen info, do not remove! */
/**
 * @brief   This function checks if a I2C slave with the given slave
 *          address is communicating on the I2C bus.
 * @param   u8Address - 7 bit I2C slave address
 * @return  TRUE when slave answers (=available), FALSE when no slave answers.
 */

bool i2c_CheckSlaveAvailable(uint8 u8Address)
{
#ifdef DEBUG_I2C_DRIVER
    DBG_vPrintf(bTraceI2c, "i2c_CheckSlaveAvailable: Enter\n");
#endif

    // First write I2C address + WRITE BIT on I2C bus
    vAHI_SiMasterWriteSlaveAddr(u8Address, /* bReadStatus */ FALSE);
    if (!bAHI_SiMasterSetCmdReg(/* bSetSTA */  E_AHI_SI_START_BIT,
                                /* bSetSTO */  E_AHI_SI_NO_STOP_BIT,
                                /* bSetRD */   FALSE,
                                /* bSetWR */   TRUE,
                                /* bSetAckCtrl */ FALSE,
                                /* bSetIACK */ FALSE))
    {
#ifdef DEBUG_I2C_DRIVER
        DBG_vPrintf(TRUE, "i2c_CheckSlaveAvailable: Exit 1\n");
#endif
        return FALSE;
    }
    while(bAHI_SiMasterPollTransferInProgress());

#ifdef DEBUG_I2C_DRIVER
    DBG_vPrintf(bTraceI2c, "i2c_CheckSlaveAvailable: Leave\n");
#endif
    /* Inverse the Nack check represents availability */
    return (!bAHI_SiMasterCheckRxNack());
}

/* Doxygen info, do not remove! */
/**
 * @brief   This function toggles the debug output if DEBUG_I2C_DRIVER is defined.
 * @return  None.
 */

void i2c_ToggleDBG(void)
{
#ifdef DEBUG_I2C_DRIVER
    bTraceI2c = !bTraceI2c;
#endif
}

#endif  // I2C_SUPPORT

/****************************************************************************/
/***        END OF FILE                                                   ***/
/****************************************************************************/
