/*****************************************************************************
 *
 * MODULE:             JN-AN-1200-ZigBee-RF4CE-Application-Template
 *
 * COMPONENT:          Remote.c
 *
 * DESCRIPTION:        Remote Implementation
 *
 * $HeadURL $
 *
 * $Revision: 12054 $
 *
 * $LastChangedBy: nxp57621 $
 *
 * $LastChangedDate: 2013-12-06 12:03:00 +0530 (Fri, 06 Dec 2013) $
 *
 * $Id: Remote.c 12054 2013-12-06 06:33:00Z nxp57621 $
 *
 ****************************************************************************
 *
 * This software is owned by NXP B.V. and/or its supplier and is protected
 * under applicable copyright laws. All rights are reserved. We grant You,
 * and any third parties, a license to use this software solely and
 * exclusively on NXP products [NXP Microcontrollers such as JN516x, JN514x,
 * JN513x].
 * You, and any third parties must reproduce the copyright and warranty notice
 * and any other legend of ownership on each copy or partial copy of the
 * software.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * Copyright NXP B.V. 2014. All rights reserved
 *
 ***************************************************************************/

/****************************************************************************/
/***        Include files                                                 ***/
/****************************************************************************/
#include <jendefs.h>
#include <PeripheralRegs.h>
#include <string.h>
#include "RF4CE_API.h"
#include "Config.h"
#include "Printf.h"
#include "app_buttons.h"
#include "LcdDriver.h"


/****************************************************************************/
/***        Macro Definitions                                             ***/
/****************************************************************************/


/****************************************************************************/
/***        Type Definitions                                              ***/
/****************************************************************************/
typedef enum
{
    E_COLD_START,
    E_WARM_START,
    E_WAIT_FOR_EV_START_CFM,
    E_WAIT_FOR_KEY,
    E_START_DISCOVERY,
    E_WAIT_FOR_EV_DISC_DONE,
    E_START_PAIRING,
    E_WAIT_FOR_EV_PAIR_CFM,
    E_PAIRED,
    E_PRE_RUNNING,
    E_RUNNING,
    E_TRANSMITTING,
    E_TRANSMIT_FAILED,
    E_UNPAIRED
} teState;

typedef struct
{
    tsIeeeAddr sIeeeAddr;
    uint16     u16PanId;
    uint8      u8Channel;
} tsDiscovered;


/****************************************************************************/
/***        Local Function Prototypes                                     ***/
/****************************************************************************/
PRIVATE void vInitSystem(void);
PRIVATE void vIdleLoop(void);
PRIVATE uint8 u8ButtonReadWithSWDebounce();
PRIVATE void vSendCommand(uint8 u8KeyPressed);
PRIVATE void vSleep(void);
PRIVATE void vTimerISR(uint32 u32Device, uint32 u32ItemBitmap);
PRIVATE void vDelayIn100ms(uint8 u8DelayIn100ms);
PRIVATE bool_t bGetNextActivePairingEntry(uint8 *u8CurrentPairingEntry);
PRIVATE uint8 u8NumOfActivePairingEntries(void);
PRIVATE void vValToDec(char *pcOutString, uint8 u8Value, char *pcLabel);


/****************************************************************************/
/***        Exported Variables                                            ***/
/****************************************************************************/
extern uint8 gMAC_u8MaxBuffers;
extern uint8 u8WakeUpButton;

/****************************************************************************/
/***        Local Variables                                               ***/
/****************************************************************************/
PRIVATE volatile teState eState;
PRIVATE tuRF4CE_NibValue uNibValue;
PRIVATE uint8 u8CurrentPairingEntry = 0xFF;
PRIVATE volatile bool_t bUnPaired = FALSE;
PRIVATE volatile uint32 u32SleepCounter = 0;
PRIVATE volatile uint32 u32DelayCounter = 0;
PRIVATE tsDiscovered sDiscovered;

tsRF4CE_AppCap sOrgAppCap = {USER_STRING_SPECIFIED,
                             NUM_OF_SUPPORTED_DEVICE_TYPES,
                             RESERVED,
                             NUM_OF_SUPPORTED_PROFILES,
                             RESERVED};
PRIVATE uint8 au8OrgDevTypeList[NUM_OF_SUPPORTED_DEVICE_TYPES] = \
                                        {RF4CE_DEVICE_TYPE_REMOTE_CONTROL};
PRIVATE uint8 au8OrgProfileIdList[NUM_OF_SUPPORTED_PROFILES] = {ZRC_PROFILE_ID};
PRIVATE uint8 au8OrgDiscProfileIdList[NUM_OF_DISCOVERABLE_PROFILES] = \
                                                            {ZRC_PROFILE_ID};


/****************************************************************************/
/***        Exported Functions                                            ***/
/****************************************************************************/


/****************************************************************************/
/***        Local Functions                                               ***/
/****************************************************************************/

/****************************************************************************
 *
 * NAME: AppColdStart
 *
 * DESCRIPTION:
 * Entry point for application from boot loader. Initializes system and runs
 * main loop.
 *
 * RETURNS:
 * Never returns.
 *
 ****************************************************************************/
PUBLIC void AppColdStart(void)
{
    gMAC_u8MaxBuffers = 2;

    vAHI_WatchdogStop();

    /* General initialisation */
    vInitSystem();

    /* Initialise stack (implementation-specific command) */
    if (bRF4CE_ImpInit(NODE_CAPABILITY, VENDOR_ID,(uint8 *)VENDOR_STRING))
    {
        vPrintf("\nColdStart");
        eState = E_COLD_START;
    }
    else
    {
        vPrintf("\nWarmStart");
        eState = E_WARM_START;
    }

    vAHI_TickTimerIntEnable(TRUE);
    /* Go to idle loop */
    vIdleLoop();
}

/****************************************************************************
 *
 * NAME: AppWarmStart
 *
 * DESCRIPTION:
 * Entry point for a wake from sleep mode with the memory contents held.
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PUBLIC void AppWarmStart(void)
{
    AppColdStart();
}

/****************************************************************************
 *
 * NAME: vInitSystem
 *
 * DESCRIPTION:
 * Initialises stack and hardware.
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vInitSystem(void)
{
    /* Initialise stack and hardware interfaces. */
    (void)u32AHI_Init();

    /* Set clock to CPU_FREQUENCY_MHz and OptimiseWaitState */
    bAHI_SetClockRate((CPU_FREQUENCY_MHz == 32)? 3 : 2);
    vAHI_OptimiseWaitStates();

    while(bAHI_Clock32MHzStable() == FALSE);

    /* Initialise tick timer to give periodic interrupt */
    vAHI_TickTimerConfigure(E_AHI_TICK_TIMER_DISABLE);
    vAHI_TickTimerWrite(0);
    vAHI_TickTimerRegisterCallback(vTimerISR);
    vAHI_TickTimerInterval(16 * 100000); /* 100ms tick from 16MHz Clock source */
    vAHI_TickTimerConfigure(E_AHI_TICK_TIMER_RESTART);

    /* Initialise UART for debug */
    vUART_printInit();

    /* Initialise buttons; if DIO8 pressed, destroy settings */
    if(bAppButtonsInitialise())
    {
        vPrintf("\nDestroy settings");
        vRF4CE_ImpDestroySettings();
    }

    /* Initialise LED */
    vAHI_DioSetPullup((LED_DIO3|LED_DIO6),0); // vAHI_DioSetPullup(on,off);
    vAHI_DioSetDirection(0,(LED_DIO3|LED_DIO6)); // vAHI_DioSetDirection(in,out);
    LED3_LOW;
    LED6_LOW;

    /* Initialise, Clear and Refresh the display */
    vLcdReset(0,12);
    vLcdClear();
    vLcdRefreshAll();
}

/****************************************************************************
 *
 * NAME: vIdleLoop
 *
 * DESCRIPTION: Application state machine
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vIdleLoop(void)
{
    char acString[8];
    uint8 u8Key;
    bool_t bPerformAction = FALSE;
    volatile uint32 u32i;

    while (1)
    {
        switch (eState)
        {
            case E_COLD_START:
                /* Cold start: reset stack and clear NIB; then start stack */
                vRF4CE_NlmeResetReq(TRUE);
                /* Stack start is asynchronous; wait for start confirmation.*/
                eState = E_WAIT_FOR_EV_START_CFM;
                vRF4CE_NlmeStartReq();
                break;

            case E_WAIT_FOR_EV_START_CFM:
                break;

            case E_WARM_START:
                /* Warm start: reset without clearing NIB */
                vRF4CE_NlmeResetReq(FALSE);

                if(bGetNextActivePairingEntry(&u8CurrentPairingEntry))
                {
                    /* Start as pairing table is non empty */
                    eState = E_PRE_RUNNING;
                }
                else
                {
                    /* Cold Start again as pairing table is empty */
                    eState = E_COLD_START;
                }
                break;

            case E_WAIT_FOR_KEY:
                /* Read Key press */
                vLcdClear();
                vLcdWriteText("Pair                      ", 7, 0);
                vLcdRefreshAll();

                /* Detect the Button 1 key press */
                if(APP_BUTTONS_SW1 == u8AppButtonsRead())
                {
                    /* Wait until key is released */
                    while(APP_BUTTONS_SW1 == u8AppButtonsRead());

                    eState = E_START_DISCOVERY;
                }
                break;

            case E_START_DISCOVERY:
                u32SleepCounter = 0;

                /* Set info in NIB */
                memset(uNibValue.au8UserString, 0, RF4CE_USER_STRING_LEN);
                memcpy(uNibValue.au8UserString, USER_STRING, USER_STRING_LEN);
                (void)eRF4CE_NlmeSetReq(E_RF4CE_NIB_ATTR_USER_STRING, 0, &uNibValue);

                uNibValue.u8MaxDiscoveryRepetitions = MAX_DISCOVERY_REPETITIONS;
                (void)eRF4CE_NlmeSetReq(E_RF4CE_NIB_ATTR_MAX_DISC_REPETITIONS,
                                        0, &uNibValue);

                /* Get info from NIB */
                (void)eRF4CE_NlmeGetReq(E_RF4CE_NIB_ATTR_BASE_CHANNEL, 0, &uNibValue);
                vPrintf("\nStarted with Channel %d",uNibValue.u8BaseChannel);

                /* Request discovery */
                vPrintf("\nTrying discovery");

                vLcdClear();
                vLcdWriteText("    Discovery and Pairing       ", 2, 0);
                vLcdWriteText("           Please Wait          ", 3, 0);
                vLcdRefreshAll();

                eState = E_WAIT_FOR_EV_DISC_DONE;

                vRF4CE_NlmeDiscoveryReq(0xffff, 0xffff,
                                        &sOrgAppCap, au8OrgDevTypeList,
                                        au8OrgProfileIdList,
                                        RF4CE_DEVICE_TYPE_MEDIA_CENTER,
                                        NUM_OF_DISCOVERABLE_PROFILES,
                                        au8OrgDiscProfileIdList,
                                        MAX_DISCOVERY_DURATION);
                break;

            case E_WAIT_FOR_EV_DISC_DONE:
                /* Waiting for discovery to complete */
                break;

            case E_START_PAIRING:
                vPrintf("\nPairing");
                vRF4CE_NlmePairReq(sDiscovered.u8Channel, sDiscovered.u16PanId,
                                   &sDiscovered.sIeeeAddr, &sOrgAppCap,
                                   au8OrgDevTypeList, au8OrgProfileIdList,
                                   KEY_EXCHANGE_TRANSFER_COUNT);
                eState = E_WAIT_FOR_EV_PAIR_CFM;
                break;

            case E_WAIT_FOR_EV_PAIR_CFM:
                /* Waiting for pairing to complete */
                break;

            case E_PAIRED:
                /* On a successful pairing, blinking LED3 */
                LED3_HIGH;

                /* As paired save the settings and get into running mode */
                vRF4CE_ImpSaveSettings(E_SAVE_MODE_FULL);
                vPrintf("\nSaved");

                /* Setting WakeUpButton to be zero before going to running state */
                u8WakeUpButton = ERR;

                eState = E_PRE_RUNNING;
                break;

            case E_PRE_RUNNING:
                vLcdClear();
                vValToDec(acString, u8CurrentPairingEntry, "  ");
                acString[2]='\0';
                vLcdWriteText("                 PairRef ", 0, 1);
                vLcdWriteText(acString, 0, 110);
                vLcdWriteText("Pair     List      0       1", 7, 0);
                vLcdRefreshAll();

                /* Keep D6 on as there is at least an active pairing entry present */
                LED3_LOW;
                LED6_HIGH;
                eState = E_RUNNING;
                break;

            case E_RUNNING:
                if(u8WakeUpButton != ERR)
                {
                    /* Upon wake up from sleep, perform appropriate action */
                    u8Key = u8WakeUpButton;
                    u8WakeUpButton = ERR;
                    bPerformAction = TRUE;
                }
                else
                {
                    /* Read button when not sleeping */
                    u8Key = u8ButtonReadWithSWDebounce();
                    if(u8Key != ERR)
                    {
                        bPerformAction = TRUE;
                    }
                }

                if(bPerformAction)
                {
                    bPerformAction = FALSE;
                    switch(u8Key)
                    {
                        case APP_BUTTONS_SW1:
                            /* Wait until key is released */
                            while(APP_BUTTONS_SW1 == u8AppButtonsRead());

                            if(u8NumOfActivePairingEntries() >= RF4CE_NWKC_MAX_PAIRING_TABLE_ENTRIES)
                            {
                                vLcdClear();
                                vLcdWriteText("        No Capacity      ", 1, 0);
                                vLcdRefreshAll();
                                for(u32i=0;u32i<3000000;u32i++);
                                eState = E_PRE_RUNNING;
                            }
                            else
                            {
                                eState = E_START_DISCOVERY;
                            }
                            break;

                        case APP_BUTTONS_SW2:
                            /* Wait until key is released */
                            while(APP_BUTTONS_SW2 == u8AppButtonsRead());

                            vSendCommand(DISPLAY_INFO);
                            break;

                        case APP_BUTTONS_SW3:
                            vSendCommand(ZERO_DIGIT);
                            break;

                        case APP_BUTTONS_SW4:
                            vSendCommand(ONE_DIGIT);
                            break;

                        case APP_BUTTONS_DIO8:
                            /* Wait until key is released */
                            while(APP_BUTTONS_DIO8 == u8AppButtonsRead());

                            bGetNextActivePairingEntry(&u8CurrentPairingEntry);
                            eState = E_PRE_RUNNING;
                            break;
                        default:
                            break;
                    }
                }

                if(bUnPaired)
                {
                    if(u8NumOfActivePairingEntries())
                    {
                        u8CurrentPairingEntry = 0xFF;
                        bGetNextActivePairingEntry(&u8CurrentPairingEntry);
                        eState = E_PRE_RUNNING;
                        bUnPaired = FALSE;
                    }
                    else
                    {
                        /* Software reset if pairing table is empty */
                        vPrintf("\nDestroy settings");
                        vRF4CE_ImpDestroySettings();
                        vAHI_SwReset();
                    }
                }
                break;

            case E_TRANSMITTING:
                /* Waiting for transmit to complete */
                break;

            case E_TRANSMIT_FAILED:
                vLcdClear();
                vLcdWriteText("  No Resp from Receiver ", 2, 0);
                vLcdWriteText("     Please check    ", 3, 0);
                vLcdWriteText("   software is running.   ", 4, 0);
                vLcdRefreshAll();
                for(u32i=0;u32i<3000000;u32i++);
                eState = E_PRE_RUNNING;
                break;

            default:
                break;
        }
    }
}

/****************************************************************************
 *
 * NAME: vRF4CE_StackEvent
 *
 * DESCRIPTION:
 * Handle the generated stack event
 *
 * PARAMETERS:          Name            RW  Usage
 *                      eEvent          R   RF4CE Stack event type
 *                      puParam         R   Pointer to parameter of RF4CE event
 *
 * RETURNS:
 * None.
 *
 ****************************************************************************/
PUBLIC void vRF4CE_StackEvent(teRF4CE_EventType eEvent,
                              tuRF4CE_EventParam *puParam)
{
    switch (eEvent)
    {
    case E_RF4CE_EV_START_CFM:      /* Use with tsRF4CE_NlmeStartCfm */
        if (E_RF4CE_STATUS_SUCCESS == puParam->sNlmeStartCfm.eStatus)
        {
            eState = E_WAIT_FOR_KEY;
        }
        else
        {
            /* Try scanning again till successful start confirm comes*/
            vRF4CE_NlmeStartReq();
        }
        break;

    case E_RF4CE_EV_DISC_CFM:   /* Use with tsRF4CE_NlmeDiscoveryCfm */
        if (E_RF4CE_STATUS_SUCCESS == puParam->sNlmeDiscoveryCfm.eStatus)
        {
            tsRF4CE_NodeDesc *psNodeDesc = puParam->sNlmeDiscoveryCfm.psNodeDescList;

            /* Store details of first match for pairing with later */
            sDiscovered.u8Channel      = psNodeDesc->u8LogicalChannel;
            sDiscovered.u16PanId       = psNodeDesc->u16PanId;
            sDiscovered.sIeeeAddr.u32H = psNodeDesc->sIeeeAddr.u32H;
            sDiscovered.sIeeeAddr.u32L = psNodeDesc->sIeeeAddr.u32L;
            eState = E_START_PAIRING;
        }
        else
        {
            vPrintf("\nDisc Cfm! Error code %x",puParam->sNlmeDiscoveryCfm.eStatus);

            if(u8NumOfActivePairingEntries())
            {
                eState = E_PRE_RUNNING;
            }
            else
            {
                eState = E_WAIT_FOR_KEY;
            }
        }
        break;

    case E_RF4CE_EV_PAIR_CFM:       /* Use with tsRF4CE_NlmePairCfm */
        vPrintf("\tPair Cfm:%x:%x:%x:%x",puParam->sNlmePairCfm.eStatus,
                            puParam->sNlmePairCfm.u8PairingRef,
                            puParam->sNlmePairCfm.u16RecVendorId);

        if (E_RF4CE_STATUS_SUCCESS == puParam->sNlmePairCfm.eStatus)
        {
            vPrintf("\nPaired");
            u8CurrentPairingEntry = puParam->sNlmePairCfm.u8PairingRef;
            eState = E_PAIRED;
        }
        else
        {
            vPrintf("\nNot Paired");
            if(u8NumOfActivePairingEntries())
            {
                eState = E_PRE_RUNNING;
            }
            else
            {
                eState = E_WAIT_FOR_KEY;
            }
        }
        break;

    case E_RF4CE_EV_COMMSTATUS_IND: /* Use with tsRF4CE_NlmeCommStatusInd */
        break;

    case E_RF4CE_EV_NLDE_CFM:       /* Use with tsRF4CE_NldeDataCfm */
        if(puParam->sNldeDataCfm.eStatus == E_RF4CE_STATUS_SUCCESS)
        {
            eState = E_RUNNING;
        }
        else
        {
            eState = E_TRANSMIT_FAILED;
        }
        break;

    case E_RF4CE_EV_UNPAIR_IND:
        vRF4CE_NlmeUnpairResp(puParam->sNlmeUnpairInd.u8PairingRef);
        vRF4CE_ImpSaveSettings(E_SAVE_MODE_FULL);
        bUnPaired = TRUE;
        break;

    case E_RF4CE_EV_NLDE_IND:
        vPrintf("\nData Ind");
        break;

    case E_RF4CE_EV_UNPAIR_CFM:     /* Use with tsRF4CE_NlmeUnpairCfm */
        vPrintf("\nUnpair Cfm");
        break;

    case E_RF4CE_EV_PAIR_IND:       /* Use with tsRF4CE_NlmePairInd */
    case E_RF4CE_EV_DISC_IND:       /* Use with tsRF4CE_NlmeDiscoveryInd */
    case E_RF4CE_EV_AUTODISC_CFM:   /* Use with tsRF4CE_NlmeDiscoveryCfm */
    default:
        vPrintf("\nUnhandled!");
        break;
    }
}



/****************************************************************************
 *
 * NAME: u8ButtonReadWithSWDebounce
 *
 * DESCRIPTION:
 * Reads button DIO pressed with software debounce of 100ms.
 *
 * RETURNS:
 * uint8 Returns the DIO button number of pressed key.
 *
 ****************************************************************************/
PRIVATE uint8 u8ButtonReadWithSWDebounce()
{
    uint8 u8Key;

    u8Key = u8AppButtonsRead();
    if(ERR != u8Key)
    {
        vDelayIn100ms(1);
        if(u8Key == u8AppButtonsRead())
        {
            u32SleepCounter = 0;
            return u8Key;
        }
    }
    return ERR;
}

/****************************************************************************
 *
 * NAME: vSendCommand
 *
 * DESCRIPTION:
 * Send key pressed data
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vSendCommand(uint8 u8KeyPressed)
{
    uint8 au8Nsdu[10];
    uint8 u8DataLen;
    uint8 u8TxOptions;

    /* Create CERC 'key pressed' command frame */
    au8Nsdu[0] = E_ZRC_USER_CONTROL_PRESSED;
    au8Nsdu[1] = u8KeyPressed;
    u8DataLen = CERC_CONTROL_PRESSED_CMD_LENGTH;
    vPrintf("\nData %x",au8Nsdu[1]);
    eState = E_TRANSMITTING;

    u8TxOptions = RF4CE_TX_OPT_ACKNOWLEDGE|RF4CE_TX_OPT_SECURITY;

    /* Spinlock to make sure that transition to 32MHz happened. */
    while(bAHI_Clock32MHzStable() == FALSE);

    vRF4CE_NldeDataReq(u8CurrentPairingEntry/*au8ReceiverPairingRef[u8CurrentTarget]*/,
                        ZRC_PROFILE_ID, VENDOR_ID,
                       u8DataLen, au8Nsdu, u8TxOptions);
}

/****************************************************************************
 *
 * NAME: vSleep
 *
 * DESCRIPTION:
 * Sends the device into E_AHI_SLEEP_OSCOFF_RAMON sleep mode.
 *
 * RETURNS:
 * None
 *
 * NOTES:
 * This function does not return at all; Program restarts at appColdStart of
 * appWarmStart.
 ****************************************************************************/
PRIVATE void vSleep()
{
    volatile uint16 u16i;
    uint32 u32DioStatus;

    vPrintf("\nSleep");
    /* Turn off the LEDs */
    LED3_LOW;
    LED6_LOW;

    /* clear any pending wake timers before sleeping */
    (void)u8AHI_WakeTimerFiredStatus();
    /* Read wake status of IO lines to clear bits and prevent multiple interrupts */
    u32DioStatus = u32AHI_DioInterruptStatus();

    vAHI_DioWakeEnable(0, APP_BUTTONS_DIO8_MASK);  // WakeEnable(enable, disable)

    vRF4CE_ImpSaveSettings(E_SAVE_MODE_MINIMAL);
    for (u16i=0;u16i<1000;u16i++);
    /* Sleep selected */
    vAHI_Sleep(E_AHI_SLEEP_OSCOFF_RAMON);
}

/****************************************************************************
 *
 * NAME: vDelayIn100ms
 *
 * DESCRIPTION:
 * Delay in multiple of 100ms. This function utilises tick timer.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  u8DelayIn100ms  R   Delay in multiple of 100ms
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
void vDelayIn100ms(uint8 u8DelayIn100ms)
{
    vAHI_TickTimerWrite(0);
    u32DelayCounter = 0;
    while(u32DelayCounter < u8DelayIn100ms);
}

/****************************************************************************
 *
 * NAME: vTimerISR
 *
 * DESCRIPTION:
 * Tick timer interrupt service routine.
 * Configured to occur at every 100ms in vInitSystem() funciton.
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vTimerISR(uint32 u32Device, uint32 u32ItemBitmap)
{
    u32SleepCounter++;
    u32DelayCounter++;

    if(u32SleepCounter >= (TIME_BEFORE_SLEEP_IN_100MS))
    {
        u32SleepCounter = 0;
        vLcdPowerOff();
        vSleep();
    }
}

/****************************************************************************
 *
 * NAME: bGetNextActivePairingEntry
 *
 * DESCRIPTION:
 * Finds the active pair entry. If passed argument points to the value which is
 * larger than RF4CE_NWKC_MAX_PAIRING_TABLE_ENTRIES, search begins from first
 * pairing entry.
 *
 * RETURNS:
 * bool_t   Returns TRUE if any active pairing entry found; FALSE otherwise.
 *
 ****************************************************************************/
PRIVATE bool_t bGetNextActivePairingEntry(uint8 *u8CurrentPairingEntry)
{
    uint8 u8i,u8StartIndex;

    if((*u8CurrentPairingEntry) >= RF4CE_NWKC_MAX_PAIRING_TABLE_ENTRIES)
    {
        u8StartIndex = 0x00;
    }
    else
    {
        u8StartIndex = (((*u8CurrentPairingEntry)+1)%RF4CE_NWKC_MAX_PAIRING_TABLE_ENTRIES);
    }

    for(u8i=u8StartIndex; u8i<(RF4CE_NWKC_MAX_PAIRING_TABLE_ENTRIES+u8StartIndex);u8i++)
    {
        (void)eRF4CE_NlmeGetReq(E_RF4CE_NIB_ATTR_PAIRING_TABLE,
                                (u8i%RF4CE_NWKC_MAX_PAIRING_TABLE_ENTRIES),
                                &uNibValue);
        if(uNibValue.sPairingTableEntry.eState  == E_PAIR_ACTIVE )
        {
            /* Active pairing entry found */
            (*u8CurrentPairingEntry) = (u8i%RF4CE_NWKC_MAX_PAIRING_TABLE_ENTRIES);
            return TRUE;
        }
    }

    /* No active pairing entry found */
    *u8CurrentPairingEntry = 0xFF;
    return FALSE;
}

/****************************************************************************
 *
 * NAME: u8NumOfActivePairingEntries
 *
 * DESCRIPTION:
 * Finds number of active pairing entries
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE uint8 u8NumOfActivePairingEntries()
{
    volatile uint8 u8i=0, u8Count=0;
    for(u8i=0; u8i<RF4CE_NWKC_MAX_PAIRING_TABLE_ENTRIES;u8i++)
    {
        (void)eRF4CE_NlmeGetReq(E_RF4CE_NIB_ATTR_PAIRING_TABLE,
                                u8i, &uNibValue);
        if(uNibValue.sPairingTableEntry.eState  == E_PAIR_ACTIVE )
        {
            /* Active pairing entry found */
            u8Count++;
        }
    }
    return u8Count;
}


/****************************************************************************
 *
 * NAME: vValToDec
 *
 * DESCRIPTION:
 * Converts an 8-bit value to a string of the textual decimal representation.
 * Adds a text string after the text.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  pcOutString     R   Location for new string
 *                  u8Value         R   Value to convert
 *                  pcLabel         R   Label to append to string
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vValToDec(char *pcOutString, uint8 u8Value, char *pcLabel)
{
    static const uint8 au8Digits[3] = {100, 10, 1};
    uint8 u8Digit;
    uint8 u8DigitIndex;
    uint8 u8Count;
    bool_t boPreviousDigitPrinted = FALSE;

    for (u8DigitIndex = 0; u8DigitIndex < 3; u8DigitIndex++)
    {
        u8Count = 0;
        u8Digit = au8Digits[u8DigitIndex];
        while (u8Value >= u8Digit)
        {
            u8Value -= u8Digit;
            u8Count++;
        }

        if ((u8Count != 0) || (boPreviousDigitPrinted == TRUE)
            || (u8DigitIndex == 2))
        {
            *pcOutString = '0' + u8Count;
            boPreviousDigitPrinted = TRUE;
            pcOutString++;
        }
    }

    strcpy(pcOutString, pcLabel);
}

/****************************************************************************/
/***        END OF FILE                                                   ***/
/****************************************************************************/
