/****************************************************************************
 *
 * MODULE:              JN-AN-1180 Demo coordinator code
 *
 * DESCRIPTION:
 * Coordinator for demonstrator. Acts as time coordinator for up to four
 * sensor endpoints and manages LCD panel and keys on central coordinator.
 * This version is specific to the JN5168 variant on the DK4 platform.
 *
****************************************************************************
*
* This software is owned by NXP B.V. and/or its supplier and is protected
* under applicable copyright laws. All rights are reserved. We grant You,
* and any third parties, a license to use this software solely and
* exclusively on NXP products [NXP Microcontrollers such as JN5148, JN5142, JN5139].
* You, and any third parties must reproduce the copyright and warranty notice
* and any other legend of ownership on each copy or partial copy of the
* software.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
* ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
* LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
* SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
* INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
* CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
* ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
* POSSIBILITY OF SUCH DAMAGE.

* Copyright NXP B.V. 2014. All rights reserved
*/

/****************************************************************************/

/****************************************************************************/
/***        Include files                                                 ***/
/****************************************************************************/
#include "jendefs.h"
#include "LcdDriver.h"
#include "HomeSensorConfig.h"
#include <AppHardwareApi.h>
#include "NXPLogo.h"
#include <AppQueueApi.h>
#include <mac_sap.h>
#include <mac_pib.h>
#include <string.h>
#include <Utilities.h>

/****************************************************************************/
/***        Macro Definitions                                             ***/
/****************************************************************************/
/* Block (time slice) values */
#define BLOCK_TIME_IN_32K_PERIODS 1600
#define BLOCK_MIN_RX          2
#define BLOCK_UPDATE          (BLOCK_MIN_RX + DEMO_ENDPOINTS)
#define MAX_BLOCKS            20

/* Control screen and alarm values */
#define CONTROL_LIST_LEN      4
#define TEMP_HIGH_MAX         100
#define LIGHT_HIGH_MAX        6

/* Setup screen values */
#define SETUP_LIST_LEN        2

#define FRAMES_MISSED_INDICATION 9

#ifdef USE_2006_SECURITY
#define FRAME_TYPE_DATA         (1)

#define TOTAL_SEC_ENTRIES       (4)
#define TOTAL_SECURITY_KEYS     (4)
#define TOTAL_KEY_USAGE_ENTRIES (1)
#define TOTAL_LEVEL_DESCRIPTORS (1)
#endif

#if (defined JENNIC_CHIP_FAMILY_JN514x) || (defined JENNIC_CHIP_FAMILY_JN516x)
#undef RESTRICTED_BUTTONS
#else
#define RESTRICTED_BUTTONS
#endif

/****************************************************************************/
/***        Type Definitions                                              ***/
/****************************************************************************/
/* Holds all stored data for a particular sensor for a node */
typedef struct
{
    uint8 u8NowValue;
    uint8 u8HighAlarm;
    uint8 u8LowAlarm;
    uint8 au8GraphData[DEMO_HISTORY_LEN];
} tsNodeElementData;

/* Holds all stored data for a node */
typedef struct
{
    bool_t boDeviceOn;
    tsNodeElementData asNodeElementData[DEMO_SENSOR_LIST_LEN];
    uint8 u8PrevRxCount;
    uint8 u8FramesMissed;
    uint8 u8SwitchOn;
    uint8 u8Rssi;
} tsNodeData;

/* Used to track an association between extended address and short address */
typedef struct
{
    MAC_ExtAddr_s sExtAddr;
    uint16 u16ShortAddr;
} tsAssocNodes;

/* System states with respect to screen display being shown */
typedef enum
{
    E_STATE_NETWORK,
    E_STATE_NODE,
    E_STATE_NODE_CONTROL,
    E_STATE_SET_CHANNEL,
    E_STATE_SETUP_SCREEN,
    E_STATE_SCANNING
} teState;

/* Button values */
typedef enum
{
    E_KEY_0 = LCD_BOARD_BUTTON_S1_VAL,
    E_KEY_1 = LCD_BOARD_BUTTON_S2_VAL,
    E_KEY_2 = LCD_BOARD_BUTTON_S3_VAL,
    E_KEY_3 = LCD_BOARD_BUTTON_S4_VAL,
    E_KEYS_0_AND_3 = (LCD_BOARD_BUTTON_S1_VAL | LCD_BOARD_BUTTON_S4_VAL)
} teKeyValues;


/* All application data with scope within the entire file is kept here,
   including all stored node data, GUI settings and current state */
typedef struct
{
    struct
    {
        tsNodeData   asNodeData[DEMO_ENDPOINTS];
        tsAssocNodes asAssocNodes[DEMO_ENDPOINTS];
        bool_t       bLocalNode;
        uint8        u8AssociatedNodes;
    } sNode;

    struct
    {
        teSensor eCurrentSensor;
        uint8    u8CurrentNode;
        uint8    u8GraphPos;
        uint8    u8ControlSelection;
        uint8    u8SetupSelection;
        bool_t   bShowFourNodes;
    } sGui;

    struct
    {
        teState eState;
        uint8   u8Channel;
        uint32  u32AppApiVersion;
        uint32  u32HwApiVersion;
        uint32  u32CalibratedTimeout;
    } sSystem;
} tsDemoData;

/****************************************************************************/
/***        Local Function Prototypes                                     ***/
/****************************************************************************/

/* initialisation routines */
PRIVATE void vInitSystem(void);
PRIVATE void vInitCoord(void);

/* radio functions */
PRIVATE void vStartBeacon(void);
PRIVATE void vUpdateBeaconPayload(void);

/* timing and interrupts */
PRIVATE void vSetTimer(void);
PRIVATE void vProcessCurrentTimeBlock(void);
PRIVATE uint8 u8UpdateTimeBlock(uint8 u8TimeBlock);
PRIVATE void vProcessInterrupts(void);
PRIVATE void vProcessUpdateBlock(void);

/* data processing */
PRIVATE void vProcessIncomingData(MAC_McpsDcfmInd_s *psMcpsInd);
PRIVATE void vProcessIncomingMlme(MAC_MlmeDcfmInd_s *psMlmeInd);

/* wake function */
PRIVATE bool_t bProcessForTimeout(AppQApiHwInd_s *psHardwareInd);

/* button processing */
PRIVATE bool_t bProcessKeys(uint8 *pu8Keys);

/* main 'channel setup' screen */
PRIVATE void vProcessSetChannelKeyPress(uint8 u8KeyMap);
PRIVATE void vBuildSetChannelScreen(void);
PRIVATE void vUpdateSetChannelScreen(void);

/* 'miscellaneous settings' screen */
PRIVATE void vProcessSetupKeyPress(uint8 u8KeyMap);
PRIVATE void vBuildSetupScreen(void);
PRIVATE void vUpdateSetupScreen(uint8 u8Selection, bool_t boUpdate);

/* 'network' screen */
PRIVATE void vProcessNetworkKeyPress(uint8 u8KeyMap);
PRIVATE void vBuildNetworkScreen(teSensor eSensor);
PRIVATE void vUpdateNetworkScreen(teSensor eSensor);
PRIVATE void vUpdateNetworkSensor(teSensor eSensor);

/* 'network node' screen */
PRIVATE void vProcessNodeKeyPress(uint8 u8KeyMap);
PRIVATE void vBuildNodeScreen(uint8 u8Node);
PRIVATE void vUpdateNodeScreen(uint8 u8Node);

/* 'network control' screen */
PRIVATE void vProcessNodeControlKeyPress(uint8 u8KeyMap);
PRIVATE void vBuildNodeControlScreen(uint8 u8Node);
PRIVATE void vUpdateNodeControlScreen(uint8 u8Node, uint8 u8Selection, bool_t boUpdate);

/* screen drawing */
PRIVATE void vLcdUpdateElement(tsNodeData *psNodeData, teSensor eSensor,
                               uint8 u8Row, uint8 u8Col, bool_t bShowRssi);
PRIVATE void vDrawGraph(uint8 *pu8GraphData, uint8 u8StartCol,
                        uint8 u8StartRow);

/* helper functions */
PRIVATE void vStringCopy(char *pcFrom,char *pcTo);
PRIVATE void vValToDec(char *pcOutString, uint8 u8Value, char *pcLabel);

PRIVATE void vAdjustAlarm(uint8 *pu8Value, uint8 u8MaxValue, uint8 u8OffValue, bool_t bUpNotDown);

PRIVATE void vDisplayError(char *pcErrorMessage, uint32 u32Data);
PRIVATE void vWriteOnOff(bool_t bOnOff, uint8 u8Row, uint8 u8Col);
PRIVATE void vToggleOnOff(bool_t *pbItem);
PRIVATE void vWriteRowLabel(uint8 u8Selection, char **ppcRowName, uint8 u8ListLen);

/****************************************************************************/
/***        Exported Variables                                            ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Variables                                               ***/
/****************************************************************************/

/* Handles from the MAC */
PRIVATE void *s_pvMac;
PRIVATE MAC_Pib_s *s_psMacPib;

/* File scope data */
PRIVATE tsDemoData sDemoData;

/* Row and column positions of info fields on LCD */
static const uint8 au8NodeLcdRow[DEMO_ENDPOINTS] = {0,  0, 3,  3};
static const uint8 au8NodeLcdCol[DEMO_ENDPOINTS] = {0, 64, 0, 64};

static const char *apcNodeNameList[DEMO_ENDPOINTS] = {
    "Hall", "Bedroom", "Lounge", "Bathroom"};

#ifdef USE_2006_SECURITY
PRIVATE MAC_KeyDescriptor_s           asKeyDescriptor[TOTAL_SECURITY_KEYS];
PRIVATE MAC_KeyIdLookupDescriptor_s   asKeyIdLookupDescriptor[TOTAL_SECURITY_KEYS];
PRIVATE MAC_KeyUsageDescriptor_s      asKeyUsageNwk[TOTAL_KEY_USAGE_ENTRIES];
PRIVATE MAC_DeviceDescriptor_s        asDeviceDescriptor[TOTAL_SEC_ENTRIES];
PRIVATE MAC_KeyDeviceDescriptor_s     asKeyDeviceList[TOTAL_SEC_ENTRIES];
PRIVATE MAC_SecurityLevelDescriptor_s asSecurityLevelDescriptor[TOTAL_LEVEL_DESCRIPTORS];

PRIVATE const uint8 au8DefaultKeySource[8] = {0x12, 0x34, 0x56, 0x78,
                                              0x9a, 0xbc, 0xde, 0xf0};
#endif

/****************************************************************************/
/***        Exported Functions                                            ***/
/****************************************************************************/


/****************************************************************************
 *
 * NAME: AppColdStart
 *
 * DESCRIPTION:
 * Entry point for application from boot loader. Initialises system and runs
 * main loop.
 *
 * RETURNS:
 * Never returns.
 *
 ****************************************************************************/
PUBLIC void AppColdStart(void)
{
    uint8 u8Keys;
    uint8 u8TimeBlock;
    bool_t bExitMainLoop;


    /* Disable watchdog if enabled by default */
    #ifdef WATCHDOG_ENABLED
    vAHI_WatchdogStop();
    #endif

    /* Wait for clock */
    while (bAHI_GetClkSource() == TRUE);
    vAHI_OptimiseWaitStates();

    /* General initialisation: reset hardware, set some values in PIB */
    vInitSystem();

    /* Main application loop. This loop is iterated whenever a soft reset is
       performed */
    while (1)
    {
        /* Initialise software elements */
        vInitCoord();

        /* Set simple scheduler to perform no activity */
        u8TimeBlock = MAX_BLOCKS;

        /* Build channel set screen */
        vBuildSetChannelScreen();

        /* Change to channel setting state */
        sDemoData.sSystem.eState = E_STATE_SET_CHANNEL;

        /* Get current key state. This is necessary after a soft reset as keys
           will still be down when entering the main loop */
        u8Keys = u8LcdButtonRead();

        /* Run main loop. The timer ensures that this runs 20 times per second,
           which allows a simple scheduler to run to start reads from the
           sensors, read results from the sensors and update the display. It
           also allows for simple button debounce by only sampling the buttons
           once during each interval. */
        do
        {
            /* Set the timer for the next interval */
            vSetTimer();

            /* Perform scheduler action */
            vProcessCurrentTimeBlock();

            /* Check keys. Returns TRUE if 'reset' combination has been pressed */
            bExitMainLoop = bProcessKeys(&u8Keys);

            /* Increment scheduler time block for next time */
            u8TimeBlock = u8UpdateTimeBlock(u8TimeBlock);

            /* Service upward indications and confirmations from MLME and
               MCPS, and process interrupts from hardware. This function
               will exit when timer fires */
            vProcessInterrupts();


        } while (bExitMainLoop == FALSE);
    }
}

/****************************************************************************
 *
 * NAME: AppWarmStart
 *
 * DESCRIPTION:
 * Entry point for application from boot loader. Simply jumps to AppColdStart
 * as, in this instance, application will never warm start.
 *
 * RETURNS:
 * Never returns.
 *
 ****************************************************************************/
PUBLIC void AppWarmStart(void)
{
    AppColdStart();
}

/****************************************************************************/
/***        Local Functions                                               ***/
/****************************************************************************/

/****************************************************************************
 *
 * NAME: vInitSystem
 *
 * DESCRIPTION:
 * Initialises stack and hardware. Also sets non-default values in the
 * 802.15.4 PIB and starts the first read of the light sensor. Subsequent
 * reads of this sensor occur automatically.
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vInitSystem(void)
{
#ifdef USE_2006_SECURITY
    uint8 u8Key;
    uint8 u8Device;
#endif

    /* Initialise stack and hardware interfaces, and register peripheral
       interrupts with AppQueueApi handler. We aren't using callbacks
       at all, just monitoring the upward queues in a loop */
    sDemoData.sSystem.u32HwApiVersion = u32AHI_Init();
    sDemoData.sSystem.u32AppApiVersion = u32AppQApiInit(NULL, NULL, NULL);

    /* Set up the MAC handles. Must be called AFTER u32AppQApiInit() */
    s_pvMac = pvAppApiGetMacHandle();
    s_psMacPib = MAC_psPibGetHandle(s_pvMac);

    vLcdButtonInit();

    /* Set up hardware and splash screen */
    vLcdResetDefault();

    /* Set Pan ID and short address in PIB (also sets match registers in hardware) */
    MAC_vPibSetPanId(s_pvMac, DEMO_PAN_ID);
    MAC_vPibSetShortAddr(s_pvMac, DEMO_COORD_ADDR);
    s_psMacPib->bAssociationPermit = TRUE;

    /* Calibrate wake timer */
    sDemoData.sSystem.u32CalibratedTimeout = BLOCK_TIME_IN_32K_PERIODS * 10000 / u32AHI_WakeTimerCalibrate();

    /* Enable timer to use for sequencing */
    vAHI_WakeTimerEnable(E_AHI_WAKE_TIMER_0, TRUE);

#ifdef USE_2006_SECURITY
    /* Enable security. Using Key ID mode 1 (default key source + 1 byte
       index) and security level 5 (encryption plus authentication with 4-byte
       MIC). Only require security for data frames. Using different key for
       each child, with a separate security entry for each.

       So:
       The Key Descriptor table has 4 entries (1 per key)
         - each entry points to a unique Key ID Lookup Descriptor (hence 4 in
           total)
         - each entry points to a unique Key Device Descriptor (hence 4 in
           total)
         - all entries point to the same Key Usage Descriptor (hence 1 in
           total)
       The Device Descriptor table has 4 entries (1 per device)
         - each entry is pointed to by exactly one Key Device Descriptor

       Also:
       The Security Level Descriptor table has 1 entry (1 per secured frame
       type)
     */

    /* Set up Security Level Descriptor table, which defines the minimum
       security level allowed for specific frame types. In this case we only
       care about data frames which must always use at least security level 5
     */
    asSecurityLevelDescriptor[0].u8FrameType = FRAME_TYPE_DATA;
    asSecurityLevelDescriptor[0].u8MinimumSecurity = 5;
    asSecurityLevelDescriptor[0].bOverideSecurityMinimum = TRUE;

    /* Key Usage Descriptor list */
    asKeyUsageNwk[0].u8FrameType = FRAME_TYPE_DATA;

    /* Key Descriptor table, Key ID Lookup table, Key Device Descriptor */
    for (u8Key = 0; u8Key < TOTAL_SECURITY_KEYS; u8Key++)
    {
        /* Set up Key ID Lookup table. All key lookups are configured as the
          default key source plus a byte for a unique index; 9 bytes in total.
          The lookup data size is set as 1, which means 9(!) */
        memcpy(asKeyIdLookupDescriptor[u8Key].au8LookupData,
               au8DefaultKeySource,
               8);
        asKeyIdLookupDescriptor[u8Key].au8LookupData[8] = u8Key + 1; /* Key index */
        asKeyIdLookupDescriptor[u8Key].u8LookupDataSize = MAC_SECURITY_KEYID_LENGTH_MODE_1;

        /* Key Device Descriptor. Note that as u32DeviceDescriptorHandle is
           index into the asDeviceDescriptor array, the way we link the data
           means that TOTAL_SEC_ENTRIES and TOTAL_SECURITY_KEYS must be the
           same value to make any sense */
        asKeyDeviceList[u8Key].u32DeviceDescriptorHandle = u8Key;
        asKeyDeviceList[u8Key].bUniqueDevice = FALSE;
        asKeyDeviceList[u8Key].bBlacklisted = FALSE;

        /* Key descriptors: for each key, Device list (Key Device Descriptor)
           is a list of one, unique to the key. Same for the Key ID Lookup
           Descriptor. All keys share the same Key Usage Descriptor
           (list of frame types). Each key is unique, but in this example the
           value is based on the key number for simplicity */
        asKeyDescriptor[u8Key].psKeyDeviceList = &asKeyDeviceList[u8Key];
        asKeyDescriptor[u8Key].u8KeyDeviceListEntries = 1;
        asKeyDescriptor[u8Key].psKeyIdLookupDescriptor = &asKeyIdLookupDescriptor[u8Key];
        asKeyDescriptor[u8Key].u8KeyIdLookupEntries = 1;
        asKeyDescriptor[u8Key].psKeyUsageList = asKeyUsageNwk;
        asKeyDescriptor[u8Key].u8KeyUsageListEntries = TOTAL_KEY_USAGE_ENTRIES;
        asKeyDescriptor[u8Key].au32SymmetricKey[0] = 0x01020304;
        asKeyDescriptor[u8Key].au32SymmetricKey[1] = 0x05060708;
        asKeyDescriptor[u8Key].au32SymmetricKey[2] = 0x090a0b0c;
        asKeyDescriptor[u8Key].au32SymmetricKey[3] = 0x0d0e0f00 + u8Key;
    }

    /* evice Descriptor table */
    for (u8Device = 0; u8Device < TOTAL_SEC_ENTRIES; u8Device++)
    {
        /* Device Descriptor table is mostly concerned with addresses. These
           will be filled in properly when each device joins */
        asDeviceDescriptor[u8Device].u16PanId = 0;
        asDeviceDescriptor[u8Device].u16Short = 0;
        asDeviceDescriptor[u8Device].sExt.u32L = 0;
        asDeviceDescriptor[u8Device].sExt.u32H =  0;
        asDeviceDescriptor[u8Device].u32FrameCounter = 0;
        asDeviceDescriptor[u8Device].bExempt = FALSE;
    }

    /* Configure the MAC... enable security... */
    vAppApiSetSecurityMode(MAC_SECURITY_2006);
    s_psMacPib->bMacSecurityEnabled = TRUE;

    /* ... set the default lookup key (same as the value in the Key ID Lookup
       Descriptors) ... */
    memcpy(s_psMacPib->au8MacDefaultKeySource, au8DefaultKeySource, 8);

    /* ... set the Key Descriptor table... */
    s_psMacPib->psMacKeyTable = asKeyDescriptor;
    s_psMacPib->u8MacKeyTableEntries = TOTAL_SECURITY_KEYS;

    /* ... set the Security Level Descriptor table... */
    s_psMacPib->psMacSecurityLevelTable = asSecurityLevelDescriptor;
    s_psMacPib->u8MacSecuirtyLevelTableEntries = TOTAL_LEVEL_DESCRIPTORS;

    /* ... and set the Device Descriptor table */
    s_psMacPib->psMacDeviceTable = asDeviceDescriptor;
    s_psMacPib->u8MacDeviceTableEntries = TOTAL_SEC_ENTRIES;
#endif
}

/****************************************************************************
 *
 * NAME: vInitCoord
 *
 * DESCRIPTION:
 * Initialises software structures and variables. Endpoint data is reset and
 * the GUI is set to the default condition.
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vInitCoord(void)
{
    tsNodeData        *psNodeData;
    tsNodeElementData *psNodeElementData;
    uint8             *pu8GraphData;
    int                i, j, k;

    /* Initialise node data */
    for (i = 0; i < DEMO_ENDPOINTS; i++)
    {
        psNodeData = &sDemoData.sNode.asNodeData[i];
        psNodeData->boDeviceOn = FALSE;

        /* Set label */
        for (j = 0; j < DEMO_SENSOR_LIST_LEN; j++)
        {
            psNodeElementData = &psNodeData->asNodeElementData[j];

            /* Clear alarms and values */
            psNodeElementData->u8NowValue = 0;
            psNodeElementData->u8HighAlarm = 0;
            psNodeElementData->u8LowAlarm = 255;

            /* Clear history list */
            pu8GraphData = psNodeElementData->au8GraphData;
            for (k = 0; k < DEMO_HISTORY_LEN; k++)
            {
                *pu8GraphData = 0;
                pu8GraphData++;
            }
        }
    }

    sDemoData.sNode.bLocalNode = FALSE;     /* For DK4 platform,local node NOT in use is the default */
    sDemoData.sNode.u8AssociatedNodes = 0;  /* As local node is NOT in use, 0 nodes are 'associated' */

    /* Initialise GUI state */
    sDemoData.sGui.eCurrentSensor = E_SENSOR_TEMP;
    sDemoData.sGui.u8CurrentNode = 0;
    sDemoData.sGui.u8GraphPos = 0;
    sDemoData.sGui.u8ControlSelection = 0;

    sDemoData.sGui.u8SetupSelection = 0;
    sDemoData.sGui.bShowFourNodes = TRUE;

    sDemoData.sSystem.u8Channel = CHANNEL_MID;
}


/****************************************************************************
 *
 * NAME: bProcessKeys
 *
 * DESCRIPTION:
 * Gets the latest button presses and detects any change since the last time
 * the buttons were checked. If there is a change it is passed to the
 * individual handler for the screen currently being displayed (the buttons
 * are all 'soft' keys so their meaning changes from screen to screen). The
 * exception to this is a button combination that causes the software to
 * shutdown and stop the LCD. There is also a reset combination.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  pu8Keys         RW  Persistent value of buttons pressed
 *
 * RETURNS:
 * TRUE if reset combination is pressed
 *
 ****************************************************************************/
PRIVATE bool_t bProcessKeys(uint8 *pu8Keys)
{
    uint8 u8KeysDown;
    uint8 u8NewKeysDown;

    u8KeysDown = *pu8Keys;

    /* Process key press */
    u8NewKeysDown = u8LcdButtonRead();

#ifdef RESTRICTED_BUTTONS
    u8NewKeysDown &= (LCD_BOARD_BUTTON_S2_VAL | LCD_BOARD_BUTTON_S4_VAL);
#endif

    if (u8NewKeysDown != 0)
    {
        if ((u8NewKeysDown | u8KeysDown) != u8KeysDown)
        {
            /* Logical OR values to enable multiple keys at once */
            u8KeysDown |= u8NewKeysDown;

            /* Key presses depend on mode */
            switch (sDemoData.sSystem.eState)
            {
            case E_STATE_NETWORK:
                vProcessNetworkKeyPress(u8KeysDown);
                break;

            case E_STATE_NODE:
                vProcessNodeKeyPress(u8KeysDown);
                break;

            case E_STATE_NODE_CONTROL:
                vProcessNodeControlKeyPress(u8KeysDown);
                break;

            case E_STATE_SET_CHANNEL:
                vProcessSetChannelKeyPress(u8KeysDown);
                break;

            case E_STATE_SETUP_SCREEN:
                vProcessSetupKeyPress(u8KeysDown);
                break;

            default:
                break;
            }
        }
    }
    else
    {
        u8KeysDown = 0;
    }

    /* Store value for use next time */
    *pu8Keys = u8KeysDown;

    return (u8KeysDown == E_KEYS_0_AND_3);
}

/****************************************************************************
 *
 * NAME: vSetTimer
 *
 * DESCRIPTION:
 * Sets wake-up timer 0 for a 50ms time-out. Assumes that timer was
 * previously enabled.
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vSetTimer(void)
{
    /* Set timer for next block */
    vAHI_WakeTimerStart(E_AHI_WAKE_TIMER_0, sDemoData.sSystem.u32CalibratedTimeout);
}

/****************************************************************************
 *
 * NAME: vProcessCurrentTimeBlock
 *
 * DESCRIPTION:
 * Operates a simple state machine. Called 20 times per second, this performs
 * several tasks over a 1 second period, with the time split into 50ms blocks.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  u8TimeBlock     R   Current time block, 0-19
 *
 * RETURNS:
 * void
 *
 * NOTES:
 * A value greater than 19 may be used for u8TimeBlock, to ensure that the
 * simple state machine remains idle.
 *
 ****************************************************************************/
PRIVATE void vProcessCurrentTimeBlock()
{

        /* Time to update the display */
        vProcessUpdateBlock();
        /* Also update the information passed back to the endpoints in the
           beacon payload */
        vUpdateBeaconPayload();

}

/****************************************************************************
 *
 * NAME: u8UpdateTimeBlock
 *
 * DESCRIPTION:
 * Moves the state machine time block on by one time period.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  u8TimeBlock     R   Previous time block
 *
 * RETURNS:
 * uint8 Next time block
 *
 ****************************************************************************/
PRIVATE uint8 u8UpdateTimeBlock(uint8 u8TimeBlock)
{
    /* Update block state for next time, if in a state where regular
       updates should be performed */
    if ((sDemoData.sSystem.eState != E_STATE_SET_CHANNEL)
        && (sDemoData.sSystem.eState != E_STATE_SETUP_SCREEN)
        && (sDemoData.sSystem.eState != E_STATE_SCANNING))
    {
        u8TimeBlock++;
        if (u8TimeBlock >= MAX_BLOCKS)
        {
            u8TimeBlock = 0;
        }
    }

    return u8TimeBlock;
}

/****************************************************************************
 *
 * NAME: vProcessInterrupts
 *
 * DESCRIPTION:
 * Loops around checking for upward events from the Application Queue API.
 * To save power, the CPU is set to doze at the start of the loop. If an event
 * is generated, the CPU is woken up and processes it, then the code here is
 * allowed to continue.
 * For each type of event, the queue is checked and, if there is something to
 * process, the appropriate handler is called. The buffer is then returned to
 * the Application Queue API. This continues until all events have been
 * processed. If the hardware event handler detects that a timeout has
 * happened, the loop is exited.
 *
 * RETURNS:
 * void when a timeout is detected
 *
 * NOTES:
 * There is a possibility of the system locking up if an interrupt is
 * generated and handled by the Application Queue API just before the request
 * to doze the CPU. Normally there won't be an issue as the wake-up timer will
 * generate an interrupt eventually, but it is theoretically possible that an
 * MLME or MCPS event just before the timer fires could create this situation
 * (has never been seen in practice). An easy solution is to use the second
 * wake-up timer as a watchdog.
 *
 ****************************************************************************/
PRIVATE void vProcessInterrupts(void)
{
    MAC_MlmeDcfmInd_s *psMlmeInd;
    MAC_McpsDcfmInd_s *psMcpsInd;
    AppQApiHwInd_s    *psAHI_Ind;
    bool_t             bDone;

    /* Wait for events */
    bDone = FALSE;
    do
    {


        /* Check for anything on the MCPS upward queue */
        do
        {
            psMcpsInd = psAppQApiReadMcpsInd();
            if (psMcpsInd != NULL)
            {
                vProcessIncomingData(psMcpsInd);
                vAppQApiReturnMcpsIndBuffer(psMcpsInd);
            }
        } while (psMcpsInd != NULL);

        /* Check for anything on the MLME upward queue */
        do
        {
            psMlmeInd = psAppQApiReadMlmeInd();
            if (psMlmeInd != NULL)
            {
                vProcessIncomingMlme(psMlmeInd);
                vAppQApiReturnMlmeIndBuffer(psMlmeInd);
            }
        } while (psMlmeInd != NULL);

        /* Check for anything on the AHI upward queue */
        do
        {
            psAHI_Ind = psAppQApiReadHwInd();
            if (psAHI_Ind != NULL)
            {
                if (bProcessForTimeout(psAHI_Ind) == TRUE)
                {
                    bDone = TRUE;
                }
                vAppQApiReturnHwIndBuffer(psAHI_Ind);
            }
        } while (psAHI_Ind != NULL);

    } while (bDone == FALSE);
}

/****************************************************************************
 *
 * NAME: vProcessIncomingData
 *
 * DESCRIPTION:
 * Deals with any incoming MCPS data events. If the event is an indication
 * from a device with a short address matching a demo endpoint, the data in
 * the payload is stored for display the next time that the LCD is updated.
 *
 * PARAMETERS: Name        RW  Usage
 *             psMcpsInd   R   Pointer to structure containing MCPS event
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vProcessIncomingData(MAC_McpsDcfmInd_s *psMcpsInd)
{
    MAC_RxFrameData_s *psFrame;
    MAC_Addr_s *psAddr;
    tsNodeData *psNodeData;
    uint16 u16NodeAddr;
    uint8 u8Node;
    uint8 u8Count;

    psFrame = &psMcpsInd->uParam.sIndData.sFrame;

    /* Check that MCPS frame is a valid sensor one */
    if ((psMcpsInd->u8Type != MAC_MCPS_IND_DATA)
        || (psFrame->u8SduLength != 8)
        || (psFrame->au8Sdu[0] != DEMO_ENDPOINT_IDENTIFIER))
    {
        return;
    }

    /* Use address to determine node */
    psAddr = &psFrame->sSrcAddr;
    if (psAddr->u8AddrMode != 2)
    {
        return;
    }

    u16NodeAddr = psAddr->uAddr.u16Short;
    if ((u16NodeAddr < DEMO_ENDPOINT_ADDR_BASE)
        || (u16NodeAddr >= (DEMO_ENDPOINT_ADDR_BASE + DEMO_ENDPOINTS)))
    {
        return;
    }

    /* Store data for node */
    u8Node = (uint8)(u16NodeAddr - DEMO_ENDPOINT_ADDR_BASE);
    psNodeData = &sDemoData.sNode.asNodeData[u8Node];

    psNodeData->u8FramesMissed = 0;
    psNodeData->u8SwitchOn = psFrame->au8Sdu[2];

    for (u8Count = 0; u8Count < DEMO_SENSOR_LIST_LEN; u8Count++)
    {
        psNodeData->asNodeElementData[u8Count].u8NowValue = psFrame->au8Sdu[u8Count + 3];
    }

    psNodeData->u8Rssi = psFrame->u8LinkQuality;
}

/****************************************************************************
 *
 * NAME: vProcessIncomingMlme
 *
 * DESCRIPTION:
 * Deals with any incoming MCPS events. If the event is an indication that
 * device wants to associate, goes through a process to determine a short
 * address for the device, either as 'next on the list' or the same address
 * that was proviously given if the device has associated before. Then sends
 * an association response via the stack.
 *
 * For debug purposes, also displays any Communication Status indications.
 *
 * PARAMETERS: Name        RW  Usage
 *             psMlmeInd   R   Pointer to structure containing MLME event
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vProcessIncomingMlme(MAC_MlmeDcfmInd_s *psMlmeInd)
{
    MAC_MlmeReqRsp_s        sMlmeReqRsp;
    MAC_MlmeSyncCfm_s       sMlmeSyncCfm;
    uint16                  u16ShortAddress;
    uint8                   u8Node;
    uint8                   u8AssocStatus;
#ifdef USE_2006_SECURITY
    MAC_DeviceDescriptor_s *psDeviceDescriptor;
#endif

    switch (psMlmeInd->u8Type)
    {
    case MAC_MLME_IND_ASSOCIATE:
        /* Only respond if in network or node modes. We aren't a coordinator
           until then */
        if ((sDemoData.sSystem.eState == E_STATE_NETWORK)
            || (sDemoData.sSystem.eState == E_STATE_NODE)
            || (sDemoData.sSystem.eState == E_STATE_NODE_CONTROL))
        {
            /* Default short address */
            u16ShortAddress = 0xffff;

            /* Check node extended address matches and device wants short address
               (for the demo, we only use short address) */
            if (psMlmeInd->uParam.sIndAssociate.u8Capability & 0x80)
            {
                /* Check if already associated (idiot proofing) */
                u8Node = 0;
                while (u8Node < sDemoData.sNode.u8AssociatedNodes)
                {
                    if ((psMlmeInd->uParam.sIndAssociate.sDeviceAddr.u32L == sDemoData.sNode.asAssocNodes[u8Node].sExtAddr.u32L) &&
                        (psMlmeInd->uParam.sIndAssociate.sDeviceAddr.u32H == sDemoData.sNode.asAssocNodes[u8Node].sExtAddr.u32H))
                    {
                        /* Already in system: give it same short address */
                        u16ShortAddress = sDemoData.sNode.asAssocNodes[u8Node].u16ShortAddr;
                    }
                    u8Node++;
                }

                /* Assume association succeeded */
                u8AssocStatus = 0;

                if (u16ShortAddress == 0xffff)
                {
                    if (sDemoData.sNode.u8AssociatedNodes < DEMO_ENDPOINTS)
                    {
                        /* Allocate short address as next in list */
                        u16ShortAddress = DEMO_ENDPOINT_ADDR_BASE + sDemoData.sNode.u8AssociatedNodes;

#ifdef USE_2006_SECURITY
                        /* Configure security for this node, and reset frame counter */
                        psDeviceDescriptor = &asDeviceDescriptor[sDemoData.sNode.u8AssociatedNodes];

                        psDeviceDescriptor->u16PanId = DEMO_PAN_ID;
                        psDeviceDescriptor->u16Short = u16ShortAddress;
                        psDeviceDescriptor->sExt.u32L = psMlmeInd->uParam.sIndAssociate.sDeviceAddr.u32L;
                        psDeviceDescriptor->sExt.u32H =  psMlmeInd->uParam.sIndAssociate.sDeviceAddr.u32H;
                        psDeviceDescriptor->u32FrameCounter = 0;
#endif

                        /* Store details for future use */
                        sDemoData.sNode.asAssocNodes[sDemoData.sNode.u8AssociatedNodes].sExtAddr = psMlmeInd->uParam.sIndAssociate.sDeviceAddr;
                        sDemoData.sNode.asAssocNodes[sDemoData.sNode.u8AssociatedNodes].u16ShortAddr = u16ShortAddress;
                        sDemoData.sNode.u8AssociatedNodes++;
                    }
                    else
                    {
                        /* PAN access denied */
                        u8AssocStatus = 2;
                    }
                }

                /* Update display if necessary */
                if (sDemoData.sSystem.eState == E_STATE_NETWORK)
                {
                    vBuildNetworkScreen(sDemoData.sGui.eCurrentSensor);
                }
            }
            else
            {
                /* PAN access denied */
                u8AssocStatus = 2;
            }

            /* Create association response */
            sMlmeReqRsp.u8Type = MAC_MLME_RSP_ASSOCIATE;
            sMlmeReqRsp.u8ParamLength = sizeof(MAC_MlmeRspAssociate_s);
            sMlmeReqRsp.uParam.sRspAssociate.sDeviceAddr = psMlmeInd->uParam.sIndAssociate.sDeviceAddr;
            sMlmeReqRsp.uParam.sRspAssociate.u16AssocShortAddr = u16ShortAddress;
            sMlmeReqRsp.uParam.sRspAssociate.u8Status = u8AssocStatus;
            sMlmeReqRsp.uParam.sRspAssociate.u8SecurityEnable = FALSE;
            sMlmeReqRsp.uParam.sRspAssociate.sSecurityData.u8SecurityLevel = 0;

            /* Send association response */
            vAppApiMlmeRequest(&sMlmeReqRsp, &sMlmeSyncCfm);

            /* There is no confirmation for an association response, hence no need to check */
        }
        break;
    }
}

/****************************************************************************
 *
 * NAME: bProcessForTimeout
 *
 * DESCRIPTION:
 * Processes any incoming hardware events to see if the event is wake-up
 * timer 0 firing, as this is the only hardware event responded to.
 *
 * PARAMETERS: Name          RW Usage
 *             psHardwareInd R  Pointer to structure containing hardware event
 *
 * RETURNS:
 * TRUE if event was wake-up timer 0 firing.
 *
 ****************************************************************************/
PRIVATE bool_t bProcessForTimeout(AppQApiHwInd_s *psHardwareInd)
{
    /* Check to see if hardware indication was for wake timer 0 */
    if ((psHardwareInd->u32DeviceId == E_AHI_DEVICE_SYSCTRL)
        && (psHardwareInd->u32ItemBitmap & (1 << E_AHI_SYSCTRL_WK0)))
    {
        return TRUE;
    }

    return FALSE;
}

/****************************************************************************
 *
 * NAME: vProcessUpdateBlock
 *
 * DESCRIPTION:
 * Called once per second to update the scrolling graphs and, if showing a
 * screen with graphs on, updating the LCD.
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vProcessUpdateBlock(void)
{
    tsNodeData *psNodeData;
    tsNodeElementData *psNodeElementData;
    uint8 *pu8GraphData;
    uint8 u8PrevPoint;
    uint8 u8Node;
    uint8 u8Sensor;
    uint8 u8Value;

    /* Update graphs */
    for (u8Node = 0; u8Node < DEMO_ENDPOINTS; u8Node++)
    {
        psNodeData = &sDemoData.sNode.asNodeData[u8Node];
        if (psNodeData->u8FramesMissed)
        {
            /* Missed data, so copy previous value forward */
            u8PrevPoint = (sDemoData.sGui.u8GraphPos - 1) & (DEMO_HISTORY_LEN - 1);
            for (u8Sensor = 0; u8Sensor < DEMO_SENSOR_LIST_LEN; u8Sensor++)
            {
                pu8GraphData = psNodeData->asNodeElementData[u8Sensor].au8GraphData;
                pu8GraphData[sDemoData.sGui.u8GraphPos] = pu8GraphData[u8PrevPoint];
            }
        }
        else
        {
            /* Data must be scaled for graph (0-13)
               Temp range is 0-52
               Humidity range is 0-104
               Light range is 0-6
            */
            for (u8Sensor = 0; u8Sensor < DEMO_SENSOR_LIST_LEN; u8Sensor++)
            {
                psNodeElementData = &psNodeData->asNodeElementData[u8Sensor];
                u8Value = psNodeElementData->u8NowValue;
                switch (u8Sensor)
                {
                case E_SENSOR_TEMP:
                    u8Value = u8Value >> 2;
                    break;

                case E_SENSOR_HTS:
                    u8Value = u8Value >> 3;
                    break;

                case E_SENSOR_ALS:
                    u8Value = u8Value * 2;
                    break;
                }
                if (u8Value > 13)
                {
                    u8Value = 13;
                }
                psNodeElementData->au8GraphData[sDemoData.sGui.u8GraphPos] = u8Value;
            }
        }

        /* For next time, assume failed until proven otherwise */
        if (psNodeData->u8FramesMissed < FRAMES_MISSED_INDICATION)
        {
            psNodeData->u8FramesMissed++;
        }
    }

    /* Increment graph position */
    sDemoData.sGui.u8GraphPos = (sDemoData.sGui.u8GraphPos + 1) & (DEMO_HISTORY_LEN - 1);

    /* Update display */
    switch (sDemoData.sSystem.eState)
    {
    case E_STATE_NETWORK:
        vUpdateNetworkScreen(sDemoData.sGui.eCurrentSensor);
        break;

    case E_STATE_NODE:
        vUpdateNodeScreen(sDemoData.sGui.u8CurrentNode);
        break;

    default:
        break;
    }
}

/****************************************************************************
*  Menu System Notes:
*
* There are five screens, they are:
* 1. Main (initial) screen with channel option ..referred to here as 'SetChannel'
* 2. Subscreen from channel menu, (miscellaneous settings) .. 'Setup'
* 3. Subscreen from done menu.. 'Network'
* 4. Subscreen from Network.. 'Node Display'
* 5. Subscreen from Node.. 'Node Control'

* All of the screens follow the convention of having three interrelated
* functions ~ Process(Screen Name).. Build(Screen Name).. Update(Screen Name)
****************************************************************************/

/****************************************************************************
 *
 * NAME: vProcessSetChannelKeyPress
 *
 * DESCRIPTION:
 * Handles button presses on the Set Channel screen. There is one parameter
 * that can be adjusted (the channel) and buttons to navigate to two other
 * screens.
 *
 * PARAMETERS:      Name        RW  Usage
 *                  u8KeyMap    R   Current buttons pressed bitmap
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vProcessSetChannelKeyPress(uint8 u8KeyMap)
{
    switch (u8KeyMap)
    {
    case E_KEY_0:
        /* Further setup button: go to setup screen */
        sDemoData.sSystem.eState = E_STATE_SETUP_SCREEN;
        vBuildSetupScreen();
        break;

    case E_KEY_1:
        /* Plus button: increment value */
    case E_KEY_2:
        /* Minus button: decrement value */

        vAdjustAlarm(&sDemoData.sSystem.u8Channel, CHANNEL_MAX, CHANNEL_MIN, u8KeyMap == E_KEY_1);
        vUpdateSetChannelScreen();
        break;

    case E_KEY_3:
        /* Done button: start beaconing and go to network screen */
        vStartBeacon();
        sDemoData.sSystem.eState = E_STATE_NETWORK;
        vBuildNetworkScreen(sDemoData.sGui.eCurrentSensor);
        break;

    default:
        break;
    }
}

/****************************************************************************
 *
 * NAME: vBuildSetChannelScreen
 *
 * DESCRIPTION:
 * Creates the Set Channel screen, consisting of a bitmap of the NXP logo
 * and labels for the soft buttons on the bottom row. Uses the related update
 * function to display the current channel and refresh the LCD.
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vBuildSetChannelScreen(void)
{
    vLcdClear();

    vLcdWriteBitmap((tsBitmap *)&sNXPLogo, 0, 1);

    /* Soft keys */
    vLcdWriteText("Ch", 7, 0);
    vLcdWriteText("\\", 7, 47);
    vLcdWriteText("]", 7, 74);
    vLcdWriteText("Done", 7, 103);

    /* Update to display the data */
    vUpdateSetChannelScreen();
}

/****************************************************************************
 *
 * NAME: vUpdateSetChannelScreen
 *
 * DESCRIPTION:
 * Updates the Set Channel screen, when it first appears or when the user
 * changes the channel number.
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vUpdateSetChannelScreen(void)
{
    char acString[5];

    vValToDec(acString, sDemoData.sSystem.u8Channel, "  ");
    vLcdWriteText(acString, 7, 16);

    vLcdRefreshAll();
}


/****************************************************************************
 *
 * NAME: vProcessSetupKeyPress
 *
 * DESCRIPTION:
 * Handles button presses on the Setup screen. The first button
 * selects which item to alter, the next two adjust the value up or down, and
 * the last button puts the device into running mode, starting the beacons
 * and moving to the Network screen.
 *
 * PARAMETERS:      Name        RW  Usage
 *                  u8KeyMap    R   Current buttons pressed bitmap
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vProcessSetupKeyPress(uint8 u8KeyMap)
{
    switch (u8KeyMap)
    {
    case E_KEY_0:
        /* Select button: move to next item in list */
        vAdjustAlarm(&sDemoData.sGui.u8SetupSelection, SETUP_LIST_LEN - 1, 0, TRUE);
        vUpdateSetupScreen(sDemoData.sGui.u8SetupSelection, TRUE);
        break;

    case E_KEY_1:
        /* Plus button: increment value */
    case E_KEY_2:
        /* Minus button: decrement value */

        switch (sDemoData.sGui.u8SetupSelection)
        {
        case 0:
            /* Local node */
            vToggleOnOff(&sDemoData.sNode.bLocalNode);
            break;

        case 1:
            /* Four node selection */
            vToggleOnOff(&sDemoData.sGui.bShowFourNodes);
            break;
        }

        vUpdateSetupScreen(sDemoData.sGui.u8SetupSelection, TRUE);
        break;

    case E_KEY_3:
        /* Done button: start beaconing and go to network screen. If
           local node is not being used, number of associated nodes is 0,
           as none can have associated yet, otherwise it is 1 as set during
           initialisation */
        if (sDemoData.sNode.bLocalNode == FALSE)
        {
            sDemoData.sNode.u8AssociatedNodes = 0;
        }

        vStartBeacon();
        sDemoData.sSystem.eState = E_STATE_NETWORK;
        vBuildNetworkScreen(sDemoData.sGui.eCurrentSensor);
        break;
    }
}

/****************************************************************************
 *
 * NAME: vBuildSetupScreen
 *
 * DESCRIPTION:
 * Builds the text for the Setup screen, then uses the update function
 * to show the values for each adjustable parameter in turn.
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vBuildSetupScreen(void)
{
    char acString[9];

    vLcdClear();
    vLcdWriteText("Settings", 0, 0);
    vLcdWriteText("Select", 7, 0);
    vLcdWriteText("\\", 7, 47);
    vLcdWriteText("]", 7, 74);
    vLcdWriteText("Done", 7, 103);

    /* Display version numbers */
    vLcdWriteText("MAC library", 4, 0);
    vUTIL_NumToString(sDemoData.sSystem.u32AppApiVersion, acString);
    vLcdWriteText(acString, 4, 75);
    vLcdWriteText("Periph` library", 5, 0);
    vUTIL_NumToString(sDemoData.sSystem.u32HwApiVersion, acString);
    vLcdWriteText(acString, 5, 75);

    /* Update node control screen multiple times to display all the data */
    vUpdateSetupScreen(1, FALSE);
    vUpdateSetupScreen(0, TRUE);
}

/****************************************************************************
 *
 * NAME: vUpdateSetupScreen
 *
 * DESCRIPTION:
 * Updates a single row of the Setup screen. The row label is either
 * highlighted or normal text, and the values are always 'on' or 'off'.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  u8Selection     R   Currently selected item (0-x)
 *                  boUpdate        R   TRUE if LCD should update afterwards
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vUpdateSetupScreen(uint8 u8Selection, bool_t boUpdate)
{
    static const char *apcRowName[SETUP_LIST_LEN] = {
        "Local node", "Four nodes"};

    /* Write row label highlighted, and previous row label in normal text */
    vWriteRowLabel(u8Selection, (char **)apcRowName, SETUP_LIST_LEN);

    switch (u8Selection)
    {
    case 0:
        vWriteOnOff(sDemoData.sNode.bLocalNode, 1, 75);
        break;

    case 1:
        vWriteOnOff(sDemoData.sGui.bShowFourNodes, 2, 75);
        break;
    }

    if (boUpdate)
    {
        vLcdRefreshAll();
    }
}

/****************************************************************************
 *
 * NAME: vProcessNetworkKeyPress
 *
 * DESCRIPTION:
 * Handles button presses on the Network screen. The buttons can move onto
 * the first Node screen (if there are any nodes) or select a particular
 * sensor.
 *
 * PARAMETERS:      Name        RW  Usage
 *                  u8KeyMap    R   Current buttons pressed bitmap
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vProcessNetworkKeyPress(uint8 u8KeyMap)
{
    switch (u8KeyMap)
    {
    case E_KEY_0:
        /* Node button: go to node screen (if there are any nodes) */
        if (sDemoData.sNode.u8AssociatedNodes > 0)
        {
            sDemoData.sSystem.eState = E_STATE_NODE;
            sDemoData.sGui.u8CurrentNode = 0;
            vBuildNodeScreen(sDemoData.sGui.u8CurrentNode);
        }
        break;

    case E_KEY_1:
        /* Temp button: change if not already there */
        vUpdateNetworkSensor(E_SENSOR_TEMP);
        break;

    case E_KEY_2:
        /* Humidity button: change if not already there */
        vUpdateNetworkSensor(E_SENSOR_HTS);
        break;

    case E_KEY_3:
        /* Temp button: change if not already there */
        vUpdateNetworkSensor(E_SENSOR_ALS);
        break;
    }
}

/****************************************************************************
 *
 * NAME: vUpdateNetworkSensor
 *
 * DESCRIPTION:
 * Simple function to save a little code. If the user presses a button on the
 * Network screen to select a sensor, this checks that the sensor is not the
 * same as the current sensor before updating the screen.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  eSensor         R   New sensor to display
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vUpdateNetworkSensor(teSensor eSensor)
{
    if (sDemoData.sGui.eCurrentSensor != eSensor)
    {
        sDemoData.sGui.eCurrentSensor = eSensor;
        vBuildNetworkScreen(eSensor);
    }
}


/****************************************************************************
 *
 * NAME: vBuildNetworkScreen
 *
 * DESCRIPTION:
 * Creates the Network screen. Depending on how the GUI has been configured
 * it may want to display up to 3 or up to 4 nodes simultaneuously. Also, it
 * only shows nodes that have successfully associated. To achieve this, it
 * makes use of an array of the four display positions on the screen, and
 * loops through this to position each node in the correct position.
 *
 * This function only draws the text required, then uses the related update
 * function to display the actual data and to refresh the LCD.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  eSensor         R   Sensor to display
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vBuildNetworkScreen(teSensor eSensor)
{
    static const char *apcSensorLabel[DEMO_SENSOR_LIST_LEN] = {
       "Temp", "Humidity", "Light"};
    static const uint8 au8LabelPos[DEMO_SENSOR_LIST_LEN] = {29, 58, 102};
    uint8 u8Node;
    int iPos;

    vLcdClear();

    /* If showing four nodes, they appear on screen as follows:
         Node 0    Node 1
         Node 2    Node 3

       If showing only three nodes, they appear as follows:
         Info      Node 0
         Node 1    Node 2
    */
    if (sDemoData.sGui.bShowFourNodes)
    {
        iPos = 0;
    }
    else
    {
        iPos = 1;
        vLcdWriteText("Network", 0, 0);
        vLcdWriteText("overview", 1, 0);
    }

    /* Show labels */
    if (sDemoData.sNode.u8AssociatedNodes == 0)
    {
        vLcdWriteText("No nodes associated", 3, 0);
    }
    else
    {
        u8Node = 0;
        while ((u8Node < sDemoData.sNode.u8AssociatedNodes) && (iPos < 4))
        {
            vLcdWriteText((char *)apcNodeNameList[u8Node], au8NodeLcdRow[iPos], au8NodeLcdCol[iPos]);
            u8Node++;
            iPos++;
        }
    }

    /* Hot buttons at bottom of screen */
    vLcdWriteText("Node", 7, 0);
    for (iPos = 0; iPos < DEMO_SENSOR_LIST_LEN; iPos++)
    {
        vLcdWriteText((char *)apcSensorLabel[iPos], 7, au8LabelPos[iPos]);
    }
    vLcdWriteInvertedText((char *)apcSensorLabel[eSensor], 7, au8LabelPos[eSensor]);

    vUpdateNetworkScreen(eSensor);
}

/****************************************************************************
 *
 * NAME: vUpdateNetworkScreen
 *
 * DESCRIPTION:
 * Draws the graphs and values for the Network screen. See the description
 * for vBuildNetworkScreen for an explanation of the positioning of elements
 * on the display.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  eSensor         R   Sensor to display
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vUpdateNetworkScreen(teSensor eSensor)
{
    uint8 u8Node;
    uint8 u8Row;
    uint8 u8Col;
    int iPos;
    bool_t bShowRssi;

    /* If showing four nodes, they appear on screen as follows:
         Node 0    Node 1
         Node 2    Node 3

       If showing only three nodes, they appear as follows:
         Info      Node 0
         Node 1    Node 2
    */
    if (sDemoData.sGui.bShowFourNodes)
    {
        iPos = 0;
    }
    else
    {
        iPos = 1;
    }

    u8Node = 0;
    bShowRssi = sDemoData.sNode.bLocalNode ? FALSE : TRUE;

    while ((u8Node < sDemoData.sNode.u8AssociatedNodes) && (iPos < 4))
    {
        u8Row = au8NodeLcdRow[iPos] + 1;
        u8Col = au8NodeLcdCol[iPos];

        vLcdUpdateElement(&sDemoData.sNode.asNodeData[u8Node],
                          eSensor, u8Row, u8Col, bShowRssi);

        u8Node++;
        iPos++;
        bShowRssi = TRUE;
    }

    vLcdRefreshAll();
}



/****************************************************************************
 *
 * NAME: vProcessNodeKeyPress
 *
 * DESCRIPTION:
 * Handles button presses on the Node screens. The first button can move to
 * the next Node screen (if there are any more nodes) or back to the Network
 * screen. Another button selects the Node Control screen and the other two
 * toggle the state of the remote switch.
 *
 * PARAMETERS:      Name        RW  Usage
 *                  u8KeyMap    R   Current buttons pressed bitmap
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vProcessNodeKeyPress(uint8 u8KeyMap)
{
    switch (u8KeyMap)
    {
    case E_KEY_0:
        /* Node button: go to next node or network screen */
        sDemoData.sGui.u8CurrentNode++;
        if (sDemoData.sGui.u8CurrentNode == sDemoData.sNode.u8AssociatedNodes)
        {
            sDemoData.sSystem.eState = E_STATE_NETWORK;
            sDemoData.sGui.eCurrentSensor = E_SENSOR_TEMP;
            vBuildNetworkScreen(E_SENSOR_TEMP);
        }
        else
        {
            vBuildNodeScreen(sDemoData.sGui.u8CurrentNode);
        }
        break;

    case E_KEY_1:
        /* Control screen button */
        sDemoData.sSystem.eState = E_STATE_NODE_CONTROL;
        sDemoData.sGui.u8ControlSelection = 0;
        vBuildNodeControlScreen(sDemoData.sGui.u8CurrentNode);
        break;


    case E_KEY_2:
    //     On button
        sDemoData.sNode.asNodeData[sDemoData.sGui.u8CurrentNode].boDeviceOn = TRUE;
        break;

    case E_KEY_3:
   //     Off button
        sDemoData.sNode.asNodeData[sDemoData.sGui.u8CurrentNode].boDeviceOn = FALSE;
        break;


    }
}

/****************************************************************************
 *
 * NAME: vBuildNodeScreen
 *
 * DESCRIPTION:
 * Builds the text to appear on a Node screen, then uses the update function
 * to populate it with data.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  u8Node          R   Node to display
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vBuildNodeScreen(uint8 u8Node)
{
    vLcdClear();
    vLcdWriteText((char *)apcNodeNameList[u8Node], 0, 0);
    vLcdWriteText("Humidity", 0, 64);
    vLcdWriteText("Temp", 3, 0);
    vLcdWriteText("Light", 3, 64);
    vLcdWriteText("Node", 7, 0);
    vLcdWriteText("Control", 7, 29);

    vLcdWriteText("On", 7, 77);
    vLcdWriteText("Off", 7, 107);

    vUpdateNodeScreen(u8Node);
}

/****************************************************************************
 *
 * NAME: vUpdateNodeScreen
 *
 * DESCRIPTION:
 * Draws the three sensor graphs for a node.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  u8Node          R   Node to display
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vUpdateNodeScreen(uint8 u8Node)
{
    tsNodeData *psNodeData;
    char acString[8];

    psNodeData = &sDemoData.sNode.asNodeData[u8Node];

    /* Status */
    if ((sDemoData.sNode.bLocalNode) && (u8Node != 0))
    {
        vValToDec(acString, psNodeData->u8Rssi, "    ");
        vLcdWriteText(acString, 1, 0);
        vValToDec(acString, psNodeData->u8FramesMissed - 1, "    ");
        vLcdWriteText(acString, 1, 20);
    }

    /* Update graphs, alarms and values */
    vLcdUpdateElement(psNodeData, E_SENSOR_TEMP, 4, 0, FALSE);
    vLcdUpdateElement(psNodeData, E_SENSOR_HTS, 1, 64, FALSE);
    vLcdUpdateElement(psNodeData, E_SENSOR_ALS, 4, 64, FALSE);

    vLcdRefreshAll();
}


/****************************************************************************
 *
 * NAME: vProcessNodeControlKeyPress
 *
 * DESCRIPTION:
 * Handles button presses on the Node Control screen. The first button
 * selects which item to alter, the next two adjust the value up or down, and
 * the last button returns to the Node screen.
 *
 * PARAMETERS:      Name        RW  Usage
 *                  u8KeyMap    R   Current buttons pressed bitmap
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vProcessNodeControlKeyPress(uint8 u8KeyMap)
{
    tsNodeData *psNodeData = &sDemoData.sNode.asNodeData[sDemoData.sGui.u8CurrentNode];
    bool_t bUpNotDown;

    switch (u8KeyMap)
    {
    case E_KEY_0:
        /* Select button: move to next item in list */
        vAdjustAlarm(&sDemoData.sGui.u8ControlSelection, CONTROL_LIST_LEN - 1, 0, TRUE);
        vUpdateNodeControlScreen(sDemoData.sGui.u8CurrentNode, sDemoData.sGui.u8ControlSelection, TRUE);
        break;

    case E_KEY_1:
        /* Plus button: increment value */
    case E_KEY_2:
        /* Minus button: decrement value */

        bUpNotDown = (u8KeyMap == E_KEY_1);

        switch (sDemoData.sGui.u8ControlSelection)
        {
        case 0:
            /* Temp high alarm */
            vAdjustAlarm(&psNodeData->asNodeElementData[E_SENSOR_TEMP].u8HighAlarm,
                         TEMP_HIGH_MAX, 0, bUpNotDown);
            break;

        case 1:
            /* Temp low alarm */
            vAdjustAlarm(&psNodeData->asNodeElementData[E_SENSOR_TEMP].u8LowAlarm,
                            TEMP_HIGH_MAX, 255, bUpNotDown);
            break;

        case 2:
            /* Light high alarm */
            vAdjustAlarm(&psNodeData->asNodeElementData[E_SENSOR_ALS].u8HighAlarm,
                            LIGHT_HIGH_MAX, 0, bUpNotDown);
            break;

        case 3:
            /* Light low alarm */
            vAdjustAlarm(&psNodeData->asNodeElementData[E_SENSOR_ALS].u8LowAlarm,
                            LIGHT_HIGH_MAX, 255, bUpNotDown);
            break;
        }

        vUpdateNodeControlScreen(sDemoData.sGui.u8CurrentNode, sDemoData.sGui.u8ControlSelection, TRUE);
        break;

    case E_KEY_3:
        /* Done button: return to node screen */
        sDemoData.sSystem.eState = E_STATE_NODE;
        vBuildNodeScreen(sDemoData.sGui.u8CurrentNode);
        break;
    }
}


/****************************************************************************
 *
 * NAME: vBuildNodeControlScreen
 *
 * DESCRIPTION:
 * Builds the text for a Node Control screen, then uses the update function
 * to show the values for each adjustable parameter in turn.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  u8Node          R   Node to display
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vBuildNodeControlScreen(uint8 u8Node)
{
    vLcdClear();
    vLcdWriteText((char *)apcNodeNameList[u8Node], 0, 0);
    vLcdWriteText("Select", 7, 0);
    vLcdWriteText("\\", 7, 47);
    vLcdWriteText("]", 7, 74);
    vLcdWriteText("Done", 7, 103);

    /* Update node control screen multiple times to display all the data */
    vUpdateNodeControlScreen(u8Node, 1, FALSE);
    vUpdateNodeControlScreen(u8Node, 2, FALSE);
    vUpdateNodeControlScreen(u8Node, 3, FALSE);
    vUpdateNodeControlScreen(u8Node, 0, TRUE);
}

/****************************************************************************
 *
 * NAME: vUpdateNodeControlScreen
 *
 * DESCRIPTION:
 * Updates a single row of a Node Control screen. The row label is either
 * highlighted or normal text, and the value must be displayed with a symbol
 * after it or, for light levels, purely as a symbol.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  u8Node          R   Node to display information for
 *                  u8Selection     R   Currently selected item (0-x)
 *                  boUpdate        R   TRUE if LCD should update afterwards
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vUpdateNodeControlScreen(uint8 u8Node, uint8 u8Selection,
                                      bool_t boUpdate)
{
    static const char *apcRowName[CONTROL_LIST_LEN] = {
        "Temp high alarm", "Temp low alarm", "Light high alarm",
        "Light low alarm"};
    char acString[10];
    tsNodeData *psNodeData = &sDemoData.sNode.asNodeData[u8Node];
    uint8 u8Value;

    /* Write row label highlighted, and previous row label in normal text */
    vWriteRowLabel(u8Selection, (char **)apcRowName, CONTROL_LIST_LEN);

    switch (u8Selection)
    {
    case 0:
        u8Value = psNodeData->asNodeElementData[E_SENSOR_TEMP].u8HighAlarm;
        if (u8Value == 0)
        {
            vLcdWriteText("off   ", 1, 90);
        }
        else
        {
            vValToDec(acString, u8Value, "[C   ");
            vLcdWriteText(acString, 1, 90);
        }
        break;

    case 1:
        u8Value = psNodeData->asNodeElementData[E_SENSOR_TEMP].u8LowAlarm;
        if (u8Value == 255)
        {
            vLcdWriteText("off   ", 2, 90);
        }
        else
        {
            vValToDec(acString, u8Value, "[C   ");
            vLcdWriteText(acString, 2, 90);
        }
        break;

    case 2:
        u8Value = psNodeData->asNodeElementData[E_SENSOR_ALS].u8HighAlarm;
        if (u8Value == 0)
        {
            vLcdWriteText("off", 3, 90);
        }
        else
        {
            acString[0] = '&' + u8Value;
            acString[1] = ' ';
            acString[2] = ' ';
            acString[3] = '\0';
            vLcdWriteText(acString, 3, 90);
        }
        break;

    default:
        u8Value = psNodeData->asNodeElementData[E_SENSOR_ALS].u8LowAlarm;
        if (u8Value == 255)
        {
            vLcdWriteText("off", 4, 90);
        }
        else
        {
            acString[0] = '&' + u8Value;
            acString[1] = ' ';
            acString[2] = ' ';
            acString[3] = '\0';
            vLcdWriteText(acString, 4, 90);
        }
        break;
    }

    if (boUpdate)
    {
        vLcdRefreshAll();
    }
}


/****************************************************************************
 *
 * NAME: vLcdUpdateElement
 *
 * DESCRIPTION:
 * Draws the graph and text for a single sensor for a single node. The text
 * includes alarm indications if the sensor value exceeds user specified
 * limits.
 *
 * PARAMETERS:  Name                RW  Usage
 *              psNodeElementData   R   Pointer to data for node
 *              u8Row               R   Character row to display on
 *              u8Col               R   Pixel column to display on
 *              bShowRssi           R   True to show RSSI data
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vLcdUpdateElement(tsNodeData *psNodeData, teSensor eSensor,
                               uint8 u8Row, uint8 u8Col, bool_t bShowRssi)
{
    char acString[10];
    uint8 u8NowValue;
    tsNodeElementData *psNodeElementData = &psNodeData->asNodeElementData[eSensor];

    u8NowValue = psNodeElementData->u8NowValue;

    switch (eSensor)
    {
    case E_SENSOR_TEMP:
        vValToDec(acString, u8NowValue, "[C ");
        break;

    case E_SENSOR_HTS:
        vValToDec(acString, u8NowValue, "% ");
        break;

    case E_SENSOR_ALS:
        /* This is a light sensor so display symbol */
        acString[0] = '&' + u8NowValue;
        acString[1] = '\0';
        break;

    default:
        break;
    }

    vLcdWriteText(acString, u8Row, u8Col);

    /* Print alarm */
    vLcdWriteText("       ", (uint8)(u8Row + 1), u8Col);

    if ((u8NowValue >= psNodeElementData->u8HighAlarm)
        && (psNodeElementData->u8HighAlarm != 0))
    {
        vLcdWriteInvertedText("High", (uint8)(u8Row + 1), u8Col);
    }
    else
    {
        if ((u8NowValue <= psNodeElementData->u8LowAlarm)
            && (psNodeElementData->u8LowAlarm != 255))
        {
            vLcdWriteInvertedText("Low", (uint8)(u8Row + 1), u8Col);
        }
        else
        {
            if (bShowRssi)
            {
                vValToDec(acString, psNodeData->u8Rssi, "");
                vLcdWriteText(acString, (uint8)(u8Row + 1), u8Col);

                vValToDec(acString, psNodeData->u8FramesMissed - 1, "");
                vLcdWriteText(acString, (uint8)(u8Row + 1), u8Col + 20);
            }
        }
    }

    /* Draw graph */
    vDrawGraph(psNodeElementData->au8GraphData, (uint8)(u8Col + 27), u8Row);
}




/****************************************************************************
 *
 * NAME: vDrawGraph
 *
 * DESCRIPTION:
 * Creates a bitmap from an array of values. Each value is represented by a
 * column on the graph, and a lookup table is used to translate each value
 * (assumed to be in the range 0 to 13) to the data required for the bitmap.
 * Finally, the bitmap is displayed via a board API.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  pu8GraphData    R   Array of 32 elements of graph data
 *                  u8StartCol      R   First column of bitmap
 *                  u8StartRow      R   Top character row of bitmap
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vDrawGraph(uint8 *pu8GraphData, uint8 u8StartCol,
                        uint8 u8StartRow)
{
    static const uint16 au16LineData[14] = {0x4000, 0x6000, 0x7000, 0x7800,
                                            0x7c00, 0x7e00, 0x7f00, 0x7f80,
                                            0x7fc0, 0x7fe0, 0x7ff0, 0x7ff8,
                                            0x7ffc, 0x7ffe};
    uint8 au8GraphBitmap[66];
    const tsBitmap sGraphBitmap = {au8GraphBitmap, 33, 2};
    int    i;
    uint16 u16LineData;
    uint8  u8DataPos = sDemoData.sGui.u8GraphPos;

    /* Draw y axis */
    au8GraphBitmap[0] = 0xfe;
    au8GraphBitmap[33] = 0x7f;

    /* Fill in data */
    for (i = 1; i <= DEMO_HISTORY_LEN; i += 1)
    {
        u16LineData = au16LineData[pu8GraphData[u8DataPos]];

        au8GraphBitmap[i] = (uint8)(u16LineData & 0xff);
        au8GraphBitmap[i + 33] = (uint8)(u16LineData >> 8);

        /* Increment data point */
        u8DataPos = (u8DataPos + 1) & (DEMO_HISTORY_LEN - 1);
    }

    /* Write bitmap to shadow memory */
    vLcdWriteBitmap((tsBitmap *)&sGraphBitmap, u8StartCol, u8StartRow);
}

/****************************************************************************
 *
 * NAME: vStringCopy
 *
 * DESCRIPTION:
 * Simple string copy as standard libraries not available.
 *
 * PARAMETERS:      Name    RW  Usage
 *                  pcFrom  R   Pointer to string to copy
 *                  pcTo    W   Pointer to store for new string
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vStringCopy(char *pcFrom, char *pcTo)
{
    while (*pcFrom != '\0')
    {
        *pcTo = *pcFrom;
        pcTo++;
        pcFrom++;
    }
    *pcTo = '\0';
}

/****************************************************************************
 *
 * NAME: vValToDec
 *
 * DESCRIPTION:
 * Converts an 8-bit value to a string of the textual decimal representation.
 * Adds a text string after the text.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  pcOutString     R   Location for new string
 *                  u8Value         R   Value to convert
 *                  pcLabel         R   Label to append to string
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vValToDec(char *pcOutString, uint8 u8Value, char *pcLabel)
{
    static const uint8 au8Digits[3] = {100, 10, 1};
    uint8 u8Digit;
    uint8 u8DigitIndex;
    uint8 u8Count;
    bool_t boPreviousDigitPrinted = FALSE;

    for (u8DigitIndex = 0; u8DigitIndex < 3; u8DigitIndex++)
    {
        u8Count = 0;
        u8Digit = au8Digits[u8DigitIndex];
        while (u8Value >= u8Digit)
        {
            u8Value -= u8Digit;
            u8Count++;
        }

        if ((u8Count != 0) || (boPreviousDigitPrinted == TRUE)
            || (u8DigitIndex == 2))
        {
            *pcOutString = '0' + u8Count;
            boPreviousDigitPrinted = TRUE;
            pcOutString++;
        }
    }

    vStringCopy(pcLabel, pcOutString);
}

/****************************************************************************
 *
 * NAME: vAdjustAlarm
 *
 * DESCRIPTION:
 * Increment a variable: If the variable is the maximum in the normal range,
 * sets it to a value that signifies 'off'. If the value is already 'off',
 * sets it to 0 (assumed to be the minimum within the normal range). This
 * function is used to set alarm levels.
 *
 * Decrement a variable: If the variable is 0 (assumed to be the minimum
 * within the normal range), sets it to a value that signifies 'off'. If the
 * value is already 'off', sets it to the maximum value in the normal range.
 * This function is used to set alarm levels.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  pu8Value        R   Pointer to variable to adjust
 *                  u8MaxValue      R   Maximum value in normal range
 *                  u8OffValue      R   Value that signifies 'off'
 *                  bUpNotDown      R   TRUE to increment, FALSE to decrement
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vAdjustAlarm(uint8 *pu8Value, uint8 u8MaxValue, uint8 u8OffValue,
                          bool_t bUpNotDown)
{
    if (bUpNotDown)
    {
        if (*pu8Value == u8MaxValue)
        {
            *pu8Value = u8OffValue;
        }
        else
        {
            if ((*pu8Value == u8OffValue ) && (u8OffValue > u8MaxValue))
            {
                *pu8Value = 0;
            }
            else
            {
                *pu8Value = *pu8Value + 1;
            }
        }
    }
    else
    {
        if (*pu8Value == u8OffValue)
        {
            *pu8Value = u8MaxValue;
        }
        else
        {
            if (*pu8Value == 0)
            {
                *pu8Value = u8OffValue;
            }
            else
            {
                *pu8Value = *pu8Value - 1;
            }
        }
    }
}

/****************************************************************************
 *
 * NAME: vStartBeacon
 *
 * DESCRIPTION:
 * Sends an MLME request to the 802.15.4 stack to start sending regular
 * beacons. All values are determined at compile time except the channel,
 * which can be adjusted by the user.
 *
 * If the request fails, generates a fatal error.
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vStartBeacon(void)
{
    MAC_MlmeReqRsp_s sMlmeReqRsp;
    MAC_MlmeSyncCfm_s sMlmeSyncCfm;

    /* Start beacons */
    sMlmeReqRsp.u8Type = MAC_MLME_REQ_START;
    sMlmeReqRsp.u8ParamLength = sizeof(MAC_MlmeReqStart_s);
    sMlmeReqRsp.uParam.sReqStart.u16PanId = DEMO_PAN_ID;
    sMlmeReqRsp.uParam.sReqStart.u8Channel = sDemoData.sSystem.u8Channel;
    sMlmeReqRsp.uParam.sReqStart.u8BeaconOrder = 3;     /* Eight beacons per second */
    sMlmeReqRsp.uParam.sReqStart.u8SuperframeOrder = 2; /* Only receive during first half of superframe: save energy */
    sMlmeReqRsp.uParam.sReqStart.u8PanCoordinator = TRUE;
    sMlmeReqRsp.uParam.sReqStart.u8BatteryLifeExt = FALSE;
    sMlmeReqRsp.uParam.sReqStart.u8Realignment = FALSE;
    sMlmeReqRsp.uParam.sReqStart.u8SecurityEnable = FALSE;
    vAppApiMlmeRequest(&sMlmeReqRsp, &sMlmeSyncCfm);

    if (sMlmeSyncCfm.u8Status != MAC_MLME_CFM_OK)
    {
        /* Error during MLME-Start: display on screen */
        vDisplayError("Failed to start beacons",
                      (uint32)sMlmeSyncCfm.u8Status
                      | (((uint32)sMlmeSyncCfm.uParam.sCfmStart.u8Status) << 8));
    }
}

/****************************************************************************
 *
 * NAME: vUpdateBeaconPayload
 *
 * DESCRIPTION:
 * Sends two MLME request to set values in teh PIB for the beacon payload.
 * The payload itself is set with alarm settings and remote switch values for
 * each node, and the payload length is set to 8 bytes.
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vUpdateBeaconPayload(void)
{
    uint8 u8Count;

    s_psMacPib->au8BeaconPayload[0] = DEMO_BEACON_IDENTIFIER;
    for (u8Count = 0; u8Count < DEMO_ENDPOINTS; u8Count++)
    {
        s_psMacPib->au8BeaconPayload[u8Count + 1]= (sDemoData.sNode.asNodeData[u8Count].u8FramesMissed << 4)
                                                   | ((sDemoData.sNode.asNodeData[u8Count].asNodeElementData[E_SENSOR_ALS].u8LowAlarm & 7) << 1)
                                                   | sDemoData.sNode.asNodeData[u8Count].boDeviceOn;
    }
    s_psMacPib->u8BeaconPayloadLength = 8;
}

/****************************************************************************
 *
 * NAME: vDisplayError
 *
 * DESCRIPTION:
 * Displays fatal errors on the LCD and then loops indefinitely.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  pcErrorMessage  R   Message string to display
 *                  u32Data         R   Data to display
 *
 * RETURNS:
 * void, never returns
 *
 ****************************************************************************/
PRIVATE void vDisplayError(char *pcErrorMessage, uint32 u32Data)
{
    char acValue[9];

    vLcdClear();
    vLcdWriteInvertedText(pcErrorMessage, 1, 0);
    //vUTIL_NumToString(u32Data, acValue);
    vLcdWriteText(acValue, 2, 0);
    vLcdRefreshAll();

    /* Fatal error has occurred, so loop indefinitely */
    while (1);
}

/****************************************************************************
 *
 * NAME: vWriteOnOff
 *
 * DESCRIPTION:
 * Displays the text 'on' or 'off' at the specified location on the screen.
 * The 'on' text includes two spaces to overwrite any previous 'off' text
 * completely.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  bOnOff  R   TRUE for on, FALSE for off
 *                  u8Row   R   Character row for string
 *                  u8Col   R   Pixel column for string
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vWriteOnOff(bool_t bOnOff, uint8 u8Row, uint8 u8Col)
{
    static const char *apcOnOff[2] = {"off", "on  "};

    vLcdWriteText((char *)apcOnOff[bOnOff], u8Row, u8Col);
}

/****************************************************************************
 *
 * NAME: vToggleOnOff
 *
 * DESCRIPTION:
 * Toggles a value between TRUE and FALSE.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  pbItem          W   Pointer to boolean to toggle
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vToggleOnOff(bool_t *pbItem)
{
    if (*pbItem == TRUE)
    {
        *pbItem = FALSE;
    }
    else
    {
        *pbItem = TRUE;
    }
}

/****************************************************************************
 *
 * NAME: vWriteRowLabel
 *
 * DESCRIPTION:
 * Used by the screen update functions, this takes an array of row labels and
 * displays the selected on as highlighted and the previous one as normal.
 * This is required when the user selects the next item in a list.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  u8Selection     R   Selected row (0-x)
 *                  ppcRowName      R   Array of pointers to label strings
 *                  u8ListLen       R   Length of list
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vWriteRowLabel(uint8 u8Selection, char **ppcRowName, uint8 u8ListLen)
{
    uint8 u8PrevRow;

    if (u8Selection == 0)
    {
        u8PrevRow = u8ListLen;
    }
    else
    {
        u8PrevRow = u8Selection;
    }

    vLcdWriteText((char *)ppcRowName[u8PrevRow - 1], u8PrevRow, 0);

    vLcdWriteInvertedText((char *)ppcRowName[u8Selection], u8Selection + 1, 0);
}

/****************************************************************************/
/***        END OF FILE                                                   ***/
/****************************************************************************/
