/****************************************************************************/
/*
 * MODULE              JN-AN-1162 JenNet-IP Smart Home
 *
 * DESCRIPTION         DeviceControl MIB Implementation (for sensors)
 */
/****************************************************************************/
/*
 * This software is owned by NXP B.V. and/or its supplier and is protected
 * under applicable copyright laws. All rights are reserved. We grant You,
 * and any third parties, a license to use this software solely and
 * exclusively on NXP products [NXP Microcontrollers such as JN5168, JN5164].
 * You, and any third parties must reproduce the copyright and warranty notice
 * and any other legend of ownership on each copy or partial copy of the
 * software.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * Copyright NXP B.V. 2014. All rights reserved
 */
/****************************************************************************/

/****************************************************************************/
/***        Include files                                                 ***/
/****************************************************************************/
/* Standard includes */
#include <string.h>
/* SDK includes */
#include <jendefs.h>
/* Hardware includes */
#include <AppHardwareApi.h>
#include <PeripheralRegs.h>
/* Stack includes */
#include <Api.h>
#include <AppApi.h>
#include <JIP.h>
#include <6LP.h>
#include <AccessFunctions.h>
/* JenOS includes */
#include <dbg.h>
#include <dbg_uart.h>
#include <os.h>
#include <pdm.h>
/* Application device includes */
#include "DeviceDefs.h"
#include "Node.h"
#include "MibCommon.h"
#include "MibSensor.h"
#include "MibDeviceControl.h"
#include "MibOccupancyControl.h"
#include "MibIlluminanceControl.h"

/****************************************************************************/
/***        Macro Definitions                                             ***/
/****************************************************************************/
/* Data access */
#define PS_MIB_OCCUPANCY_CONTROL   ((tsMibOccupancyControl *)   psMibDeviceControl->pvMibOccupancyControl)
#define PS_MIB_ILLUMINANCE_CONTROL ((tsMibIlluminanceControl *) psMibDeviceControl->pvMibIlluminanceControl)

/****************************************************************************/
/***        Type Definitions                                              ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Function Prototypes                                     ***/
/****************************************************************************/

/****************************************************************************/
/***        Exported Variables                                            ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Variables                                               ***/
/****************************************************************************/
PRIVATE tsMibDeviceControl *psMibDeviceControl;  /* MIB data */

/****************************************************************************/
/***        Exported Functions                                            ***/
/****************************************************************************/

/****************************************************************************
 *
 * NAME: MibDeviceControl_vInit
 *
 * DESCRIPTION:
 * Initialises data
 *
 ****************************************************************************/
PUBLIC void	MibDeviceControl_vInit(thJIP_Mib			  hMibDeviceControlInit,
						 		   tsMibDeviceControl	*psMibDeviceControlInit,
			                       void 				*pvMibOccupancyControlInit,
			                       void 				*pvMibIlluminanceControlInit)
{
    /* Debug */
    DBG_vPrintf(DEBUG_MIB_DEVICE_CONTROL, "\n%sMibDeviceControl_vInit() {%d}", acDebugIndent, sizeof(tsMibDeviceControl));

    /* Valid data pointer ? */
    if (psMibDeviceControlInit != (tsMibDeviceControl *) NULL)
    {
        /* Take copy of pointer to data */
        psMibDeviceControl = psMibDeviceControlInit;
        /* Take a copy of the MIB handle */
        psMibDeviceControl->hMib = hMibDeviceControlInit;

		/* Building OccupancyControl MIB ? */
		#if MK_BLD_MIB_OCCUPANCY_CONTROL
		{
			/* Take a copy of the occupancy control MIB pointer */
			psMibDeviceControl->pvMibOccupancyControl   = pvMibOccupancyControlInit;
			/* Got an OccupancyControl MIB pointer ? */
			if (PS_MIB_OCCUPANCY_CONTROL != NULL)
			{
				/* Is the occupancy monitor turned on ? */
				if (PS_MIB_OCCUPANCY_CONTROL->sPerm.u8Mode != VAR_VAL_OCCUPANCY_CONTROL_MODE_OFF)
				{
					/* Initialise the DeviceControl Mode variable to on too */
					psMibDeviceControl->sTemp.u8Mode = VAR_VAL_OCCUPANCY_CONTROL_MODE_ON;
				}
			}
		}
		#endif

		/* Building IlluminanceControl MIB ? */
		#if MK_BLD_MIB_ILLUMINANCE_CONTROL
		{
			/* Take a copy of the illuminance control MIB pointer */
			psMibDeviceControl->pvMibIlluminanceControl = pvMibIlluminanceControlInit;
			/* Got an IlluminanceControl MIB pointer ? */
			if (PS_MIB_ILLUMINANCE_CONTROL != NULL)
			{
				/* Is the illuminance monitor turned on ? */
				if (PS_MIB_ILLUMINANCE_CONTROL->sPerm.u8Mode != VAR_VAL_ILLUMINANCE_CONTROL_MODE_OFF)
				{
					/* Initialise the DeviceControl Mode variable to on too */
					psMibDeviceControl->sTemp.u8Mode = VAR_VAL_OCCUPANCY_CONTROL_MODE_ON;
				}
				/* Take a copy of the IlluminanceControl SceneId variable */
				psMibDeviceControl->sTemp.u16SceneId = PS_MIB_ILLUMINANCE_CONTROL->sPerm.u16SceneId;
			}
		}
		#endif
    }
}

/****************************************************************************
 *
 * NAME: MibDeviceControl_vRegister
 *
 * DESCRIPTION:
 * Registers MIB
 *
 ****************************************************************************/
PUBLIC void MibDeviceControl_vRegister(void)
{
    teJIP_Status eStatus;

    /* Register MIB */
    eStatus = eJIP_RegisterMib(psMibDeviceControl->hMib);
    /* Debug */
    DBG_vPrintf(DEBUG_MIB_DEVICE_CONTROL, "\n%sMibDeviceControl_vRegister() { eJIP_RegisterMib(DeviceControl) = %d }", acDebugIndent, eStatus);
}

/****************************************************************************
 *
 * NAME: MibDeviceControl_eSetMode
 *
 * DESCRIPTION:
 * Mode set data callback
 *
 ****************************************************************************/
PUBLIC teJIP_Status MibDeviceControl_eSetMode(uint8 u8Val, void *pvCbData)
{
    teJIP_Status eReturn            = E_JIP_ERROR_DISABLED;
    teJIP_Status eOccupancyReturn   = E_JIP_ERROR_DISABLED;
    teJIP_Status eIlluminanceReturn = E_JIP_ERROR_DISABLED;

    /* Invalid mode ? */
    if (u8Val >= VAR_VAL_DEVICE_CONTROL_MODE_COUNT)
    {
		/* Return bad value */
		eReturn = E_JIP_ERROR_BAD_VALUE;
	}
	/* Valid value ? */
	else
	{
		/* Toggling mode ? */
		if (u8Val == VAR_VAL_DEVICE_CONTROL_MODE_TOGGLE)
		{
			/* Calculate toggled value */
			if (psMibDeviceControl->sTemp.u8Mode == VAR_VAL_DEVICE_CONTROL_MODE_OFF)
			{
				u8Val = VAR_VAL_DEVICE_CONTROL_MODE_ON;
			}
			else
			{
				u8Val = VAR_VAL_DEVICE_CONTROL_MODE_OFF;
			}
		}

		/* Building OccupancyControl MIB ? */
		#if MK_BLD_MIB_OCCUPANCY_CONTROL
		{
			/* Got an OccupancyControl MIB pointer ? */
			if (PS_MIB_OCCUPANCY_CONTROL != NULL)
			{
				/* Call OccupancyControl function */
				eOccupancyReturn = MibOccupancyControl_eSetMode(u8Val, &PS_MIB_ILLUMINANCE_CONTROL->sPerm.u8Mode);
				/* Success ? */
				if (eOccupancyReturn == E_JIP_OK)
				{
					/* Set notify for this variable */
					PS_MIB_OCCUPANCY_CONTROL->u32NotifyChanged |= (1 << VAR_IX_OCCUPANCY_CONTROL_MODE);
				}
			}
		}
		#endif

		/* Building IlluminanceControl MIB ? */
		#if MK_BLD_MIB_ILLUMINANCE_CONTROL
		{
			/* Got an IlluminanceControl MIB pointer ? */
			if (PS_MIB_ILLUMINANCE_CONTROL != NULL)
			{
				/* Call Illuminance control function */
				eIlluminanceReturn = MibIlluminanceControl_eSetMode(u8Val, &PS_MIB_ILLUMINANCE_CONTROL->sPerm.u8Mode);
				/* Success ? */
				if (eIlluminanceReturn == E_JIP_OK)
				{
					/* Set notify for this variable */
					PS_MIB_ILLUMINANCE_CONTROL->u32NotifyChanged |= (1 << VAR_IX_ILLUMINANCE_CONTROL_MODE);
				}
			}
		}
		#endif

		/* Did Occupancy or Illuminance succeed ? */
		if (eOccupancyReturn == E_JIP_OK || eIlluminanceReturn == E_JIP_OK)
		{
			/* Call standard function */
			eReturn = eSetUint8(u8Val, pvCbData);
		}
	}

	return eReturn;
}

/****************************************************************************
 *
 * NAME: MibDeviceControl_eSetSceneId
 *
 * DESCRIPTION:
 * Generic set data callback
 *
 ****************************************************************************/
PUBLIC teJIP_Status MibDeviceControl_eSetSceneId(uint16 u16Val, void *pvCbData)
{
    teJIP_Status eReturn;
    teJIP_Status eIlluminanceReturn = E_JIP_ERROR_DISABLED;

	/* Building IlluminanceControl MIB ? */
	#if MK_BLD_MIB_ILLUMINANCE_CONTROL
	{
		/* Got an IlluminanceControl MIB pointer ? */
		if (PS_MIB_ILLUMINANCE_CONTROL != NULL)
		{
			/* Call Illuminance control function */
			eIlluminanceReturn = MibIlluminanceControl_eSetSceneId(u16Val, &PS_MIB_ILLUMINANCE_CONTROL->sPerm.u16SceneId);
			/* Success ? */
			if (eIlluminanceReturn == E_JIP_OK)
			{
				/* Set notify for this variable */
				PS_MIB_ILLUMINANCE_CONTROL->u32NotifyChanged |= (1 << VAR_IX_ILLUMINANCE_CONTROL_SCENE_ID);
			}
		}
	}
	#endif

	/* Did Illuminance succeed ? */
	if (eIlluminanceReturn == E_JIP_OK)
	{
		/* Call standard function */
		eReturn = eSetUint16(u16Val, pvCbData);
	}
	/* Illuminance failed ? */
	else
	{
		/* Return value from Illuminance */
		eReturn = eIlluminanceReturn;
	}

    return eReturn;
}

/****************************************************************************/
/***        END OF FILE                                                   ***/
/****************************************************************************/
