/****************************************************************************/
/*
 * MODULE              JN-AN-1162 JenNet-IP Smart Home
 *
 * DESCRIPTION         BulbScene MIB Implementation
 */
/****************************************************************************/
/*
 * This software is owned by NXP B.V. and/or its supplier and is protected
 * under applicable copyright laws. All rights are reserved. We grant You,
 * and any third parties, a license to use this software solely and
 * exclusively on NXP products [NXP Microcontrollers such as JN5168, JN5164].
 * You, and any third parties must reproduce the copyright and warranty notice
 * and any other legend of ownership on each copy or partial copy of the
 * software.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * Copyright NXP B.V. 2014. All rights reserved
 */
/****************************************************************************/

/****************************************************************************/
/***        Include files                                                 ***/
/****************************************************************************/
/* Standard includes */
#include <string.h>
/* SDK includes */
#include <jendefs.h>
/* Hardware includes */
#include <AppHardwareApi.h>
#include <PeripheralRegs.h>
/* Stack includes */
#include <Api.h>
#include <AppApi.h>
#include <JIP.h>
#include <6LP.h>
#include <AccessFunctions.h>
/* JenOS includes */
#include <dbg.h>
#include <dbg_uart.h>
#include <os.h>
#include <pdm.h>
/* Application common includes */
#include "DeviceDefs.h"
#include "Node.h"
#include "MibBulb.h"
#include "Table.h"
/* Application device includes */
#include "MibBulb.h"
#include "MibBulbScene.h"
#include "MibCommon.h"
#include "MibDeviceScene.h"

/****************************************************************************/
/***        Macro Definitions                                             ***/
/****************************************************************************/

/****************************************************************************/
/***        Type Definitions                                              ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Function Prototypes                                     ***/
/****************************************************************************/

/****************************************************************************/
/***        Exported Variables                                            ***/
/****************************************************************************/

/****************************************************************************/
/***        External Variables                                            ***/
/****************************************************************************/
/* MIB structures */
extern tsMibBulbScene     sMibBulbScene;
extern thJIP_Mib 	      hMibBulbScene;

#if MK_BLD_MIB_DEVICE_SCENE
extern tsMibDeviceScene  sMibDeviceScene;
#endif

/****************************************************************************/
/***        Local Variables                                               ***/
/****************************************************************************/

/****************************************************************************/
/***        Exported Functions                                            ***/
/****************************************************************************/

/****************************************************************************
 *
 * NAME: MibBulbScene_vInit
 *
 * DESCRIPTION:
 * Initialises data
 *
 ****************************************************************************/
PUBLIC void MibBulbScene_vInit(void)
{
	/* Debug */
	DBG_vPrintf(DEBUG_MIB_BULB_SCENE,  "\n%sMibBulbScene_vInit() {%d}", acDebugIndent, sizeof(tsMibBulbScene));
}

/****************************************************************************
 *
 * NAME: MibBulbScene_vRegister
 *
 * DESCRIPTION:
 * Registers MIB
 *
 ****************************************************************************/
PUBLIC void MibBulbScene_vRegister(void)
{
	teJIP_Status eStatus;

	/* Debug */
	DBG_vPrintf(DEBUG_MIB_BULB_SCENE, "\n%sMibBulbScene_vRegister()", acDebugIndent);
	Node_vDebugIndent(DEBUG_MIB_BULB_SCENE);
	/* Register MIB */
	eStatus = eJIP_RegisterMib(hMibBulbScene);
	/* Debug */
	DBG_vPrintf(DEBUG_MIB_BULB_SCENE, "\n%seJIP_RegisterMib(BulbScene) = %d", acDebugIndent, eStatus);

	/* Configure table structures */
	sMibBulbScene.sSceneId.pvData	         = (void *) sMibBulbScene.sTemp.au16SceneId;
	sMibBulbScene.sSceneId.u32Size	         = sizeof(uint16);
	sMibBulbScene.sSceneId.u16Entries        = DEVICE_SCENES;

	sMibBulbScene.sSceneMode.pvData			 = (void *) sMibBulbScene.sTemp.au8SceneMode;
	sMibBulbScene.sSceneMode.u32Size		 = sizeof(uint8);
	sMibBulbScene.sSceneMode.u16Entries		 = DEVICE_SCENES;

	sMibBulbScene.sSceneLumTarget.pvData	 = (void *) sMibBulbScene.sTemp.au8SceneLumTarget;
	sMibBulbScene.sSceneLumTarget.u32Size	 = sizeof(uint8);
	sMibBulbScene.sSceneLumTarget.u16Entries = DEVICE_SCENES;

	/* Debug */
	Node_vDebugOutdent(DEBUG_MIB_BULB_SCENE);
}

/****************************************************************************
 *
 * NAME: MibBulbScene_vTick
 *
 * DESCRIPTION:
 * Timer function called when stack is running
 *
 ****************************************************************************/
PUBLIC void MibBulbScene_vTick(void)
{
	/* Are there any variable notifications pending ? */
	if ((sMibBulbScene.u32NotifyChanged & VAR_MASK_BULB_SCENE) != 0)
	{
		/* Use common function to output notifications */
		Node_vJipNotifyChanged(hMibBulbScene, &sMibBulbScene.u32NotifyChanged, VAR_MASK_BULB_SCENE, VAR_COUNT_BULB_SCENE);
	}
}

/****************************************************************************
 *
 * NAME: MibBulbScene_eSetAddSceneId
 *
 * DESCRIPTION:
 * AddSceneId set data callback
 *
 ****************************************************************************/
PUBLIC teJIP_Status MibBulbScene_eSetAddSceneId(uint16 u16Val, void *pvCbData)
{
	teJIP_Status eReturn = E_JIP_ERROR_FAILED;

	/* DeviceScene MIB is being built ? */
	#if MK_BLD_MIB_DEVICE_SCENE
	{
		/* Attempt to make a scene from current settings */
		eReturn = MibDeviceScene_eMakeScene(u16Val);
		/* Did we add this scene ? */
		if (eReturn == E_JIP_OK)
		{
			/* Set this variable */
			(void) eSetUint16(u16Val, pvCbData);

			/* Echo to DeviceScene MIB's AddSceneId variable */
			(void) eSetUint16(u16Val, (void *) &sMibDeviceScene.sTemp.u16AddSceneId);
			/* Queue notification */
			sMibDeviceScene.u32NotifyChanged |= (1 << VAR_IX_DEVICE_SCENE_ADD_SCENE_ID);
		}
	}
	#endif

	return eReturn;
}

/****************************************************************************
 *
 * NAME: MibBulbScene_eSetDelSceneId
 *
 * DESCRIPTION:
 * DelSceneId set data callback
 *
 ****************************************************************************/
PUBLIC teJIP_Status MibBulbScene_eSetDelSceneId(uint16 u16Val, void *pvCbData)
{
	teJIP_Status eReturn = E_JIP_ERROR_FAILED;

	/* DeviceScene MIB is being built ? */
	#if MK_BLD_MIB_DEVICE_SCENE
	{
		/* Attempt to delete scene from current settings */
		eReturn = MibDeviceScene_eDelScene(u16Val);

		/* Did we delete this scene ? */
		if (eReturn == E_JIP_OK)
		{
			/* Set this variable */
			(void) eSetUint16(u16Val, pvCbData);

			/* Echo to DeviceScene MIB's DelSceneId variable */
			(void) eSetUint16(u16Val, (void *) &sMibDeviceScene.sTemp.u16DelSceneId);
			/* Queue notification */
			sMibDeviceScene.u32NotifyChanged |= (1 << VAR_IX_DEVICE_SCENE_DEL_SCENE_ID);
		}
	}
	#endif

	return eReturn;
}

/****************************************************************************/
/***        END OF FILE                                                   ***/
/****************************************************************************/
