/*
 * @brief Image programming code
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2014
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */

#include <stdint.h>
#include <string.h>
#include "board.h"

#include "stdlib.h"

#include "sl_common.h"
#include "sl_protocol.h"
#include "app_usbd_cfg.h"
#include "sl_flash.h"

#include "crc_drv.h"

#include "sbl_lpc.h"

/* DFU boot definitions */
#define DFU_DEST_BASE         SL_APP_START_ADDR
#define DFU_MAX_IMAGE_LEN     (SL_FLASH_END - SL_APP_START_ADDR)
#define DFU_MAX_BLOCKS        (DFU_MAX_IMAGE_LEN / USB_DFU_XFER_SIZE)

extern volatile uint32_t dwSysTicks;

extern void dfu_detach(USBD_HANDLE_T hUsb);

volatile uint8_t g_SubBlockWrData[SL_FLASH_BLOCK_SZ];
volatile uint16_t g_SubBlockWrOffset = 0;

#if CRP_ENABLE
static uint32_t keyFlash[SL_KEY_BYTES / sizeof(uint32_t)];
#endif

static int      dfuDone          = FALSE;
static int      reinvokeISP      = FALSE;
static uint32_t reinvokeISPDelay = 0x4000;
static uint32_t appAddress       = SL_APP_START_ADDR;

//static const char *g_sblVersion = "(" __DATE__ " " __TIME__ ")";

#if CRP_ENABLE
/**
 * @brief	Clean flags in RAM
 * @param   Nothing
 * @return	Nothing
 */
void cleanRAM(void)
{
	memset(&keyFlash[0], 0x00, SL_KEY_BYTES);
}
#endif

/**
 * @brief	Check Image header is valid or not
 * @param   Image header
 * @return	true/false
 */
static bool ImgHdrValid(void *new_hdr)
{
	IMG_HEADER_T *pNewImageHeader = (IMG_HEADER_T *) new_hdr;
	/* Compare with the IMAGE Header marker */
	if (pNewImageHeader->header_marker != IMG_HEADER_MARKER)
	{
		/* return false, when the marker is not same */
		return FALSE;
	}

	return TRUE;
}

/**
 * @brief	Copy valid firmware data into flash
 * @param   param1: data buffer -> g_SubBlockWrData[]
 *          param2: data length -> SL_FLASH_BLOCK_SZ
 * @return	Nothing
 */
void doflashWrite(uint8_t *data, uint32_t size)
{
	IMG_HEADER_T *pNewImageHeader;
	/* The fist package from Host */
    if (gLPCflashOffset == 0)
    {
		pNewImageHeader = (IMG_HEADER_T *)&data[SL_BOOTAPP_IMGHDR_OFS];
		if ( pNewImageHeader->img_type == DFU_SBL_UPDATE )
		{
			/* Mass storage isp for factory image update */
			reinvokeISP = TRUE;
			reinvokeISPDelay = 0x4000;
			g_SubBlockWrOffset = 0;
			return;
		}

        /* This is the actual image being downloaded so calculate the image address */

        /* Extract Reset Vector address */
        appAddress = *(((uint32_t *)data) + 1);
        if(appAddress >= gLPCdualImageBoundary)
        {
            appAddress = gLPCdualImageBoundary;
        }
        else
        {
            appAddress = SL_APP_START_ADDR;
        }
	}

	/* LPC11U6x write a block of flash */
	flashWriteBlock((uint32_t *) &data[0],(gLPCflashOffset + appAddress)/SL_FLASH_BLOCK_SZ);

	g_SubBlockWrOffset = 0;
	gLPCflashOffset += size;
	return;
}

/**
 * @brief	Process DFU actions
 * @param   Nothing
 * @return	Nothing
 */
void DFU_ProcessData(void)
{
	if (reinvokeISP)
	{
		/* if DFU require reinvoke ISP */
		if ( reinvokeISPDelay == 0x0 )
		{
			/* clear flags and init delay counter value */
			reinvokeISP = FALSE;
			reinvokeISPDelay = 0x4000;
			/* Call ReinvokeISP */
			ReinvokeISP();
		}
		else
		{
			/* minus delay counter value */
			reinvokeISPDelay--;
		}
	}
}

/**
 * @brief	Process DFU read data action
 * @param   param1:flash block number, param2:data buffer, param3:Amount of data copied to destination buffer
 * @return	DFU_STATUS_ : values defined in mw_usbd_dfu.h in case of errors
 *          0           : If end of memory reached
 */
uint32_t dfu_rd(uint32_t block_num, uint8_t * *pBuff, uint32_t length)
{
	uint32_t src_addr = DFU_DEST_BASE;
	dwSysTicks = 0;
	if ( length != 0 ) {
		if (block_num == DFU_MAX_BLOCKS)
		{
			return 0;
		}

		if (block_num > DFU_MAX_BLOCKS)
		{
			return DFU_STATUS_errADDRESS;
		}

		src_addr += (block_num * USB_DFU_XFER_SIZE);

		memcpy((void *) (*pBuff), (void *) src_addr, length);
	}
	
#if   defined ( __GNUC__ )
	__ASM volatile ("cpsid i" : : : "memory");
#elif defined ( __CC_ARM )
	__ASM ("cpsid i");
#else
	XXXX
#endif
	
	return length;
}

/**
 * @brief	Process DFU write data action
 * @param   param1:flash block number, param2:data buffer, param3:Amount of data copied from source buffer
 * @return	DFU_STATUS_ : values defined in mw_usbd_dfu.h
 */
uint8_t dfu_wr(uint32_t block_num, uint8_t * *pBuff, uint32_t length,
			   uint8_t *bwPollTimeout)
{
	/* Clear SysTick Counter */
	dwSysTicks = 0;
	if ( length != 0 )
	{
		if (block_num >= DFU_MAX_BLOCKS)
		{
			return DFU_STATUS_errADDRESS;
		}

		if ( block_num == 0x0 )
		{
			g_SubBlockWrOffset = 0;
		}

		if (g_SubBlockWrOffset == 0)
		{
			memset((void *)&g_SubBlockWrData[0], 0xFF, SL_FLASH_BLOCK_SZ);
		}
		memcpy((void *)&g_SubBlockWrData[g_SubBlockWrOffset], &((*pBuff)[0]), length);
		g_SubBlockWrOffset += length;
//		flash_block_nr = (SL1_BOOTAPP_ADDR + (block_num * USB_DFU_XFER_SIZE))/SL_FLASH_BLOCK_SZ;
	}

	if (g_SubBlockWrOffset >= SL_FLASH_BLOCK_SZ)
	{
		if ( (gLPCflashOffset == 0) && (ImgHdrValid((void *)&g_SubBlockWrData[SL_BOOTAPP_IMGHDR_OFS]) == FALSE) )
		{
			g_SubBlockWrOffset = 0;
			return DFU_STATUS_errFIRMWARE;
		}
		else
		{
			doflashWrite((uint8_t *)g_SubBlockWrData, SL_FLASH_BLOCK_SZ);
			g_SubBlockWrOffset = 0;
		}
	}

	return DFU_STATUS_OK;
}

/**
 * @brief	Process DFU finished action
 * @param   Nothing
 * @return	Nothing
 */
void dfu_done(void)
{
	/* Mark dfuDone flag and reset system */
	dfuDone = TRUE;
	//DEBUGOUT("\r\n dfu done \r\n");
	NVIC_SystemReset();
}

/**
 * @brief	Process USB detached event
 * @param   Nothing
 * @return	Nothing
 */
void dfu_detach(USBD_HANDLE_T hUsb)
{
	/* TODO: Should be User code here */
	return;
}

/**
 * @brief	initial usb dfu
 * @param   param1: usb device handler, 
 *          param2: usb deivce descriptor
 *          param3: usb device buffer address
 *          param4: usb device buffer length
 * @return	Error Code
 */
ErrorCode_t usb_dfu_init(USBD_HANDLE_T hUsb,
						 USB_INTERFACE_DESCRIPTOR *pIntfDesc,
						 uint32_t *mem_base,
						 uint32_t *mem_size)
{
	USBD_DFU_INIT_PARAM_T dfu_param;
	ErrorCode_t ret = LPC_OK;											/* initial return value as LPC_OK */

	memset((void *) &dfu_param, 0, sizeof(USBD_DFU_INIT_PARAM_T));		/* initial dfu_param */
	dfu_param.mem_base = *mem_base;										/* set USB device buffer memory address pointer */
	dfu_param.mem_size = *mem_size;										/* Set USB device buffer length */
	/* DFU params */
	dfu_param.wTransferSize = USB_DFU_XFER_SIZE;						/* Set USB device transfer length */
	/* Check descriptor is valid */
	if ((pIntfDesc == 0) ||
		(pIntfDesc->bInterfaceClass != USB_DEVICE_CLASS_APP) ||
		(pIntfDesc->bInterfaceSubClass != USB_DFU_SUBCLASS) ) {
		return ERR_FAILED;
	}

	dfu_param.intf_desc = (uint8_t *) pIntfDesc;
	/* user defined functions */
	dfu_param.DFU_Write     = dfu_wr;
	dfu_param.DFU_Read      = dfu_rd;
	dfu_param.DFU_Done      = dfu_done;
	dfu_param.DFU_Detach    = dfu_detach;
	/* Call USB DFU Initial ROM API */
	ret = USBD_API->dfu->init(hUsb, &dfu_param, DFU_STATE_dfuIDLE);
	/* update memory variables */
	*mem_base = dfu_param.mem_base;
	*mem_size = dfu_param.mem_size;
	
	return ret;
}

// end file

