/*
 * @brief CRC32 check of image code
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2014
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */

#include "sl_common.h"
#include "crc_drv.h"
#include "sbl_lpc.h"

/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/

/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/

/*****************************************************************************
 * Private functions
 ****************************************************************************/

/*****************************************************************************
 * Public functions
 ****************************************************************************/

/* Compute CRC32 for the image programmed in flash. */
uint32_t computeAppCRC(uint32_t imgAddr)
{
    uint32_t crc, appEnd = ((imgAddr < LPC_SBL_GetDualImgBoundary()) ? LPC_SBL_GetDualImgBoundary() : LPC_SBL_GetFlashSize());
    uint32_t len = ((IMG_HEADER_T *)(imgAddr + SL_BOOTAPP_IMGHDR_OFS))->crc_len;
    uint32_t crcValAddr = (uint32_t)(&(((IMG_HEADER_T *)(imgAddr + SL_BOOTAPP_IMGHDR_OFS))->crc_value));

    /* Sanity checks */
    if ((len == 0) || (len > (appEnd - imgAddr)))
    {
        return len;
    }

    /* Initialize CRC engine */
    CRC32_Initialise(&crc);

    /* CRC32 computation in bytes, skipping crc val field */
    CRC32_Generate(&crc, (uint8_t *)imgAddr, (int32_t)(crcValAddr - imgAddr));
    len = len - (crcValAddr - imgAddr);
    if (len != 0)
    {
        /* Skip over crc value and adjust length */
        len = len - 4;
        CRC32_Generate(&crc, (uint8_t *)(crcValAddr + 4), (int32_t)(len));
    }
    CRC32_Complete(&crc);

	return crc;
}


/* Verify that application's CRC is correct. Used to determine if a valid
   application is located in FLASH before jumping to it. */
uint32_t checkAppCRC(uint32_t imgAddr)
{
    uint32_t calcCrc = 0;
    uint32_t flashCrc = ((IMG_HEADER_T *)(imgAddr + SL_BOOTAPP_IMGHDR_OFS))->crc_value;

    /* compute app CRC */
    calcCrc = computeAppCRC(imgAddr);

    /* Compare */
    if (flashCrc == calcCrc)
    {
        calcCrc = 0;
    }

	return calcCrc;
}

/* Calculate the CRC valus based on the length(len) and data buffer(data) */
uint32_t computeCrc32(uint32_t *data, uint32_t len)
{
	uint32_t crc;

	crc = CRC32_Compute((uint8_t *)(data), (int32_t)(len));

	return crc;
}

// end file
