/*
 * @brief UART API in ROM (USART API ROM) interrupt example
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2012
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */

#include "board.h"
#include "fw_iap.h"
#include <string.h>


/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/
/* message type definitions */
#define 	FW_UPDATE_REQ 		(0x10)				/*!< firmware update request*/
#define		FW_DATA_PACK 			(0x11)				/*!< firmware data package*/
#define		FW_RSP_ACK 				(0x12)				/*!< firmware ack*/

/* UART handle and memory for ROM API */
static UART_HANDLE_T *uartHandle;

/* Use a buffer size larger than the expected return value of
   uart_get_mem_size() for the static UART handle type */
static uint32_t uartHandleMEM[0x10];

static volatile bool line_got;

/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/
FW_MESSAGE_T	message;	
/*****************************************************************************
 * Private functions
 ****************************************************************************/
/* Turn on LED to indicate an error */
static void errorUART(void)
{
	Board_LED_Set(0, true);
	while (1) {}
}

/* UART callback */
static void waitCallback(uint32_t err_code, uint32_t n)
{
	line_got = true;
}

/* Send a string on the UART terminated by a NULL character using
   polling mode. */
static void putLineUART(char *send_data, uint32_t size)
{
	UART_PARAM_T param;

	param.buffer = (uint8_t *) send_data;
	param.size = size;

	/* Interrupt mode, do not append CR/LF to sent data */
	param.transfer_mode = TX_MODE_BUF_EMPTY;
	param.driver_mode = DRIVER_MODE_INTERRUPT;

	/* Setup the transmit callback, this will get called when the
	   transfer is complete */
	param.callback_func_pt = (UART_CALLBK_T) waitCallback;

	/* Transmit the data using interrupt mode, the function will
	   return */
	line_got = false;
	if (LPC_UARTD_API->uart_put_line(uartHandle, &param)) {
		errorUART();
	}
}

/* Receive a string on the UART terminated by a LF character using
   polling mode. */
static void getLineUART(char *receive_buffer, uint32_t length)
{
	UART_PARAM_T param;

	param.buffer = (uint8_t *) receive_buffer;
	param.size = length;

	/* Receive data up to the CR/LF character in polling mode. Will
	   truncate at length if too long.	*/
	param.transfer_mode = RX_MODE_BUF_FULL;
	param.driver_mode = DRIVER_MODE_INTERRUPT;

	/* Setup the receive callback, this will get called when the
	   transfer is complete */
	param.callback_func_pt = (UART_CALLBK_T) waitCallback;

	/* Receive the data */
	line_got = false;
	if (LPC_UARTD_API->uart_get_line(uartHandle, &param)) {
		errorUART();
	}
}

/*****************************************************************************
 * Public functions
 ****************************************************************************/
/* UART Pin mux function - note that SystemInit() may already setup your
   pin muxing at system startup */
void Init_UART_PinMux(void)
{
	/* Enable the clock to the Switch Matrix */
	Chip_Clock_EnablePeriphClock(SYSCTL_CLOCK_SWM);

	Chip_Clock_SetUARTClockDiv(1);	/* divided by 1 */

#if (defined(BOARD_NXP_LPCXPRESSO_812) || defined(BOARD_LPC812MAX) || defined(BOARD_NXP_LPCXPRESSO_824))
	/* Connect the U0_TXD_O and U0_RXD_I signals to port pins(P0.4, P0.0) */
	Chip_SWM_DisableFixedPin(SWM_FIXED_ACMP_I1);
	Chip_SWM_MovablePinAssign(SWM_U0_TXD_O, 4);
	Chip_SWM_MovablePinAssign(SWM_U0_RXD_I, 0);
#else
	/* Configure your own UART pin muxing here if needed */
#warning "No UART pin muxing defined"
#endif

	/* Disable the clock to the Switch Matrix to save power */
	Chip_Clock_DisablePeriphClock(SYSCTL_CLOCK_SWM);
}

/* Setup UART handle and parameters */
void setupUART()
{
	uint32_t frg_mult;

	/* 115.2KBPS, 8N1, ASYNC mode, no errors, clock filled in later */
	UART_CONFIG_T cfg = {
		0,				/* U_PCLK frequency in Hz */
		115200,		/* Baud Rate in Hz */
		1,				/* 8N1 */
		0,				/* Asynchronous Mode */
		NO_ERR_EN	/* Enable No Errors */
	};

	/* Perform a sanity check on the storage allocation */
	if (LPC_UARTD_API->uart_get_mem_size() > sizeof(uartHandleMEM)) {
		/* Example only: this should never happen and probably isn't needed for
		   most UART code. */
		errorUART();
	}

	/* Setup the UART handle */
	uartHandle = LPC_UARTD_API->uart_setup((uint32_t) LPC_USART0, (uint8_t *) &uartHandleMEM);
	if (uartHandle == NULL) {
		errorUART();
	}

	/* Need to tell UART ROM API function the current UART peripheral clock
	     speed */
	cfg.sys_clk_in_hz = Chip_Clock_GetSystemClockRate();

	/* Initialize the UART with the configuration parameters */
	frg_mult = LPC_UARTD_API->uart_init(uartHandle, &cfg);
	if (frg_mult) {
		Chip_SYSCTL_SetUSARTFRGDivider(0xFF);	/* value 0xFF should be always used */
		Chip_SYSCTL_SetUSARTFRGMultiplier(frg_mult);
	}
}


/**
 * @brief	Handle UART interrupt
 * @return	Nothing
 */
void UART0_IRQHandler(void)
{
	LPC_UARTD_API->uart_isr(uartHandle);
}

/**
 * @brief	get message via UART
 * @parameter	the current status to message transfer
 */

//uint32_t iap_dbg(void)
//{
//	static uint32_t i=0, size=DATA_SIZE;
//		
//	//got F/W 
//	if((i+DATA_SIZE)>sizeof(LR0)) {
//	 size = sizeof(LR0) - i;
//	}
//	memcpy(message.data, LR0+i, size);
//	i+=DATA_SIZE;
//	
//	if(i>sizeof(LR0)) {
//		return 1;
//	} else {
//		return 0;
//	}
//}

void UART_get_message(uint8_t *ms_status)
{
	uint8_t status;

	status = *ms_status;
	switch (status) {
		case MS_RX_START:
			memset(&message, 0xff, sizeof(message));
			message.ctrl_flag = 0x00;
			getLineUART((char *)&message, sizeof(message));	
			*ms_status = MS_RX_WAIT;				
		break;
		
		case MS_RX_WAIT:
			if(line_got == true) {
				switch (message.type) {
				case FW_UPDATE_REQ:
					*ms_status = MS_FW_REQ_RX_OK;
				break;
				case FW_DATA_PACK:
					*ms_status = MS_FW_DATA_RX_OK;
				break;
				default: 
					*ms_status = MS_RX_START;
				break;
			}
		}
		break;
		
		case MS_FW_REQ_RX_OK:
		case MS_FW_DATA_RX_OK:
			message.type = FW_RSP_ACK;
		  putLineUART((char *)&message, sizeof(message));
			*ms_status = MS_RX_ACK;
		break;
		
		case MS_RX_ACK:
			if(line_got == true) {
				*ms_status = MS_RX_START;
			}
		break;
		
		case MS_RX_ERR:
			*ms_status = MS_RX_START;	
		break;
			
		default:
			
		break;
	}
}
