/*
 * @brief FLASH IAP programming & FLASH signature example using IAP commands
 * to write to FLASH memory and a FLASH signature generator
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2013
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */

#include "board.h"
#include "fw_iap.h"
#include <stdio.h>
#include <string.h>

//#define USER_CODE1
/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/

#define TICKRATE_HZ (2)	/* tick counter per second */
static bool systick_task;

/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/
extern FW_MESSAGE_T	message;
/*****************************************************************************
 * Private functions
 ****************************************************************************/

/* Converts given value @a val into a hexadecimal string
 * and stores the result to @a dest with leading zeros
 * RETURN: Number of hexdigits excluding leading zeros
 */

/*****************************************************************************
 * Public functions
 ****************************************************************************/
void my_memcpy(uint8_t *dst, uint8_t *src, uint32_t size)
{
	uint32_t i;

	i=0;
	while(i<size)
	{
		*(dst++) = *(src++);
		i++;
	}
}
/**
 * @brief	Handle interrupt from SysTick timer
 * @return	Nothing
 */
void SysTick_Handler(void)
{
	systick_task = true;
}

typedef enum {
	IAP_START,
	IAP_ERROR,
	IAP_END_OK,
}IAP_STATUS_T;
static bool start_fw_rx;

extern void execute_new_firmware(void);
/**
 * @brief	Main program body
 * @return	Always returns 0
 */
static ALIGN(64) uint8_t	msg_data[DATA_SIZE];
int main(void)
{
	static uint8_t ms_status;
	uint8_t iap_status, err;
	uint32_t user_addr;
	
	/* Generic Initialization */
	SystemCoreClockUpdate();
	Board_Init();

	Init_UART_PinMux();
	Chip_UART_Init(LPC_USART0);
	Board_LED_Set(0, false);

	/* Allocate UART handle, setup UART parameters, and initialize UART
	   clocking */
	setupUART();
	/* Enable the IRQ for the UART */
	NVIC_EnableIRQ(UART0_IRQn);
	
	/* Enable SysTick Timer */
	SysTick_Config(SystemCoreClock / TICKRATE_HZ);

	ms_status = MS_RX_START;
	iap_status = IAP_START;
	start_fw_rx = false;
	systick_task = false;
	while(1) {
		if(systick_task==true) {
			systick_task = false;
#ifdef USER_CODE1
			Board_LED_Toggle(1);
#else 
			Board_LED_Toggle(2);
#endif
		}
		
		/* F/W IAP */
		UART_get_message(&ms_status);
		
		if(ms_status == MS_FW_REQ_RX_OK) {	
			//start F/W rx
			start_fw_rx = true;
			//setup IAP
			user_addr = setup_IAP();		
			memcpy(message.data, &user_addr, sizeof(uint32_t)); 
		}
				
		if((start_fw_rx == true)&&(ms_status == MS_FW_DATA_RX_OK)) {	//got F/W 
			memcpy(msg_data, message.data, sizeof(message.data));
			err = firmware_IAP((uint32_t *)msg_data, sizeof(msg_data));
			if(err) {
				start_fw_rx = false;
				ms_status = MS_RX_ERR;
				iap_status = IAP_ERROR;
			}
			if(message.ctrl_flag == LAST_DATA_PACK) {
				if(iap_status != IAP_ERROR) {
					iap_status = IAP_END_OK;
				}
			}
		}
		/*execute the updated F/W*/
		if(iap_status == IAP_END_OK) {



			/* generate F/W signature value for verification */
			gen_firmware_signature();
			/* NOTE: must firstly update F/W check information, and then boot information!!! */
			err = update_chk_info();
			if(!err) {
				err = update_boot_info();
			}
			/* execute new firmware*/
			if(!err) {
				execute_new_firmware();
			}
		}
	}
}
